package com.greenpay.datacollector;

import android.content.Context;

import androidx.annotation.Keep;

import com.greenpay.datacollector.utils.KountUtils;
import com.greenpay.datacollector.utils.Logger;
import com.greenpay.datacollector.utils.ResponseCallback;

public class GDataCollector {

    private static final String TAG = GDataCollector.class.getSimpleName();
    private static volatile GDataCollector instance = null;

    /**
     * Initializes GDataCollector
     */
    protected GDataCollector () {
        com.kount.api.DataCollector.getInstance().setDebug(false);
        com.kount.api.DataCollector.getInstance().setMerchantID(BuildConfig.KOUNT_ID);
        com.kount.api.DataCollector.getInstance().setEnvironment(com.kount.api.DataCollector.ENVIRONMENT_PRODUCTION);
        com.kount.api.DataCollector.getInstance().setLocationCollectorConfig(com.kount.api.DataCollector.LocationConfig.COLLECT);
    }

    /**
     * Location Configuration Enum
     */
    @Keep
    public enum LocationConfig {
        COLLECT, SKIP
    }

    /**
     * Gets singleton
     */
    @Keep
    public static GDataCollector getInstance() {
        if  (instance == null) {
            synchronized (GDataCollector.class) {
                if (instance == null) {
                    instance = new GDataCollector();
                }
            }
        }
        return instance;
    }


    /**
     * Sets context
     */
    @Keep
    public void setContext (Context context) {
        com.kount.api.DataCollector.getInstance().setContext(context);
    }

    /**
     Sets location collector configuration permissions.
     */
    @Keep
    public void setLocationConfig(LocationConfig locationConfiguration) {
        switch (locationConfiguration) {
            case COLLECT:
                com.kount.api.DataCollector.getInstance().setLocationCollectorConfig(com.kount.api.DataCollector.LocationConfig.COLLECT);
                Logger.i(TAG, "==== LOCATION CONFIGURATION RESOLVED TO COLLECT");
                break;
            case SKIP:
                com.kount.api.DataCollector.getInstance().setLocationCollectorConfig(com.kount.api.DataCollector.LocationConfig.SKIP);
                Logger.i(TAG, "==== LOCATION CONFIGURATION RESOLVED TO SKIP");
                break;
        }
    }

    /**
     * Enables sandbox mode
     */
    @Keep
    public void enableSandboxMode() {
        com.kount.api.DataCollector.getInstance().setEnvironment(com.kount.api.DataCollector.ENVIRONMENT_TEST);
        Logger.i(TAG, "==== SANDBOX MODE ENABLED");
    }

    /**
     * Enables logs
     */
    @Keep
    public void enableLogs() {
        Logger.enableLogs();
        Logger.i(TAG, "==== LOGS ENABLED");
    }

    /**
     * Collects device and location data
     * @param responseCallback  Callback to call when collection is done
     */
    @Keep
    public void collect(final ResponseCallback<String> responseCallback){
        this.collect(KountUtils.generateSessionID(), responseCallback);
    }

    /**
     * Collects device and location data
     * @param forSession Session Identifier
     * @param responseCallback  Callback to call when collection is done
     */
    @Keep
    public void collect(String forSession, final ResponseCallback<String> responseCallback){
        com.kount.api.DataCollector.getInstance().collectForSession(forSession, new com.kount.api.DataCollector.CompletionHandler() {

            @Override
            public void completed(String sessionId) {
                Logger.i(TAG, "==== DATA COLLECTION COMPLETED");
                responseCallback.onSuccess(sessionId);
            }

            @Override
            public void failed(String sessionId, final com.kount.api.DataCollector.Error error) {
                Logger.e(TAG, "==== DATA COLLECTION FAILED");
                responseCallback.onError(error.getDescription());
            }

        });
    }

}
