/*
 * Decompiled with CFR 0.152.
 */
package com.greatmancode.com.zaxxer.hikari.pool;

import com.greatmancode.com.zaxxer.hikari.HikariConfig;
import com.greatmancode.com.zaxxer.hikari.IConnectionCustomizer;
import com.greatmancode.com.zaxxer.hikari.metrics.IMetricsTracker;
import com.greatmancode.com.zaxxer.hikari.metrics.MetricsFactory;
import com.greatmancode.com.zaxxer.hikari.metrics.MetricsTracker;
import com.greatmancode.com.zaxxer.hikari.pool.HikariMBeanElf;
import com.greatmancode.com.zaxxer.hikari.pool.HikariPoolMBean;
import com.greatmancode.com.zaxxer.hikari.proxy.IHikariConnectionProxy;
import com.greatmancode.com.zaxxer.hikari.proxy.ProxyFactory;
import com.greatmancode.com.zaxxer.hikari.util.ConcurrentBag;
import com.greatmancode.com.zaxxer.hikari.util.DriverDataSource;
import com.greatmancode.com.zaxxer.hikari.util.PoolUtilities;
import com.greatmancode.com.zaxxer.hikari.util.PropertyBeanSetter;
import com.greatmancode.org.slf4j.Logger;
import com.greatmancode.org.slf4j.LoggerFactory;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import javax.sql.DataSource;

public final class HikariPool
implements HikariPoolMBean,
ConcurrentBag.IBagStateListener {
    private static final Logger LOGGER = LoggerFactory.getLogger(HikariPool.class);
    private final DataSource dataSource;
    private final IConnectionCustomizer connectionCustomizer;
    private final HikariConfig configuration;
    private final ConcurrentBag<IHikariConnectionProxy> connectionBag;
    private final ThreadPoolExecutor addConnectionExecutor;
    private final IMetricsTracker metricsTracker;
    private final AtomicReference<Throwable> lastConnectionFailure;
    private final AtomicInteger totalConnections;
    private final Timer houseKeepingTimer;
    private final boolean isAutoCommit;
    private final boolean isIsolateInternalQueries;
    private final boolean isReadOnly;
    private final boolean isRecordMetrics;
    private final boolean isRegisteredMbeans;
    private final boolean isJdbc4ConnectionTest;
    private final long leakDetectionThreshold;
    private final String catalog;
    private final String username;
    private final String password;
    private volatile long connectionTimeout;
    private volatile boolean isShutdown;
    private int transactionIsolation;

    public HikariPool(HikariConfig configuration) {
        this(configuration, configuration.getUsername(), configuration.getPassword());
    }

    public HikariPool(HikariConfig configuration, String username, String password) {
        this.configuration = configuration;
        this.username = username;
        this.password = password;
        this.totalConnections = new AtomicInteger();
        this.connectionBag = new ConcurrentBag();
        this.connectionBag.addBagStateListener(this);
        this.lastConnectionFailure = new AtomicReference();
        this.connectionTimeout = configuration.getConnectionTimeout();
        this.catalog = configuration.getCatalog();
        this.connectionCustomizer = this.initializeCustomizer();
        this.isAutoCommit = configuration.isAutoCommit();
        this.isIsolateInternalQueries = configuration.isIsolateInternalQueries();
        this.isReadOnly = configuration.isReadOnly();
        this.isRegisteredMbeans = configuration.isRegisterMbeans();
        this.isJdbc4ConnectionTest = configuration.isJdbc4ConnectionTest();
        this.leakDetectionThreshold = configuration.getLeakDetectionThreshold();
        this.transactionIsolation = configuration.getTransactionIsolation();
        this.isRecordMetrics = configuration.isRecordMetrics();
        this.metricsTracker = MetricsFactory.createMetricsTracker(this.isRecordMetrics ? configuration.getMetricsTrackerClassName() : "com.greatmancode.com.zaxxer.hikari.metrics.MetricsTracker", configuration.getPoolName());
        this.dataSource = this.initializeDataSource();
        if (this.isRegisteredMbeans) {
            HikariMBeanElf.registerMBeans(configuration, this);
        }
        this.addConnectionExecutor = PoolUtilities.createThreadPoolExecutor(configuration.getMaximumPoolSize(), "HikariCP connection filler");
        this.fillPool();
        long delayPeriod = Long.getLong("com.greatmancode.com.zaxxer.hikari.housekeeping.periodMs", TimeUnit.SECONDS.toMillis(30L));
        this.houseKeepingTimer = new Timer("Hikari Housekeeping Timer (pool " + configuration.getPoolName() + ")", true);
        this.houseKeepingTimer.scheduleAtFixedRate((TimerTask)new HouseKeeper(), delayPeriod, delayPeriod);
    }

    public Connection getConnection() throws SQLException {
        long start = System.currentTimeMillis();
        IMetricsTracker.MetricsContext context = this.isRecordMetrics ? this.metricsTracker.recordConnectionRequest(start) : MetricsTracker.NO_CONTEXT;
        long timeout = this.connectionTimeout;
        try {
            do {
                IHikariConnectionProxy connection;
                if ((connection = this.connectionBag.borrow(timeout, TimeUnit.MILLISECONDS)) == null) {
                    break;
                }
                long now = System.currentTimeMillis();
                connection.unclose(now);
                if (now <= connection.getExpirationTime() && (now - connection.getLastAccess() <= 1000L || this.isConnectionAlive(connection, timeout))) {
                    if (this.leakDetectionThreshold != 0L) {
                        connection.captureStack(this.leakDetectionThreshold, this.houseKeepingTimer);
                    }
                    IHikariConnectionProxy iHikariConnectionProxy = connection;
                    return iHikariConnectionProxy;
                }
                this.closeConnection(connection);
            } while ((timeout -= PoolUtilities.elapsedTimeMs(start)) > 0L);
        }
        catch (InterruptedException e) {
            throw new SQLException("Interrupted during connection acquisition", e);
        }
        finally {
            context.stop();
        }
        this.logPoolState("Timeout failure ");
        throw new SQLException(String.format("Timeout of %dms encountered waiting for connection.", this.configuration.getConnectionTimeout()), this.lastConnectionFailure.getAndSet(null));
    }

    public void releaseConnection(IHikariConnectionProxy connectionProxy, boolean isBroken) {
        if (this.isRecordMetrics) {
            this.metricsTracker.recordConnectionUsage(PoolUtilities.elapsedTimeMs(connectionProxy.getLastOpenTime()));
        }
        if (isBroken || this.isShutdown) {
            LOGGER.debug("Connection returned to pool {} is broken, or the pool is shutting down.  Closing connection.", (Object)this.configuration.getPoolName());
            this.closeConnection(connectionProxy);
            return;
        }
        this.connectionBag.requite(connectionProxy);
    }

    public void shutdown() throws InterruptedException {
        if (!this.isShutdown) {
            this.isShutdown = true;
            LOGGER.info("HikariCP pool {} is shutting down.", (Object)this.configuration.getPoolName());
            this.logPoolState("Before shutdown ");
            this.houseKeepingTimer.cancel();
            this.addConnectionExecutor.shutdownNow();
            long start = System.currentTimeMillis();
            do {
                this.closeIdleConnections();
                this.abortActiveConnections();
            } while ((this.getIdleConnections() > 0 || this.getActiveConnections() > 0) && PoolUtilities.elapsedTimeMs(start) < TimeUnit.SECONDS.toMillis(5L));
            this.logPoolState("After shutdown ");
            if (this.isRegisteredMbeans) {
                HikariMBeanElf.unregisterMBeans(this.configuration, this);
            }
        }
    }

    public DataSource getDataSource() {
        return this.dataSource;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void closeConnection(IHikariConnectionProxy connectionProxy) {
        try {
            int tc = this.totalConnections.decrementAndGet();
            if (tc < 0) {
                LOGGER.warn("Internal accounting inconsistency, totalConnections={}", (Object)tc, (Object)new Exception());
            }
            connectionProxy.realClose();
        }
        catch (SQLException e) {
            return;
        }
        finally {
            this.connectionBag.remove(connectionProxy);
        }
    }

    public String toString() {
        return this.configuration.getPoolName();
    }

    @Override
    public void addBagItem() {
        class AddConnection
        implements Runnable {
            AddConnection() {
            }

            @Override
            public void run() {
                long sleepBackoff = 200L;
                int maxPoolSize = HikariPool.this.configuration.getMaximumPoolSize();
                int minIdle = HikariPool.this.configuration.getMinimumIdle();
                while (!(HikariPool.this.isShutdown || HikariPool.this.totalConnections.get() >= maxPoolSize || minIdle != 0 && HikariPool.this.getIdleConnections() >= minIdle)) {
                    if (!HikariPool.this.addConnection()) {
                        PoolUtilities.quietlySleep(sleepBackoff);
                        sleepBackoff = Math.min(1000L, (long)((double)sleepBackoff * 1.5));
                        continue;
                    }
                    if (minIdle != 0) continue;
                    break;
                }
            }
        }
        this.addConnectionExecutor.submit(new AddConnection());
    }

    @Override
    public int getActiveConnections() {
        return Math.min(this.configuration.getMaximumPoolSize(), this.totalConnections.get() - this.getIdleConnections());
    }

    @Override
    public int getIdleConnections() {
        return this.connectionBag.getCount(0);
    }

    @Override
    public int getTotalConnections() {
        return this.totalConnections.get();
    }

    @Override
    public int getThreadsAwaitingConnection() {
        return this.connectionBag.getPendingQueue();
    }

    @Override
    public void closeIdleConnections() {
        for (IHikariConnectionProxy connectionProxy : this.connectionBag.values(0)) {
            if (!this.connectionBag.reserve(connectionProxy)) continue;
            this.closeConnection(connectionProxy);
        }
    }

    private boolean addConnection() {
        Connection connection = null;
        try {
            if (this.totalConnections.incrementAndGet() > this.configuration.getMaximumPoolSize() || this.isShutdown) {
                this.totalConnections.decrementAndGet();
                return true;
            }
            connection = this.username == null && this.password == null ? this.dataSource.getConnection() : this.dataSource.getConnection(this.username, this.password);
            this.transactionIsolation = this.transactionIsolation < 0 ? connection.getTransactionIsolation() : this.transactionIsolation;
            this.connectionCustomizer.customize(connection);
            PoolUtilities.executeSqlAutoCommit(connection, this.configuration.getConnectionInitSql());
            IHikariConnectionProxy proxyConnection = ProxyFactory.getProxyConnection(this, connection, this.configuration.getMaxLifetime(), this.transactionIsolation, this.isAutoCommit, this.isReadOnly, this.catalog);
            proxyConnection.resetConnectionState();
            this.connectionBag.add(proxyConnection);
            this.lastConnectionFailure.set(null);
            return true;
        }
        catch (Exception e) {
            this.totalConnections.decrementAndGet();
            this.lastConnectionFailure.set(e);
            PoolUtilities.quietlyCloseConnection(connection);
            LOGGER.debug("Connection attempt to database {} failed: {}", this.configuration.getPoolName(), e.getMessage(), e);
            return false;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isConnectionAlive(IHikariConnectionProxy connection, long timeoutMs) {
        boolean bl;
        boolean timeoutEnabled = this.configuration.getConnectionTimeout() != Integer.MAX_VALUE;
        long l = timeoutMs = timeoutEnabled ? Math.max(1000L, timeoutMs) : 0L;
        if (this.isJdbc4ConnectionTest) {
            return connection.isValid((int)TimeUnit.MILLISECONDS.toSeconds(timeoutMs));
        }
        Statement statement = connection.createStatement();
        try {
            statement.setQueryTimeout((int)TimeUnit.MILLISECONDS.toSeconds(timeoutMs));
            statement.executeQuery(this.configuration.getConnectionTestQuery());
            bl = true;
        }
        catch (Throwable throwable) {
            try {
                statement.close();
                if (this.isIsolateInternalQueries && !this.isAutoCommit) {
                    connection.rollback();
                }
                throw throwable;
            }
            catch (SQLException e) {
                LOGGER.warn("Exception during keep alive check, that means the connection must be dead.", e);
                return false;
            }
        }
        statement.close();
        if (this.isIsolateInternalQueries && !this.isAutoCommit) {
            connection.rollback();
        }
        return bl;
    }

    private void fillPool() {
        if (this.configuration.isInitializationFailFast()) {
            for (int maxIters = this.configuration.getMinimumIdle(); maxIters > 0; --maxIters) {
                if (this.addConnection()) continue;
                throw new RuntimeException("Fail-fast during pool initialization", this.lastConnectionFailure.getAndSet(null));
            }
        } else if (this.configuration.getMinimumIdle() > 0) {
            this.addBagItem();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void abortActiveConnections() throws InterruptedException {
        ThreadPoolExecutor assassinExecutor = PoolUtilities.createThreadPoolExecutor(this.configuration.getMaximumPoolSize(), "HikariCP connection assassin");
        for (IHikariConnectionProxy connectionProxy : this.connectionBag.values(1)) {
            try {
                if (PoolUtilities.IS_JAVA7) {
                    connectionProxy.abort(assassinExecutor);
                    continue;
                }
                connectionProxy.close();
            }
            catch (SQLException e) {
                this.totalConnections.decrementAndGet();
                try {
                    this.connectionBag.remove(connectionProxy);
                }
                catch (IllegalStateException ise) {}
            }
            finally {
                this.totalConnections.decrementAndGet();
                try {
                    this.connectionBag.remove(connectionProxy);
                }
                catch (IllegalStateException ise) {}
            }
        }
        assassinExecutor.shutdown();
        assassinExecutor.awaitTermination(5L, TimeUnit.SECONDS);
    }

    private DataSource initializeDataSource() {
        String dsClassName = this.configuration.getDataSourceClassName();
        if (this.configuration.getDataSource() == null && dsClassName != null) {
            DataSource dataSource = PoolUtilities.createInstance(dsClassName, DataSource.class, new Object[0]);
            PropertyBeanSetter.setTargetFromProperties(dataSource, this.configuration.getDataSourceProperties());
            return dataSource;
        }
        if (this.configuration.getJdbcUrl() != null) {
            return new DriverDataSource(this.configuration.getJdbcUrl(), this.configuration.getDataSourceProperties(), this.username, this.password);
        }
        return this.configuration.getDataSource();
    }

    private IConnectionCustomizer initializeCustomizer() {
        if (this.configuration.getConnectionCustomizerClassName() != null) {
            return PoolUtilities.createInstance(this.configuration.getConnectionCustomizerClassName(), IConnectionCustomizer.class, new Object[0]);
        }
        return this.configuration.getConnectionCustomizer();
    }

    private void logPoolState(String ... prefix) {
        int total = this.totalConnections.get();
        int idle = this.getIdleConnections();
        LOGGER.debug("{}pool stats {} (total={}, inUse={}, avail={}, waiting={})", prefix.length > 0 ? prefix[0] : "", this.configuration.getPoolName(), total, total - idle, idle, this.getThreadsAwaitingConnection());
    }

    private class HouseKeeper
    extends TimerTask {
        private HouseKeeper() {
        }

        @Override
        public void run() {
            HikariPool.this.logPoolState(new String[]{"Before cleanup "});
            HikariPool.this.connectionTimeout = HikariPool.this.configuration.getConnectionTimeout();
            HikariPool.this.houseKeepingTimer.purge();
            long now = System.currentTimeMillis();
            long idleTimeout = HikariPool.this.configuration.getIdleTimeout();
            for (IHikariConnectionProxy connectionProxy : HikariPool.this.connectionBag.values(0)) {
                if (!HikariPool.this.connectionBag.reserve(connectionProxy)) continue;
                if (idleTimeout > 0L && now > connectionProxy.getLastAccess() + idleTimeout || now > connectionProxy.getExpirationTime()) {
                    HikariPool.this.closeConnection(connectionProxy);
                    continue;
                }
                HikariPool.this.connectionBag.unreserve(connectionProxy);
            }
            HikariPool.this.logPoolState(new String[]{"After cleanup "});
            HikariPool.this.addBagItem();
        }
    }
}

