package com.gradle.publish;

import org.gradle.api.Action;
import org.gradle.api.Project;
import org.gradle.api.artifacts.Configuration;
import org.gradle.api.artifacts.ConfigurationContainer;
import org.gradle.api.artifacts.Dependency;
import org.gradle.api.component.AdhocComponentWithVariants;
import org.gradle.api.component.ConfigurationVariantDetails;
import org.gradle.api.component.SoftwareComponent;
import org.gradle.api.logging.Logger;
import org.gradle.api.plugins.AppliedPlugin;
import org.gradle.jvm.tasks.Jar;

/* This code needs to be in a separate class, so loading it won't be attempted when using old Gradle versions. */
public class PublishTaskShadowAction implements Action<AppliedPlugin> {

    private final Project project;
    private final Logger logger;

    public PublishTaskShadowAction(Project project, Logger logger) {
        this.project = project;
        this.logger = logger;
    }

    @Override
    public void execute(AppliedPlugin appliedPlugin) {
        logger.lifecycle("Shadow plugin detected. Will automatically select a fat jar as the main plugin artifact.");
        ConfigurationContainer configurations = project.getConfigurations();
        manipulateVariants(configurations);
        manipulateDependencies(configurations);
    }

    private void manipulateVariants(ConfigurationContainer configurations) {
        if (configurations.getByName("shadowRuntimeElements").getAllArtifacts().stream()
                .anyMatch(a -> a.getClassifier() != null && !a.getClassifier().isEmpty())) {
            throw new RuntimeException("Please configure the `shadowJar` task to not add a classifier to the jar it produces");
        }

        // add a classifier to the regular jar, because otherwise it will write to the same file as the shadowJar task
        project.getTasks().named("jar", Jar.class, jar -> {
            jar.getArchiveClassifier().set("main");
        });

        SoftwareComponent javaComponent = project.getComponents().getByName("java");
        AdhocComponentWithVariants adhocComponent = (AdhocComponentWithVariants) javaComponent;
        adhocComponent.addVariantsFromConfiguration(configurations.getByName("apiElements"), ConfigurationVariantDetails::skip);
        adhocComponent.addVariantsFromConfiguration(configurations.getByName("runtimeElements"), ConfigurationVariantDetails::skip);
        adhocComponent.addVariantsFromConfiguration(configurations.getByName("shadowRuntimeElements"), configVariantDetails -> configVariantDetails.mapToMavenScope("runtime"));
    }

    private void manipulateDependencies(ConfigurationContainer configurations) {
        Dependency gradleApi = project.getDependencies().gradleApi();

        Configuration api = configurations.findByName("api");
        if (api != null) {
            api.getDependencies().remove(gradleApi);
        }

        Configuration compileOnlyApi = configurations.findByName("compileOnlyApi");
        if (compileOnlyApi != null) {
            compileOnlyApi.getDependencies().remove(gradleApi);
        }

        configurations.getByName("shadow").getDependencies().add(gradleApi);
    }
}
