package com.gradle.publish;

import com.gradle.publish.protocols.v1.models.publish.ArtifactTypeCodec;
import com.gradle.publish.protocols.v1.models.publish.BuildMetadata;
import com.gradle.publish.protocols.v1.models.publish.PublishArtifact;
import com.gradle.publish.protocols.v1.models.publish.PublishMavenCoordinates;
import com.gradle.publish.protocols.v1.models.publish.PublishNewVersionRequest;
import org.gradle.api.*;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.api.publish.Publication;
import org.gradle.api.publish.PublishingExtension;
import org.gradle.api.publish.internal.PublicationInternal;
import org.gradle.api.publish.maven.MavenArtifact;
import org.gradle.api.publish.maven.MavenPublication;
import org.gradle.api.publish.maven.internal.publication.DefaultMavenPublication;
import org.gradle.api.publish.maven.internal.publisher.MavenNormalizedPublication;
import org.gradle.api.publish.maven.tasks.GenerateMavenPom;
import org.gradle.api.publish.tasks.GenerateModuleMetadata;
import org.gradle.api.resources.MissingResourceException;
import org.gradle.api.tasks.TaskAction;
import org.gradle.internal.deprecation.DeprecationLogger;
import org.gradle.plugin.devel.GradlePluginDevelopmentExtension;
import org.gradle.plugin.devel.PluginDeclaration;
import org.gradle.plugins.signing.SigningExtension;
import org.gradle.util.GradleVersion;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.gradle.publish.Util.isBlank;
import static java.lang.String.format;
import static org.codehaus.groovy.runtime.StringGroovyMethods.capitalize;

public class PublishTask extends DefaultTask {
    public static final String GRADLE_PUBLISH_SECRET = "gradle.publish.secret";
    public static final String GRADLE_PUBLISH_KEY = "gradle.publish.key";
    public static final String GRADLE_PUBLISH_SECRET_ENV = "GRADLE_PUBLISH_SECRET";
    public static final String GRADLE_PUBLISH_KEY_ENV = "GRADLE_PUBLISH_KEY";

    private static final String SKIP_NAMESPACE_CHECK_PROPERTY = "gradle.publish.skip.namespace.check";
    private static final String MAVEN_PUBLISH_POM_TASK_NAME = "generatePomFileForPluginMavenPublication";
    private static final String MAVEN_PUBLISH_GMM_TASK_NAME = "generateMetadataFileForPluginMavenPublication";
    static final String MAVEN_PUBLICATION_NAME = "pluginMaven";
    static final String MARKER_PUBLICATION_SUFFIX = "PluginMarkerMaven";

    private static final Logger LOGGER = Logging.getLogger(PublishTask.class);
    private final PortalPublisher portalPublisher = new PortalPublisher(getProject());

    private PluginBundleExtension bundleConfig;

    private boolean useLegacyConfig;
    private GradlePluginDevelopmentExtension pluginConfig;
    private AbstractConfigValidator validator;

    @TaskAction
    void publish() throws Exception {
        useLegacyConfig = useLegacyConfig();

        boolean skipNamespaceCheck = System.getProperty(SKIP_NAMESPACE_CHECK_PROPERTY, "false").equals("true");
        validator = useLegacyConfig ? new LegacyConfigValidator(bundleConfig, skipNamespaceCheck) : new ConfigValidator(skipNamespaceCheck);
        pluginConfig = getProject().getExtensions().getByType(GradlePluginDevelopmentExtension.class); //TODO: reduce project access during task execution (ie. do more in the constructor)
        validator.validateConfig(pluginConfig);

        PublishMavenCoordinates mavenCoords = ensurePomIsAvailable();
        validateVersion(mavenCoords.getVersion());

        List<PublishNewVersionRequest> requests = buildPublishRequests(mavenCoords);
        validatePluginDescriptors(requests);

        Map<PublishArtifact, File> artifacts = collectArtifacts();
        portalPublisher.publishToPortal(requests, mavenCoords, artifacts);
    }

    private boolean useLegacyConfig() {
        if (runningOnPreGradle("7.6-dev-1")) {
            return true;
        }

        if (!runningOnPreGradle("8.0-dev-1")) {
            return false;
        }

        if (!bundleConfig.isEmpty()) {
            DeprecationLogger.deprecate("Using the `pluginBundle` block to configure plugin publication")
                    .withAdvice("Use new properties of `gradlePlugin` instead.")
                    .willBeRemovedInGradle8()
                    .withUserManual("publishing_gradle_plugins", "sec:configuring-publish-plugin")
                    .nagUser();
            return true;
        }

        return false;
    }

    private void validatePluginDescriptors(List<PublishNewVersionRequest> requests) {
        File artifactFile = findMainArtifact();
        try (ZipFile zip = new ZipFile(artifactFile)) {
            for (PublishNewVersionRequest request : requests) {
                //TODO: we need to check artifacts of other variants too
                // but have the ability to specify that some artifact doesn't need it
                validatePluginDescriptor(zip, request.getPluginId());
            }
        } catch (IOException e) {
            throw new RuntimeException("Unable to validate plugin jar " + artifactFile.getPath(), e);
        }
    }

    private void validateVersion(String version) {
        String regExp = "[a-zA-Z0-9\\-\\.\\[\\]\\:\\+]*";
        String numberRegExp = ".*?[0-9].*?";
        if (isBlank(version) || version.equals("unspecified")) {
            throw new IllegalArgumentException("Please set the project version");
        }
        if (!version.matches(numberRegExp) || !version.matches(regExp) || version.length () > 50) {
            throw new RuntimeException("Invalid version '"+version+"'. Project version string must: " +
                    "1) not be empty and less than 50 characters long; " +
                    "2) include a number; " +
                    "3) match the regular expression: " + regExp);
        }
        if (version.trim().endsWith("-SNAPSHOT")) {
            throw new IllegalArgumentException("-SNAPSHOT plugin versions not supported, please use a fixed version instead.");
        }
    }

    private void validatePluginDescriptor(ZipFile zip, String pluginId) throws IOException {
        String resPath = format("META-INF/gradle-plugins/%s.properties", pluginId);
        ZipEntry descriptorEntry = zip.getEntry(resPath);
        if (descriptorEntry == null) {
            throw new IllegalArgumentException(
                    format("No plugin descriptor for plugin ID '%s'.\nCreate a " +
                                    "'META-INF/gradle-plugins/%s.properties' file with a " +
                                    "'implementation-class' property pointing to the plugin class " +
                                    "implementation.",
                            pluginId,
                            pluginId));
        }
        Properties descriptor = new Properties();
        descriptor.load(zip.getInputStream(descriptorEntry));
        String pluginClassName = descriptor.getProperty("implementation-class");
        if (isBlank(pluginClassName)) {
            throw new IllegalArgumentException(
                    format("Plugin descriptor for plugin ID '%s' does not specify a plugin\n"
                            + "class with the implementation-class property", pluginId));
        }

        String pluginClassResourcePath = pluginClassName.replace('.', '/').concat(".class");
        if (zip.getEntry(pluginClassResourcePath) == null) {
            throw new IllegalArgumentException(
                    format("Plugin descriptor for plugin ID '%s' specifies a plugin\n"
                            + "class '%s' that is not present in the jar file", pluginId, pluginClassName));
        }
    }

    private PublishMavenCoordinates ensurePomIsAvailable() {
        GenerateMavenPom pomTask = (GenerateMavenPom) getProject().getTasks().getByName(MAVEN_PUBLISH_POM_TASK_NAME); //TODO: reduce project access during task execution (ie. do more in the constructor)
        File pomFile = pomTask.getDestination();
        if (!pomFile.exists()) {
            throw new MissingResourceException(pomFile.toURI(), "Could not use POM from " + MAVEN_PUBLISH_POM_TASK_NAME + " task because it does not exist");
        }
        PublishingExtension publishing = getProject().getExtensions().getByType(PublishingExtension.class);
        MavenPublication pluginPublication = (MavenPublication) publishing.getPublications().getByName(PublishTask.MAVEN_PUBLICATION_NAME);
        return validateMavenCoordinates(pluginPublication.getGroupId(), pluginPublication.getArtifactId(), pluginPublication.getVersion());
    }

    void addAndHashArtifact(Map<PublishArtifact, File> artifacts, File file, String type, String classifier) throws IOException {
        if (file != null && file.exists()) {
            try (FileInputStream fis = new FileInputStream(file)) {
                String hash = Hasher.hash(fis);
                try {
                    String artifactType = ArtifactTypeCodec.encode(type, classifier);
                    artifacts.put(new PublishArtifact(artifactType, hash), file);
                } catch (IllegalArgumentException e) {
                    LOGGER.warn("Ignoring unknown artifact with type \"{}\" and " +
                                    "classifier \"{}\".\nYou can only upload normal jars, " +
                                    "sources jars, javadoc jars and groovydoc jars\n" +
                                    "with or without signatures to the Plugin Portal at this time.",
                            type, classifier);
                }
            }
        }
    }

    private Map<PublishArtifact, File> collectArtifacts() throws IOException {
        Map<PublishArtifact, File> artifacts = new LinkedHashMap<>();
        for (MavenArtifact mavenArtifact : getNormalizedMavenPublication().getAllArtifacts()) {
            addAndHashArtifact(artifacts, mavenArtifact.getFile(), mavenArtifact.getExtension(), mavenArtifact.getClassifier());
        }
        return artifacts;
    }

    private File findMainArtifact() {
        MavenNormalizedPublication publication = getNormalizedMavenPublication();

        if (isMainArtifact(publication.getMainArtifact())) {
            return publication.getMainArtifact().getFile();
        }

        Set<MavenArtifact> artifacts = publication.getAllArtifacts();
        Optional<MavenArtifact> artifact = artifacts.stream().filter(this::isMainArtifact).findFirst();
        if (artifact.isPresent()) {
            return artifact.get().getFile();
        }

        throw new IllegalArgumentException("Cannot determine main artifact to upload - could not find jar artifact with empty classifier");
    }

    private boolean isMainArtifact(MavenArtifact artifact) {
        if (artifact == null) {
            return false;
        }

        if (!"jar".equals(artifact.getExtension())) {
            return false;
        }

        return isBlank(artifact.getClassifier());
    }

    private MavenNormalizedPublication getNormalizedMavenPublication() {
        PublishingExtension publishingExtension = getProject().getExtensions().getByType(PublishingExtension.class);
        PublicationInternal<MavenArtifact> publication =
                (PublicationInternal<MavenArtifact>) publishingExtension.getPublications().getByName(PublishTask.MAVEN_PUBLICATION_NAME);
        MavenNormalizedPublication normalizedPublication = ((DefaultMavenPublication) publication).asNormalisedPublication();
        return normalizedPublication;
    }

    private PublishMavenCoordinates validateMavenCoordinates(String groupId, String artifactId, String version) {
        validator.validateMavenCoordinates(groupId, artifactId, version);
        return new PublishMavenCoordinates(groupId, artifactId, version);
    }

    private List<PublishNewVersionRequest> buildPublishRequests(PublishMavenCoordinates mavenCoords) {
        String website = useLegacyConfig ? bundleConfig.getWebsite() : Util.getWebsite(pluginConfig);
        String vcsUrl = useLegacyConfig ? bundleConfig.getVcsUrl() : Util.getVcsUrl(pluginConfig);

        List<PublishNewVersionRequest> reqs = new ArrayList<>();
        for (PluginDeclaration plugin : pluginConfig.getPlugins()) {
            reqs.add(buildPublishRequest(mavenCoords, website, vcsUrl, plugin));
        }
        return reqs;
    }

    private List<String> getTags(PluginDeclaration pluginDeclaration) {
        Collection<String> tags = useLegacyConfig ? Util.getTags(bundleConfig, pluginDeclaration.getName()) : Util.getTags(pluginDeclaration);
        return tags.stream()
                .map(String::toLowerCase)
                .distinct()
                .collect(Collectors.toList());
    }

    private PublishNewVersionRequest buildPublishRequest(PublishMavenCoordinates mavenCoords, String website, String vcsUrl, PluginDeclaration plugin) {
        PublishNewVersionRequest request = new PublishNewVersionRequest();
        BuildMetadata buildMetadata = new BuildMetadata(getProject().getGradle().getGradleVersion());
        request.setBuildMetadata(buildMetadata);

        request.setPluginId(plugin.getId());

        request.setPluginVersion(mavenCoords.getVersion());
        request.setDisplayName(plugin.getDisplayName());

        String desc = plugin.getDescription();
        desc = desc == null && useLegacyConfig ? bundleConfig.getDescription() : desc;
        request.setDescription(desc);
        request.setTags(getTags(plugin));
        request.setWebSite(website);
        request.setVcsUrl(vcsUrl);

        return request;
    }

    public void setBundleConfig(PluginBundleExtension bundleConfig) {
        this.bundleConfig = bundleConfig;
    }

    public void afterProjectEvaluate() {
        Project project = getProject();

        dependsOn(project.getTasks().matching(task -> MAVEN_PUBLISH_POM_TASK_NAME.equals(task.getName())));

        GenerateModuleMetadata gmmTask = (GenerateModuleMetadata) project.getTasks().findByName(MAVEN_PUBLISH_GMM_TASK_NAME);
        if (gmmTask != null && gmmTask.getEnabled()) {
            dependsOn(gmmTask.getOutputFile());
        }

        project.getPluginManager().withPlugin(PublishPlugin.SIGNING_PLUGIN_ID, signingPlugin -> {
            LOGGER.lifecycle("Signing plugin detected. Will automatically sign the published artifacts.");

            SigningExtension signing = project.getExtensions().getByType(SigningExtension.class);
            PublishingExtension publishing = getProject().getExtensions().getByType(PublishingExtension.class);
            wireInSigningTask(project, signing, publishing.getPublications().getByName(MAVEN_PUBLICATION_NAME));

            NamedDomainObjectContainer<PluginDeclaration> plugins = getProject().getExtensions().getByType(GradlePluginDevelopmentExtension.class).getPlugins();
            for(PluginDeclaration plugin : plugins) {
                String markerPublicationName = plugin.getName() + MARKER_PUBLICATION_SUFFIX;
                Publication publication = publishing.getPublications().getByName(markerPublicationName);
                wireInSigningTask(project, signing, publication);
            }
        });
    }

    private void wireInSigningTask(Project project, SigningExtension signing, Publication publication) {
        String signTaskName = determineSignTaskNameForPublication(publication);
        Task signTask = project.getTasks().findByName(signTaskName);
        if (signTask == null) {
            signTask = signing.sign(publication).get(0);
        }
        dependsOn(signTask);
    }

    private static boolean runningOnPreGradle(String version) {
        return GradleVersion.current().compareTo(GradleVersion.version(version)) < 0;
    }

    private static String determineSignTaskNameForPublication(Publication publication) {
        return "sign" + capitalize((CharSequence) publication.getName()) + "Publication";
    }

}
