package com.gradle.enterprise.gradleplugin.testdistribution;

import org.gradle.api.provider.Property;
import org.gradle.api.provider.SetProperty;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Internal;

import java.time.Duration;

/**
 * Task extension for configuring test distribution.
 * <p>
 * This extension is added with name {@value #NAME} to all {@link org.gradle.api.tasks.testing.Test Test} tasks.
 * <p>
 * Please refer to the
 * <a href="https://docs.gradle.com/enterprise/test-distribution-gradle-plugin/">plugin documentation</a> for details.
 */
public interface TestDistributionExtension {

    /**
     * The name of the extension added to each test task.
     */
    String NAME = "distribution";

    /**
     * Whether test distribution is enabled.
     * <p>
     * This setting defaults to {@code false}.
     *
     * @return whether test distribution is enabled
     */
    @Internal
    Property<Boolean> getEnabled();

    /**
     * The number of executors to use for running tests on this machine.
     * <p>
     * This setting defaults to {@link org.gradle.api.tasks.testing.Test#getMaxParallelForks() maxParallelForks}.
     * The number of local executors that can be used is limited by
     * {@link org.gradle.StartParameter#getMaxWorkerCount() maxWorkerCount}.
     * Setting this to 0 causes tests to be only executed on the remote test execution infrastructure.
     *
     * @return the number of local executors to use.
     */
    @Internal
    Property<Integer> getMaxLocalExecutors();

    /**
     * The number of remote executors that should be requested from the remote test execution infrastructure in order
     * run the tests of this test task.
     * <p>
     * This setting has no default value meaning that as much remote executors as are possible should be requested.
     * Setting this to zero causes tests to be only executed using local executors.
     * <p>
     * Note that setting this to a fixed number doesn't necessarily mean that this test task will get that many remote
     * executors. It's just an indicator for the remote test execution infrastructure for how many executors it should
     * try to make available to this test task. It may well be that this test task does not get any remote executors at
     * all e.g. if no executors matching all {@link #getRequirements() requirements} become available.
     *
     * @return the number of remote executors to request.
     */
    @Internal
    Property<Integer> getMaxRemoteExecutors();

    /**
     * If enabled, remote executors are tried first, falling back to local ones after {@link #getWaitTimeout() waitTimeout} expires.
     * <p>
     * This setting defaults to {@code false}.
     *
     * @return whether remote executors should be preferred over local executors
     */
    @Internal
    Property<Boolean> getRemoteExecutionPreferred();

    /**
     * The duration the build waits for the remote test execution infrastructure to respond.
     * <p>
     * This setting defaults to 30 seconds and must be set to a positive value.
     *
     * @return the wait timeout duration.
     * @since 1.2
     */
    @Internal
    Property<Duration> getWaitTimeout();

    /**
     * If enabled, the tests will be retried in the same JVM.
     * <p>
     * This setting defaults to {@code true}.
     *
     * @return whether to retry tests in the same JVM
     * @since 2.2.1
     */
    @Internal
    Property<Boolean> getRetryInSameJvm();

    /**
     * The requirements that need to be fulfilled by executors in order to execute the tests of this test task.
     * <p>
     * Requirements are matched against the capabilities of the connected remote executors. Only those executors
     * having all required capabilities will be assigned for running the tests of this task. Local executors are
     * assumed to provide any capabilities. This means that no matter the requirements, local executors will always
     * be used as long as {@link #getMaxLocalExecutors() maxLocalExecutors} is greater than 0.
     * <p>
     * A requirement String may only contain alphanumeric characters, dashes, underscores, periods, and a single equals
     * sign. Apart from that, users are free in how they want to structure requirements. The recommendation is to use
     * key value pairs or tag-like requirements:
     * <pre>
     * test {
     *     distribution {
     *         requirements = ['os=linux', 'db=mysql', 'EXECUTOR_POOL_1']
     *     }
     * }
     * </pre>
     * <p>
     * The requirements will always have the value {@code jdk=<version>} added with {@code <version>} being derived from
     * the executable set on this test task via {@link org.gradle.api.tasks.testing.Test#getExecutable()}.
     *
     * @return the requirements remote executors need to fulfill for running tests of this test task.
     */
    @Input
    SetProperty<String> getRequirements();

}
