package com.gradle.enterprise.gradleplugin;

import com.gradle.obfuscation.Keep;
import com.gradle.scan.agent.client.http.DevelocityServerAddress;
import org.gradle.api.Action;
import org.gradle.caching.configuration.AbstractBuildCache;
import org.gradle.caching.configuration.BuildCacheConfiguration;

import javax.annotation.Nullable;
import java.util.Objects;

import static com.gradle.enterprise.java.Checks.checkNotNull;

/**
 * Allows configuring the use of a Gradle Enterprise build cache.
 * <p>
 * This can be used with {@link BuildCacheConfiguration#remote(Class, Action)} to enable
 * use of the Gradle Enterprise remote build caching.
 * The method {@link GradleEnterpriseExtension#getBuildCache()} must be used to obtain the type object to use.
 * This type cannot be used directly.
 * <p>
 * The following example demonstrates usage in a build's settings script:
 * <p>
 * <pre>{@code
 * buildCache {
 *   remote(gradleEnterprise.buildCache) {
 *     // configure
 *   }
 * }
 * }</pre>
 * <p>
 * Without any explicit configuration, the default build cache provided by the installation specified by
 * {@link GradleEnterpriseExtension#getServer()} will be used.
 * <p>
 * If an access key is available for the {@link GradleEnterpriseExtension#getServer()} value it will be used to authenticate with the build cache.
 *
 * @since 3.11
 * @deprecated since 3.17, replaced by {@link com.gradle.develocity.agent.gradle.buildcache.DevelocityBuildCache}
 */
@Keep
@Deprecated
public class GradleEnterpriseBuildCache extends AbstractBuildCache {

    @Nullable
    private DevelocityServerAddress server;

    @Nullable
    private String path;

    @Nullable
    private Boolean allowUntrustedServer;

    private boolean allowInsecureProtocol;
    private boolean useExpectContinue;

    @Nullable
    private UsernameAndPassword usernameAndPassword;

    /**
     * The address of the build cache server to use, if not using the built-in build cache of the server specified by {@link GradleEnterpriseExtension#getServer()}.
     *
     * @return {@code null} if the built-in build cache of the Gradle Enterprise server should be used, or the address of the build cache server that should be used
     * @see #setServer(String)
     */
    @Nullable
    public String getServer() {
        return server == null ? null : server.asString();
    }

    /**
     * Sets the build cache server address to use, if not using the default build cache of the server specified by {@link GradleEnterpriseExtension#getServer()}.
     * <p>
     * Setting this option is necessary when using a dedicated build cache node instead of the Gradle Enterprise built-in build cache.
     * <p>
     * The value set must not include any request path or query string.
     * It must only specify the protocol, hostname and port.
     * <p>
     * If an access key is available for the {@link GradleEnterpriseExtension#getServer()} value it will be used to authenticate with the build cache,
     * regardless of the build cache server address set with this method.
     *
     * @param server the server address
     */
    public void setServer(@Nullable String server) {
        this.server = server == null ? null : DevelocityServerAddress.parse(server);
    }

    /**
     * The custom request path for the build cache server.
     * <p>
     * It is generally not necessary to set this value, as the default value used is suitable for the default
     * configuration of Gradle Enterprise build cache nodes (including the built-in build cache).
     *
     * @return the custom request path for the build cache server
     * @see #setPath(String)
     */
    @Nullable
    public String getPath() {
        return path;
    }

    /**
     * Set a custom request path for the build cache server specified by {@link #getServer()} or {@link GradleEnterpriseExtension#getServer()}.
     * <p>
     * It is generally not necessary to set this value, as the default value used is suitable for the default
     * configuration of Gradle Enterprise build cache nodes (including the built-in build cache).
     *
     * @param path a custom request path for the build cache server
     */
    public void setPath(@Nullable String path) {
        this.path = path;
    }

    /**
     * Whether it is acceptable to communicate with a build cache server with an untrusted SSL certificate.
     * <p>
     * If not set then will be inherited from {@link GradleEnterpriseExtension#getAllowUntrustedServer()}.
     *
     * @return {@code null} if the value should be inherited from {@link GradleEnterpriseExtension#getAllowUntrustedServer()}, or whether it is acceptable to communicate with a build scan server with an untrusted SSL certificate
     * @see #setAllowUntrustedServer(boolean)
     **/
    @Nullable
    public Boolean getAllowUntrustedServer() {
        return allowUntrustedServer;
    }

    /**
     * Specifies whether it is acceptable to communicate with the build cache server using an untrusted SSL certificate.
     * <p>
     * If the build cache server are uses an internally provisioned or self-signed certificate it will not be trusted by default.
     * Allowing communication with untrusted servers keeps data encrypted during transmission,
     * but makes it easy for a man-in-the-middle to impersonate the intended server and capture data.
     * <p>
     * In such a scenario, you can either configure the build JVM environment to trust the certificate,
     * or call this method with {@code true} to disable verification of the server's identity.
     * <p>
     * This value has no effect if communicating with the server using the HTTP protocol (i.e. has SSL disabled).
     *
     * @param allowUntrustedServer whether to allow communication with an HTTPS server with an untrusted certificate
     */
    public void setAllowUntrustedServer(boolean allowUntrustedServer) {
        this.allowUntrustedServer = allowUntrustedServer;
    }

    /**
     * Whether the use of plain/unencrypted HTTP is allowed when communicating with the build cache server.
     *
     * @return whether the use of plain/unencrypted HTTP is allowed when communicating with the build cache server
     * @see #setAllowInsecureProtocol(boolean)
     */
    public boolean getAllowInsecureProtocol() {
        return allowInsecureProtocol;
    }

    /**
     * Sets whether the use of plain/unencrypted HTTP is allowed when communicating with the build cache server.
     * <p>
     * This value defaults to {@code false}.
     * <p>
     * Use of unencrypted HTTP allows interception of build cache entries while in transit and is a
     * significant privacy risk. Use of HTTPS instead of enabling this setting is strongly recommended
     *
     * @param allowInsecureProtocol whether to allow the use of plain/unencrypted HTTP
     */
    public void setAllowInsecureProtocol(boolean allowInsecureProtocol) {
        this.allowInsecureProtocol = allowInsecureProtocol;
    }

    /**
     * Sets the username/password to use to authenticate with the build cache server.
     * <p>
     * This is a convenient variant of {@link #setUsernameAndPassword(UsernameAndPassword)}.
     *
     * @param username the username
     * @param password the password
     */
    public void usernameAndPassword(String username, String password) {
        setUsernameAndPassword(new UsernameAndPassword(username, password));
    }

    /**
     * Sets the username/password to use to authenticate with the build cache server.
     * <p>
     * If specified, the username/password will be used instead of any available Gradle Enterprise access key.
     * Use of access keys should be preferred.
     *
     * @param usernameAndPassword the username and password
     */
    public void setUsernameAndPassword(@Nullable UsernameAndPassword usernameAndPassword) {
        this.usernameAndPassword = usernameAndPassword;
    }

    /**
     * The username/password to use to authenticate with the build cache server.
     *
     * @return the username/password to use to authenticate with the build cache server
     * @see #setUsernameAndPassword(UsernameAndPassword)
     */
    @Nullable
    public UsernameAndPassword getUsernameAndPassword() {
        return usernameAndPassword;
    }

    /**
     * Whether to use <a href="https://www.rfc-editor.org/rfc/rfc9110.html#section-10.1.1">HTTP Expect-Continue</a> when storing data on the build cache server.
     *
     * @return whether the use of HTTP Expect-Continue is configured when communicating with the build cache server
     * @see #setUseExpectContinue(boolean)
     * @since 3.12
     */
    public boolean getUseExpectContinue() {
        return useExpectContinue;
    }

    /**
     * Sets whether to use <a href="https://www.rfc-editor.org/rfc/rfc9110.html#section-10.1.1">HTTP Expect-Continue</a> when storing data on the build cache server.
     * <p>
     * This causes PUT requests to happen in two parts: first a check whether a body would be accepted, then transmission of the body if the server indicates it will accept it.
     * This is particularly suitable for Build Cache servers that routinely redirect or reject PUT requests, as it avoids transmitting the cache entry just to have it rejected.
     * This additional check incurs extra latency when the server accepts the request, but reduces latency when the request is rejected or redirected.
     * <p>
     * This value defaults to {@code false}.
     * <p>
     * While the Gradle Enterprise Build Cache Node supports Expect-Continue, not all HTTP servers and proxies reliably do. Be sure to check that your Build Cache server does support it before enabling.
     *
     * @param useExpectContinue whether to use HTTP Expect-Continue
     * @since 3.12
     */
    public void setUseExpectContinue(boolean useExpectContinue) {
        this.useExpectContinue = useExpectContinue;
    }

    /**
     * A value type representing both a username and password.
     *
     * @see #setUsernameAndPassword(UsernameAndPassword)
     * @deprecated since 3.17, replaced by {@link com.gradle.develocity.agent.gradle.buildcache.DevelocityBuildCache.UsernameAndPassword}
     */
    @Deprecated
    public static final class UsernameAndPassword {

        private final String username;
        private final String password;

        public UsernameAndPassword(String username, String password) {
            this.username = checkNotNull(username, () -> "username cannot be null");
            this.password = checkNotNull(password, () -> "password cannot be null");
        }

        public static UsernameAndPassword of(String username, String password) {
            return new UsernameAndPassword(username, password);
        }

        public String getUsername() {
            return username;
        }

        public String getPassword() {
            return password;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            UsernameAndPassword that = (UsernameAndPassword) o;
            return username.equals(that.username) && password.equals(that.password);
        }

        @Override
        public int hashCode() {
            return Objects.hash(username, password);
        }
    }

}
