package com.gradle.develocity.agent.gradle.test;

import com.gradle.develocity.agent.gradle.internal.test.ImportJUnitXmlEvent;
import com.gradle.develocity.agent.gradle.internal.util.CurrentBuildAgentToolVersion;
import com.gradle.enterprise.version.gradle.GradleVersions;
import com.gradle.junit.xml.streaming.parser.ParserDialect;
import com.gradle.nullability.Nullable;
import com.gradle.obfuscation.Keep;
import com.gradle.obfuscation.KeepMethodNames;
import com.gradle.obfuscation.KeepName;
import org.gradle.api.DefaultTask;
import org.gradle.api.InvalidUserDataException;
import org.gradle.api.Task;
import org.gradle.api.file.ConfigurableFileCollection;
import org.gradle.api.internal.TaskInternal;
import org.gradle.api.internal.project.DefaultProjectRegistry;
import org.gradle.api.internal.project.ProjectInternal;
import org.gradle.api.internal.project.ProjectRegistry;
import org.gradle.api.internal.project.taskfactory.TaskIdentity;
import org.gradle.api.invocation.Gradle;
import org.gradle.api.model.ObjectFactory;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Internal;
import org.gradle.api.tasks.TaskAction;
import org.gradle.api.tasks.TaskContainer;
import org.gradle.api.tasks.TaskProvider;
import org.gradle.internal.operations.BuildOperationListener;
import org.gradle.internal.operations.BuildOperationListenerManager;
import org.gradle.internal.operations.CurrentBuildOperationRef;
import org.gradle.internal.operations.OperationProgressEvent;
import org.gradle.internal.scan.time.BuildScanClock;
import org.gradle.util.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.inject.Inject;
import java.io.File;
import java.util.Set;

import static com.gradle.enterprise.java.Checks.checkNotNull;
import static java.util.Objects.requireNonNull;

/**
 * Imports JUnit XML reports into the build's Develocity build scan.
 *
 * @since 3.17
 */
@Keep
@KeepName
@KeepMethodNames
@SuppressWarnings("WeakerAccess")
public abstract class ImportJUnitXmlReports extends DefaultTask {

    protected static final Logger LOGGER = LoggerFactory.getLogger(ImportJUnitXmlReports.class);
    private static final String TASK_NAME_SUFFIX = "ImportJUnitXmlReports";

    private final BuildOperationListener eventEmitter;
    private final BuildScanClock clock;
    @Nullable("when loaded from configuration cache")
    private final transient Property<Task> referenceTask;

    @SuppressWarnings("this-escape")
    @Inject
    public ImportJUnitXmlReports(
        BuildOperationListenerManager listenerManager,
        BuildScanClock clock,
        ObjectFactory objectFactory,
        Gradle gradle
    ) {
        this.eventEmitter = listenerManager.getBroadcaster();
        this.clock = clock;
        this.referenceTask = objectFactory.property(Task.class);
        getReferenceTaskBuildPath().convention(referenceTask.map(it -> getBuildPath((TaskInternal) it)));
        getReferenceTaskTaskPath().convention(referenceTask.map(it -> getTaskPath((TaskInternal) it)));
        if (CurrentBuildAgentToolVersion.get(gradle).isAtLeast(GradleVersions.V7_4)) { // notCompatibleWithConfigurationCache did not exist before Gradle 7.4
            gradle.getTaskGraph().whenReady(__ -> {
                // This is after this task has been configured and before the configuration cache will check for an incompatibility reason
                if (!isPartOfSameBuildAsReferenceTask()) {
                    notCompatibleWithConfigurationCache("Referencing tasks across included builds is not supported");
                }
            });
        }
    }

    /**
     * Configures the JUnit XML dialect.
     *
     * @see JUnitXmlDialect
     */
    @Input
    public abstract Property<JUnitXmlDialect> getDialect();

    // Use internal for `getReports` and `getReferenceTask` so that they don't become inputs
    // gradle will not execute finalizer tasks if they have an input dependency
    // on the task they are finalizing and that task has failed
    // https://github.com/gradle/gradle/issues/17633

    /**
     * The JUnit XML report files that should be imported.
     */
    @Internal
    public abstract ConfigurableFileCollection getReports();

    /**
     * The reference task to which the tests should be attributed.
     * <p>
     * Can be omitted, in which case the tests will be attributed to the import task.
     */
    @Internal
    @Nullable
    public Property<Task> getReferenceTask() {
        return referenceTask;
    }

    @Internal
    public abstract Property<String> getReferenceTaskBuildPath();

    @Internal
    public abstract Property<String> getReferenceTaskTaskPath();

    @TaskAction
    void importReports() {
        Task referenceTask = findReferenceTask();
        if (referenceTask != null && !referenceTask.getDidWork()) {
            setDidWork(false);
            return;
        }
        TaskInternal taskInternal = referenceTask == null ? this : (TaskInternal) referenceTask;
        TaskIdentity<?> referenceTaskIdentity = taskInternal.getTaskIdentity();
        Set<File> xmlFiles = getReports()
            .filter(File::exists)
            .filter(file -> file.getName().endsWith(".xml"))
            .getFiles();
        if (xmlFiles.isEmpty()) {
            LOGGER.warn("No JUnit XML reports found in any of the declared inputs!");
        } else {
            ParserDialect dialect = getDialect().get().toParserDialect();
            xmlFiles.stream()
                .map(xml -> new TaskImportJUnitXmlEvent(xml.getAbsoluteFile(), referenceTaskIdentity, dialect))
                .forEach(this::emitEvent);
        }
    }

    @Nullable
    private Task findReferenceTask() {
        if (referenceTask == null) {
            String buildPath = getReferenceTaskBuildPath().getOrNull();
            String taskPath = getReferenceTaskTaskPath().getOrNull();
            if (buildPath == null || taskPath == null) {
                return null;
            }
            if (isPartOfSameBuildAsReferenceTask()) {
                Path referencePath = Path.path(taskPath);
                String referenceProjectPath = requireNonNull(referencePath.getParent()).getPath();
                ProjectInternal referenceProject = requireNonNull(getProjectRegistry().getProject(referenceProjectPath));
                String referenceTaskName = requireNonNull(referencePath.getName());
                TaskContainer tasks = referenceProject.getServices().get(TaskContainer.class);
                return tasks.getByName(referenceTaskName);
            }
            throw new InvalidUserDataException("Referencing tasks across included builds is not supported when using the configuration cache: " + taskPath + " in " + buildPath);
        }
        return referenceTask.getOrNull();
    }

    private boolean isPartOfSameBuildAsReferenceTask() {
        String referenceBuildPath = getReferenceTaskBuildPath().getOrNull();
        return referenceBuildPath == null || getBuildPath(this).equals(referenceBuildPath);
    }

    @SuppressWarnings("unchecked")
    private ProjectRegistry<ProjectInternal> getProjectRegistry() {
        return getServices().get(DefaultProjectRegistry.class);
    }

    // Inlined https://github.com/gradle/gradle/blob/a2b8e0a7ef47c8359beec3e9ffafaa241645a0a8/subprojects/base-services/src/main/java/org/gradle/internal/operations/BuildOperationProgressEventEmitter.java#L60-L71
    // to support Gradle versions <6.6
    private void emitEvent(ImportJUnitXmlEvent event) {
        eventEmitter.progress(
            checkNotNull(CurrentBuildOperationRef.instance().getId(), () -> "operationIdentifier is null"),
            new OperationProgressEvent(clock.getCurrentTime(), event)
        );
    }

    /**
     * Helper method to lazily register and configure an import task.
     * <p>
     * <pre><code>
     * afterEvaluate {
     *     ImportJUnitXmlReports.register(tasks, tasks.named("connectedDebugAndroidTest"), com.gradle.scan.plugin.test.JUnitXmlDialect.ANDROID_CONNECTED)
     * }
     * </code></pre>
     * <p>
     * Or if the given task doesn't declare the outputs automatically, we can specify the reports explicitly, for example:
     * <p>
     * <pre><code>
     * afterEvaluate {
     *     ImportJUnitXmlReports.register(tasks, tasks.named("execFlank"), com.gradle.scan.plugin.test.JUnitXmlDialect.ANDROID_FIREBASE)
     *         .configure { reports.from(file("${buildDir}/fladle/outputs/JUnitReport.xml")) }
     * }
     * </code></pre>
     * <p>
     * The resulting task uses the name of the supplied {@code testTask} appended with the {@value #TASK_NAME_SUFFIX} suffix,
     * will scan its output files for JUnit XML reports, and run as one of its finalizer tasks.
     *
     * @param tasks    the gradle task container, i.e., {@code project.tasks}
     * @param testTask the task that produces the JUnit XML, the tests will also be attributed to this task
     * @param dialect  the dialect of the JUnit XML, see {@link JUnitXmlDialect} for more information.
     * @return the task provider, can be used to manually set the report files for tasks that don't declare outputs.
     */
    @SuppressWarnings("unused") // used in buildscripts
    public static TaskProvider<ImportJUnitXmlReports> register(TaskContainer tasks, TaskProvider<?> testTask, JUnitXmlDialect dialect) {
        TaskProvider<ImportJUnitXmlReports> finalizerTask = registerFinalizerTask(tasks, testTask, dialect);
        testTask.configure(task -> task.finalizedBy(finalizerTask));
        return finalizerTask;
    }

    private static TaskProvider<ImportJUnitXmlReports> registerFinalizerTask(TaskContainer tasks, TaskProvider<?> testTask, JUnitXmlDialect dialect) {
        return tasks.register(
            testTask.getName() + TASK_NAME_SUFFIX,
            ImportJUnitXmlReports.class,
            importTask -> {
                importTask.getDialect().set(dialect);
                requireNonNull(importTask.getReferenceTask()).set(testTask);
                importTask.getReports().from(testTask.map(task -> task.getOutputs().getFiles().getAsFileTree()));
            }
        );
    }

    private static String getTaskPath(TaskInternal task) {
        return task.getTaskIdentity().projectPath.getPath();
    }

    private static String getBuildPath(TaskInternal task) {
        return task.getTaskIdentity().buildPath.getPath();
    }

    private static class TaskImportJUnitXmlEvent implements ImportJUnitXmlEvent {

        private final File xmlFile;
        private final TaskIdentity<?> referenceTask;
        private final ParserDialect dialect;

        private TaskImportJUnitXmlEvent(File xmlFile, TaskIdentity<?> referenceTask, ParserDialect dialect) {
            this.xmlFile = xmlFile;
            this.referenceTask = referenceTask;
            this.dialect = dialect;
        }

        @Override
        public File getXmlFile() {
            return xmlFile;
        }

        @Override
        public TaskIdentity<?> getReferenceTask() {
            return referenceTask;
        }

        @Override
        public ParserDialect getDialect() {
            return dialect;
        }
    }
}
