package com.gradle.develocity.agent.gradle;

import com.gradle.develocity.agent.gradle.buildcache.DevelocityBuildCache;
import com.gradle.develocity.agent.gradle.scan.BuildScanConfiguration;
import com.gradle.obfuscation.Keep;
import org.gradle.api.Action;
import org.gradle.api.NonExtensible;
import org.gradle.api.provider.Property;
import org.gradle.caching.configuration.BuildCacheConfiguration;

/**
 * The DSL extension for configuring Develocity, with the name {@code develocity}.
 *
 * <pre>{@code
 * develocity {
 *   // Develocity configuration
 * }
 * }</pre>
 *
 * @since 3.17
 */
@Keep
@NonExtensible
public interface DevelocityConfiguration {

    /**
     * The build scan configuration.
     *
     * @return the build scan configuration
     */
    BuildScanConfiguration getBuildScan();

    /**
     * Configure build scans.
     *
     * @param action the configuration action
     */
    void buildScan(Action<? super BuildScanConfiguration> action);

    /**
     * The URL of the Develocity server.
     */
    Property<String> getServer();

    /**
     * The project identifier.
     */
    Property<String> getProjectId();

    /**
     * Specifies whether it is acceptable to communicate with a Develocity server using an untrusted SSL certificate.
     * <p>
     * The default (public) build scan server uses SSL certificates that are trusted by default by standard modern Java environments.
     * If you are using a different build scan server via Develocity, it may use an untrusted certificate.
     * This may be due to the use of an internally provisioned or self-signed certificate.
     * <p>
     * In such a scenario, you can either configure the build JVM environment to trust the certificate,
     * or call this method with {@code true} to disable verification of the server's identity.
     * Alternatively, you may disable SSL completely for Develocity installation but this is not recommended.
     * <p>
     * Allowing communication with untrusted servers keeps data encrypted during transmission,
     * but makes it easy for a man-in-the-middle to impersonate the intended server and capture data.
     * <p>
     * This value has no effect if a server is specified using the HTTP protocol (i.e. has SSL disabled).
     **/
    Property<Boolean> getAllowUntrustedServer();

    /**
     * The access key for authenticating with the Develocity server.
     * <p>
     * Only the value of the explicitly configured access key is returned but
     * not the value of an access key configured via the {@code DEVELOCITY_ACCESS_KEY} environment variable
     * or access key file entry.
     * <p>
     * An access key configured this way will take precedence over the {@code DEVELOCITY_ACCESS_KEY}
     * environment variable or access key file entry associated with the server.
     *
     * @return the configured Develocity server access key, if available; otherwise, {@code null}
     */
    Property<String> getAccessKey();

    /**
     * The type to be used for registering the Develocity build cache with Gradle.
     * <p>
     * This should be used with {@link BuildCacheConfiguration#remote(Class, Action)} to enable
     * use of the Develocity remote build caching.
     * <p>
     * The following example demonstrates usage in a build's settings script:
     * <p>
     * <pre>{@code
     * buildCache {
     *   remote(develocity.buildCache) {
     *     // configure
     *   }
     * }
     * }</pre>
     *
     * @return the type to be used for registering the Develocity build cache with Gradle
     * @see DevelocityBuildCache
     */
    Class<? extends DevelocityBuildCache> getBuildCache();
}
