/*
 * Copyright (c) Terl Tech Ltd • 08/05/18 22:59 • goterl.com
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v2.0. If a copy of the MPL was not distributed with this
 * file, you can obtain one at http://mozilla.org/MPL/2.0/.
 */

package com.goterl.lazycode.lazysodium.interfaces;


import com.goterl.lazycode.lazysodium.exceptions.SodiumException;
import com.goterl.lazycode.lazysodium.utils.BaseChecker;

public interface Auth {

    int HMACSHA512256_BYTES = 32,
        HMACSHA512256_KEYBYTES = 32,

        BYTES = HMACSHA512256_BYTES,
        KEYBYTES = HMACSHA512256_KEYBYTES;


    class Checker extends BaseChecker {

    }

    interface Native {

        /**
         * Generate an authentication key.
         * @param k Auth key of size {@link #KEYBYTES}.
         */
        void cryptoAuthKeygen(byte[] k);

        /**
         * Computes a tag for the message in, whose length is inLen bytes, and the key k.
         * @param tag Tag of size {@link #BYTES}.
         * @param in A message.
         * @param inLen Message size.
         * @param key The key as generated by {@link #cryptoAuthKeygen(byte[])}.
         * @return True if successful.
         */
        boolean cryptoAuth(byte[] tag, byte[] in, long inLen, byte[] key);

        /**
         * Verifies that the tag stored at h is a
         * valid tag for the message in whose length
         * is inLen bytes, and the key k.
         * @param tag The tag.
         * @param in The message.
         * @param inLen Message bytes.
         * @param key The key as generated by {@link #cryptoAuthKeygen(byte[])}.
         * @return True if successful verification.
         */
        boolean cryptoAuthVerify(byte[] tag, byte[] in, long inLen, byte[] key);
    }

    interface Lazy {

        /**
         * Generate an authentication key.
         * @return An authentication key.
         */
        String cryptoAuthKeygen();

        /**
         * Computes a tag for the message in.
         * @param message A message.
         * @param key The key as generated by {@link #cryptoAuthKeygen()}.
         * @return True if successful.
         */
        String cryptoAuth(String message, String key) throws SodiumException;


        /**
         * Verifies that the tag
         * valid tag for the message.
         * @param tag The tag.
         * @param key The key as generated by {@link #cryptoAuthKeygen()}.
         * @return True if successful verification.
         */
        boolean cryptoAuthVerify(String tag, String message, String key) throws SodiumException;

    }


}
