/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.local;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteStatement;
import android.text.TextUtils;
import androidx.annotation.VisibleForTesting;
import com.google.firebase.firestore.local.EncodedPath;
import com.google.firebase.firestore.local.LocalSerializer;
import com.google.firebase.firestore.local.MemoryIndexManager;
import com.google.firebase.firestore.local.Persistence;
import com.google.firebase.firestore.local.SQLitePersistence;
import com.google.firebase.firestore.local.TargetData;
import com.google.firebase.firestore.model.ResourcePath;
import com.google.firebase.firestore.proto.Target;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.Consumer;
import com.google.firebase.firestore.util.Logger;
import com.google.protobuf.InvalidProtocolBufferException;
import java.util.ArrayList;
import java.util.List;

class SQLiteSchema {
    static final int VERSION = 16;
    @VisibleForTesting
    static final int MIGRATION_BATCH_SIZE = 100;
    private final SQLiteDatabase db;
    private final LocalSerializer serializer;

    SQLiteSchema(SQLiteDatabase db, LocalSerializer serializer) {
        this.db = db;
        this.serializer = serializer;
    }

    void runSchemaUpgrades() {
        this.runSchemaUpgrades(0);
    }

    void runSchemaUpgrades(int fromVersion) {
        this.runSchemaUpgrades(fromVersion, 16);
    }

    void runSchemaUpgrades(int fromVersion, int toVersion) {
        long startTime = System.currentTimeMillis();
        if (fromVersion < 1 && toVersion >= 1) {
            this.createV1MutationQueue();
            this.createV1TargetCache();
            this.createV1RemoteDocumentCache();
        }
        if (fromVersion < 3 && toVersion >= 3 && fromVersion != 0) {
            this.dropV1TargetCache();
            this.createV1TargetCache();
        }
        if (fromVersion < 4 && toVersion >= 4) {
            this.ensureTargetGlobal();
            this.addTargetCount();
        }
        if (fromVersion < 5 && toVersion >= 5) {
            this.addSequenceNumber();
        }
        if (fromVersion < 6 && toVersion >= 6) {
            this.removeAcknowledgedMutations();
        }
        if (fromVersion < 7 && toVersion >= 7) {
            this.ensureSequenceNumbers();
        }
        if (fromVersion < 8 && toVersion >= 8) {
            this.createV8CollectionParentsIndex();
        }
        if (fromVersion < 9 && toVersion >= 9) {
            if (!this.hasReadTime()) {
                this.addReadTime();
            } else {
                this.dropLastLimboFreeSnapshotVersion();
            }
        }
        if (fromVersion == 9 && toVersion >= 10) {
            this.dropLastLimboFreeSnapshotVersion();
        }
        if (fromVersion < 11 && toVersion >= 11) {
            this.rewriteCanonicalIds();
        }
        if (fromVersion < 12 && toVersion >= 12) {
            this.createBundleCache();
        }
        if (fromVersion < 13 && toVersion >= 13) {
            this.addPathLength();
            this.ensurePathLength();
        }
        if (fromVersion < 14 && toVersion >= 14) {
            this.createOverlays();
            this.createDataMigrationTable();
            this.addPendingDataMigration(Persistence.DATA_MIGRATION_BUILD_OVERLAYS);
        }
        if (fromVersion < 15 && toVersion >= 15) {
            this.ensureReadTime();
        }
        if (fromVersion < 16 && toVersion >= 16) {
            this.createFieldIndex();
        }
        Logger.debug("SQLiteSchema", "Migration from version %s to %s took %s milliseconds", fromVersion, toVersion, System.currentTimeMillis() - startTime);
    }

    private void ifTablesDontExist(String[] tables, Runnable fn) {
        boolean tablesFound = false;
        String allTables = "[" + TextUtils.join((CharSequence)", ", (Object[])tables) + "]";
        for (int i = 0; i < tables.length; ++i) {
            String table = tables[i];
            boolean tableFound = this.tableExists(table);
            if (i == 0) {
                tablesFound = tableFound;
                continue;
            }
            if (tableFound == tablesFound) continue;
            String msg = "Expected all of " + allTables + " to either exist or not, but ";
            msg = tablesFound ? msg + tables[0] + " exists and " + table + " does not" : msg + tables[0] + " does not exist and " + table + " does";
            throw new IllegalStateException(msg);
        }
        if (!tablesFound) {
            fn.run();
        } else {
            Logger.debug("SQLiteSchema", "Skipping migration because all of " + allTables + " already exist", new Object[0]);
        }
    }

    private void createV1MutationQueue() {
        this.ifTablesDontExist(new String[]{"mutation_queues", "mutations", "document_mutations"}, () -> {
            this.db.execSQL("CREATE TABLE mutation_queues (uid TEXT PRIMARY KEY, last_acknowledged_batch_id INTEGER, last_stream_token BLOB)");
            this.db.execSQL("CREATE TABLE mutations (uid TEXT, batch_id INTEGER, mutations BLOB, PRIMARY KEY (uid, batch_id))");
            this.db.execSQL("CREATE TABLE document_mutations (uid TEXT, path TEXT, batch_id INTEGER, PRIMARY KEY (uid, path, batch_id))");
        });
    }

    private void removeAcknowledgedMutations() {
        SQLitePersistence.Query mutationQueuesQuery = new SQLitePersistence.Query(this.db, "SELECT uid, last_acknowledged_batch_id FROM mutation_queues");
        mutationQueuesQuery.forEach(mutationQueueEntry -> {
            String uid = mutationQueueEntry.getString(0);
            long lastAcknowledgedBatchId = mutationQueueEntry.getLong(1);
            SQLitePersistence.Query mutationsQuery = new SQLitePersistence.Query(this.db, "SELECT batch_id FROM mutations WHERE uid = ? AND batch_id <= ?").binding(uid, lastAcknowledgedBatchId);
            mutationsQuery.forEach(value -> this.removeMutationBatch(uid, value.getInt(0)));
        });
    }

    private void removeMutationBatch(String uid, int batchId) {
        SQLiteStatement mutationDeleter = this.db.compileStatement("DELETE FROM mutations WHERE uid = ? AND batch_id = ?");
        mutationDeleter.bindString(1, uid);
        mutationDeleter.bindLong(2, (long)batchId);
        int deleted = mutationDeleter.executeUpdateDelete();
        Assert.hardAssert(deleted != 0, "Mutation batch (%s, %d) did not exist", uid, batchId);
        this.db.execSQL("DELETE FROM document_mutations WHERE uid = ? AND batch_id = ?", new Object[]{uid, batchId});
    }

    private void createV1TargetCache() {
        this.ifTablesDontExist(new String[]{"targets", "target_globals", "target_documents"}, () -> {
            this.db.execSQL("CREATE TABLE targets (target_id INTEGER PRIMARY KEY, canonical_id TEXT, snapshot_version_seconds INTEGER, snapshot_version_nanos INTEGER, resume_token BLOB, last_listen_sequence_number INTEGER,target_proto BLOB)");
            this.db.execSQL("CREATE INDEX query_targets ON targets (canonical_id, target_id)");
            this.db.execSQL("CREATE TABLE target_globals (highest_target_id INTEGER, highest_listen_sequence_number INTEGER, last_remote_snapshot_version_seconds INTEGER, last_remote_snapshot_version_nanos INTEGER)");
            this.db.execSQL("CREATE TABLE target_documents (target_id INTEGER, path TEXT, PRIMARY KEY (target_id, path))");
            this.db.execSQL("CREATE INDEX document_targets ON target_documents (path, target_id)");
        });
    }

    private void dropV1TargetCache() {
        if (this.tableExists("targets")) {
            this.db.execSQL("DROP TABLE targets");
        }
        if (this.tableExists("target_globals")) {
            this.db.execSQL("DROP TABLE target_globals");
        }
        if (this.tableExists("target_documents")) {
            this.db.execSQL("DROP TABLE target_documents");
        }
    }

    private void createV1RemoteDocumentCache() {
        this.ifTablesDontExist(new String[]{"remote_documents"}, () -> this.db.execSQL("CREATE TABLE remote_documents (path TEXT PRIMARY KEY, contents BLOB)"));
    }

    private void createFieldIndex() {
        this.ifTablesDontExist(new String[]{"index_configuration", "index_state", "index_entries"}, () -> {
            this.db.execSQL("CREATE TABLE index_configuration (index_id INTEGER, collection_group TEXT, index_proto BLOB, PRIMARY KEY (index_id))");
            this.db.execSQL("CREATE TABLE index_state (index_id INTEGER, uid TEXT, sequence_number INTEGER, read_time_seconds INTEGER, read_time_nanos INTEGER, document_key TEXT, largest_batch_id INTEGER, PRIMARY KEY (index_id, uid))");
            this.db.execSQL("CREATE TABLE index_entries (index_id INTEGER, uid TEXT, array_value BLOB, directional_value BLOB, document_key TEXT, PRIMARY KEY (index_id, uid, array_value, directional_value, document_key))");
            this.db.execSQL("CREATE INDEX read_time ON remote_documents(read_time_seconds, read_time_nanos)");
        });
    }

    private void ensureTargetGlobal() {
        boolean targetGlobalExists;
        boolean bl = targetGlobalExists = DatabaseUtils.queryNumEntries((SQLiteDatabase)this.db, (String)"target_globals") == 1L;
        if (!targetGlobalExists) {
            this.db.execSQL("INSERT INTO target_globals (highest_target_id, highest_listen_sequence_number, last_remote_snapshot_version_seconds, last_remote_snapshot_version_nanos) VALUES (?, ?, ?, ?)", (Object[])new String[]{"0", "0", "0", "0"});
        }
    }

    private void addTargetCount() {
        if (!this.tableContainsColumn("target_globals", "target_count")) {
            this.db.execSQL("ALTER TABLE target_globals ADD COLUMN target_count INTEGER");
        }
        long count = DatabaseUtils.queryNumEntries((SQLiteDatabase)this.db, (String)"targets");
        ContentValues cv = new ContentValues();
        cv.put("target_count", Long.valueOf(count));
        this.db.update("target_globals", cv, null, null);
    }

    private void addSequenceNumber() {
        if (!this.tableContainsColumn("target_documents", "sequence_number")) {
            this.db.execSQL("ALTER TABLE target_documents ADD COLUMN sequence_number INTEGER");
        }
    }

    private void addPathLength() {
        if (!this.tableContainsColumn("remote_documents", "path_length")) {
            this.db.execSQL("ALTER TABLE remote_documents ADD COLUMN path_length INTEGER");
        }
    }

    private boolean hasReadTime() {
        boolean hasReadTimeNanos;
        boolean hasReadTimeSeconds = this.tableContainsColumn("remote_documents", "read_time_seconds");
        Assert.hardAssert(hasReadTimeSeconds == (hasReadTimeNanos = this.tableContainsColumn("remote_documents", "read_time_nanos")), "Table contained just one of read_time_seconds or read_time_nanos", new Object[0]);
        return hasReadTimeSeconds && hasReadTimeNanos;
    }

    private void addReadTime() {
        this.db.execSQL("ALTER TABLE remote_documents ADD COLUMN read_time_seconds INTEGER");
        this.db.execSQL("ALTER TABLE remote_documents ADD COLUMN read_time_nanos INTEGER");
    }

    private void dropLastLimboFreeSnapshotVersion() {
        new SQLitePersistence.Query(this.db, "SELECT target_id, target_proto FROM targets").forEach(cursor -> {
            int targetId = cursor.getInt(0);
            byte[] targetProtoBytes = cursor.getBlob(1);
            try {
                Target targetProto = Target.parseFrom(targetProtoBytes);
                targetProto = (Target)((Target.Builder)targetProto.toBuilder()).clearLastLimboFreeSnapshotVersion().build();
                this.db.execSQL("UPDATE targets SET target_proto = ? WHERE target_id = ?", new Object[]{targetProto.toByteArray(), targetId});
            }
            catch (InvalidProtocolBufferException e) {
                throw Assert.fail("Failed to decode Query data for target %s", targetId);
            }
        });
    }

    private void ensureSequenceNumbers() {
        SQLitePersistence.Query sequenceNumberQuery = new SQLitePersistence.Query(this.db, "SELECT highest_listen_sequence_number FROM target_globals LIMIT 1");
        Long boxedSequenceNumber = (Long)sequenceNumberQuery.firstValue(c -> c.getLong(0));
        Assert.hardAssert(boxedSequenceNumber != null, "Missing highest sequence number", new Object[0]);
        long sequenceNumber = boxedSequenceNumber;
        SQLiteStatement tagDocument = this.db.compileStatement("INSERT INTO target_documents (target_id, path, sequence_number) VALUES (0, ?, ?)");
        SQLitePersistence.Query untaggedDocumentsQuery = new SQLitePersistence.Query(this.db, "SELECT RD.path FROM remote_documents AS RD WHERE NOT EXISTS (SELECT TD.path FROM target_documents AS TD WHERE RD.path = TD.path AND TD.target_id = 0) LIMIT ?").binding(100);
        boolean[] resultsRemaining = new boolean[]{false};
        do {
            untaggedDocumentsQuery.forEach(row -> {
                resultsRemaining[0] = true;
                tagDocument.clearBindings();
                tagDocument.bindString(1, row.getString(0));
                tagDocument.bindLong(2, sequenceNumber);
                Assert.hardAssert(tagDocument.executeInsert() != -1L, "Failed to insert a sentinel row", new Object[0]);
            });
        } while (resultsRemaining[0]);
    }

    private void createV8CollectionParentsIndex() {
        this.ifTablesDontExist(new String[]{"collection_parents"}, () -> this.db.execSQL("CREATE TABLE collection_parents (collection_id TEXT, parent TEXT, PRIMARY KEY(collection_id, parent))"));
        MemoryIndexManager.MemoryCollectionParentIndex cache = new MemoryIndexManager.MemoryCollectionParentIndex();
        SQLiteStatement addIndexEntry = this.db.compileStatement("INSERT OR REPLACE INTO collection_parents (collection_id, parent) VALUES (?, ?)");
        Consumer<ResourcePath> addEntry = collectionPath -> {
            if (cache.add((ResourcePath)collectionPath)) {
                String collectionId = collectionPath.getLastSegment();
                ResourcePath parentPath = (ResourcePath)collectionPath.popLast();
                addIndexEntry.clearBindings();
                addIndexEntry.bindString(1, collectionId);
                addIndexEntry.bindString(2, EncodedPath.encode(parentPath));
                addIndexEntry.execute();
            }
        };
        SQLitePersistence.Query remoteDocumentsQuery = new SQLitePersistence.Query(this.db, "SELECT path FROM remote_documents");
        remoteDocumentsQuery.forEach(row -> {
            ResourcePath path = EncodedPath.decodeResourcePath(row.getString(0));
            addEntry.accept((ResourcePath)path.popLast());
        });
        SQLitePersistence.Query documentMutationsQuery = new SQLitePersistence.Query(this.db, "SELECT path FROM document_mutations");
        documentMutationsQuery.forEach(row -> {
            ResourcePath path = EncodedPath.decodeResourcePath(row.getString(0));
            addEntry.accept((ResourcePath)path.popLast());
        });
    }

    private boolean tableContainsColumn(String table, String column) {
        List<String> columns = this.getTableColumns(table);
        return columns.indexOf(column) != -1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    List<String> getTableColumns(String table) {
        ArrayList<String> columns = new ArrayList<String>();
        try (Cursor c = null;){
            c = this.db.rawQuery("PRAGMA table_info(" + table + ")", null);
            int nameIndex = c.getColumnIndex("name");
            while (c.moveToNext()) {
                columns.add(c.getString(nameIndex));
            }
        }
        return columns;
    }

    private void rewriteCanonicalIds() {
        new SQLitePersistence.Query(this.db, "SELECT target_id, target_proto FROM targets").forEach(cursor -> {
            int targetId = cursor.getInt(0);
            byte[] targetProtoBytes = cursor.getBlob(1);
            try {
                Target targetProto = Target.parseFrom(targetProtoBytes);
                TargetData targetData = this.serializer.decodeTargetData(targetProto);
                String updatedCanonicalId = targetData.getTarget().getCanonicalId();
                this.db.execSQL("UPDATE targets SET canonical_id  = ? WHERE target_id = ?", new Object[]{updatedCanonicalId, targetId});
            }
            catch (InvalidProtocolBufferException e) {
                throw Assert.fail("Failed to decode Query data for target %s", targetId);
            }
        });
    }

    private void ensurePathLength() {
        SQLitePersistence.Query documentsToMigrate = new SQLitePersistence.Query(this.db, "SELECT path FROM remote_documents WHERE path_length IS NULL LIMIT ?").binding(100);
        SQLiteStatement insertKey = this.db.compileStatement("UPDATE remote_documents SET path_length = ? WHERE path = ?");
        boolean[] resultsRemaining = new boolean[]{false};
        do {
            documentsToMigrate.forEach(row -> {
                resultsRemaining[0] = true;
                String encodedPath = row.getString(0);
                ResourcePath decodedPath = EncodedPath.decodeResourcePath(encodedPath);
                insertKey.clearBindings();
                insertKey.bindLong(1, (long)decodedPath.length());
                insertKey.bindString(2, encodedPath);
                Assert.hardAssert(insertKey.executeUpdateDelete() != -1, "Failed to update document path", new Object[0]);
            });
        } while (resultsRemaining[0]);
    }

    private void ensureReadTime() {
        this.db.execSQL("UPDATE remote_documents SET read_time_seconds = 0, read_time_nanos = 0 WHERE read_time_seconds IS NULL");
    }

    private void createBundleCache() {
        this.ifTablesDontExist(new String[]{"bundles", "named_queries"}, () -> {
            this.db.execSQL("CREATE TABLE bundles (bundle_id TEXT PRIMARY KEY, create_time_seconds INTEGER, create_time_nanos INTEGER, schema_version INTEGER, total_documents INTEGER, total_bytes INTEGER)");
            this.db.execSQL("CREATE TABLE named_queries (name TEXT PRIMARY KEY, read_time_seconds INTEGER, read_time_nanos INTEGER, bundled_query_proto BLOB)");
        });
    }

    private void createOverlays() {
        this.ifTablesDontExist(new String[]{"document_overlays"}, () -> {
            this.db.execSQL("CREATE TABLE document_overlays (uid TEXT, collection_path TEXT, document_id TEXT, collection_group TEXT, largest_batch_id INTEGER, overlay_mutation BLOB, PRIMARY KEY (uid, collection_path, document_id))");
            this.db.execSQL("CREATE INDEX batch_id_overlay ON document_overlays (uid, largest_batch_id)");
            this.db.execSQL("CREATE INDEX collection_group_overlay ON document_overlays (uid, collection_group)");
        });
    }

    private void createDataMigrationTable() {
        this.ifTablesDontExist(new String[]{"data_migrations"}, () -> this.db.execSQL("CREATE TABLE data_migrations (migration_name TEXT, PRIMARY KEY (migration_name))"));
    }

    private void addPendingDataMigration(String migration) {
        this.db.execSQL("INSERT OR IGNORE INTO data_migrations (migration_name) VALUES (?)", (Object[])new String[]{migration});
    }

    private boolean tableExists(String table) {
        return !new SQLitePersistence.Query(this.db, "SELECT 1=1 FROM sqlite_master WHERE tbl_name = ?").binding(table).isEmpty();
    }
}

