/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.crashlytics.ndk;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.google.firebase.crashlytics.internal.Logger;
import com.google.firebase.crashlytics.internal.common.CommonUtils;
import com.google.firebase.crashlytics.internal.model.StaticSessionData;
import com.google.firebase.crashlytics.internal.persistence.FileStore;
import com.google.firebase.crashlytics.ndk.NativeApi;
import com.google.firebase.crashlytics.ndk.SessionFiles;
import com.google.firebase.crashlytics.ndk.SessionMetadataJsonSerializer;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;

public class CrashpadController {
    private static final Charset UTF_8 = Charset.forName("UTF-8");
    private static final String SESSION_METADATA_FILE = "session.json";
    private static final String APP_METADATA_FILE = "app.json";
    private static final String DEVICE_METADATA_FILE = "device.json";
    private static final String OS_METADATA_FILE = "os.json";
    private final Context context;
    private final NativeApi nativeApi;
    private final FileStore fileStore;

    CrashpadController(Context context, NativeApi nativeApi, FileStore fileStore) {
        this.context = context;
        this.nativeApi = nativeApi;
        this.fileStore = fileStore;
    }

    public boolean initialize(String sessionId, String generator, long startedAtSeconds, StaticSessionData sessionData) {
        File crashReportDirectory = this.fileStore.getNativeSessionDir(sessionId);
        try {
            String crashReportPath;
            if (crashReportDirectory != null && this.nativeApi.initialize(crashReportPath = crashReportDirectory.getCanonicalPath(), this.context.getAssets())) {
                this.writeBeginSession(sessionId, generator, startedAtSeconds);
                this.writeSessionApp(sessionId, sessionData.appData());
                this.writeSessionOs(sessionId, sessionData.osData());
                this.writeSessionDevice(sessionId, sessionData.deviceData());
                return true;
            }
        }
        catch (IOException e) {
            Logger.getLogger().e("Error initializing Crashlytics NDK", (Throwable)e);
        }
        return false;
    }

    public boolean hasCrashDataForSession(String sessionId) {
        File crashFile = this.getFilesForSession((String)sessionId).minidump;
        return crashFile != null && crashFile.exists();
    }

    @NonNull
    public SessionFiles getFilesForSession(String sessionId) {
        File sessionFileDirectory = this.fileStore.getNativeSessionDir(sessionId);
        File sessionFileDirectoryForMinidump = new File(sessionFileDirectory, "pending");
        Logger.getLogger().v("Minidump directory: " + sessionFileDirectoryForMinidump.getAbsolutePath());
        File minidump = CrashpadController.getSingleFileWithExtension(sessionFileDirectoryForMinidump, ".dmp");
        Logger.getLogger().v("Minidump file " + (minidump != null && minidump.exists() ? "exists" : "does not exist"));
        SessionFiles.Builder builder = new SessionFiles.Builder();
        if (sessionFileDirectory != null && sessionFileDirectory.exists() && sessionFileDirectoryForMinidump.exists()) {
            builder.minidumpFile(CrashpadController.getSingleFileWithExtension(sessionFileDirectoryForMinidump, ".dmp")).metadataFile(CrashpadController.getSingleFileWithExtension(sessionFileDirectory, ".device_info")).sessionFile(new File(sessionFileDirectory, SESSION_METADATA_FILE)).appFile(new File(sessionFileDirectory, APP_METADATA_FILE)).deviceFile(new File(sessionFileDirectory, DEVICE_METADATA_FILE)).osFile(new File(sessionFileDirectory, OS_METADATA_FILE));
        }
        return builder.build();
    }

    public void writeBeginSession(String sessionId, String generator, long startedAtSeconds) {
        String json = SessionMetadataJsonSerializer.serializeBeginSession(sessionId, generator, startedAtSeconds);
        CrashpadController.writeSessionJsonFile(this.fileStore, sessionId, json, SESSION_METADATA_FILE);
    }

    public void writeSessionApp(String sessionId, StaticSessionData.AppData appData) {
        String json = SessionMetadataJsonSerializer.serializeSessionApp(appData.appIdentifier(), appData.versionCode(), appData.versionName(), appData.installUuid(), appData.deliveryMechanism(), appData.developmentPlatformProvider().getDevelopmentPlatform(), appData.developmentPlatformProvider().getDevelopmentPlatformVersion());
        CrashpadController.writeSessionJsonFile(this.fileStore, sessionId, json, APP_METADATA_FILE);
    }

    public void writeSessionOs(String sessionId, StaticSessionData.OsData osData) {
        String json = SessionMetadataJsonSerializer.serializeSessionOs(osData.osRelease(), osData.osCodeName(), osData.isRooted());
        CrashpadController.writeSessionJsonFile(this.fileStore, sessionId, json, OS_METADATA_FILE);
    }

    public void writeSessionDevice(String sessionId, StaticSessionData.DeviceData deviceData) {
        String json = SessionMetadataJsonSerializer.serializeSessionDevice(deviceData.arch(), deviceData.model(), deviceData.availableProcessors(), deviceData.totalRam(), deviceData.diskSpace(), deviceData.isEmulator(), deviceData.state(), deviceData.manufacturer(), deviceData.modelClass());
        CrashpadController.writeSessionJsonFile(this.fileStore, sessionId, json, DEVICE_METADATA_FILE);
    }

    private static void writeSessionJsonFile(FileStore fileStore, String sessionId, String json, String fileName) {
        File sessionDirectory = fileStore.getNativeSessionDir(sessionId);
        File jsonFile = new File(sessionDirectory, fileName);
        CrashpadController.writeTextFile(jsonFile, json);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeTextFile(File file, String text) {
        BufferedWriter writer = null;
        try {
            writer = new BufferedWriter(new OutputStreamWriter((OutputStream)new FileOutputStream(file), UTF_8));
            writer.write(text);
        }
        catch (IOException iOException) {
        }
        finally {
            CommonUtils.closeOrLog((Closeable)writer, (String)("Failed to close " + file));
        }
    }

    @Nullable
    private static File getSingleFileWithExtension(File directory, String extension) {
        File[] files = directory.listFiles();
        if (files == null) {
            return null;
        }
        for (File file : files) {
            if (!file.getName().endsWith(extension)) continue;
            return file;
        }
        return null;
    }
}

