/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.crashlytics.buildtools.ndk.internal.elf;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.firebase.crashlytics.buildtools.Buildtools;
import com.google.firebase.crashlytics.buildtools.ndk.internal.dwarf.DebugLineEntry;
import com.google.firebase.crashlytics.buildtools.ndk.internal.dwarf.DwarfDataParser;
import com.google.firebase.crashlytics.buildtools.ndk.internal.dwarf.NamedRanges;
import com.google.firebase.crashlytics.buildtools.ndk.internal.elf.DebugElfSectionHeaders;
import com.google.firebase.crashlytics.buildtools.ndk.internal.elf.ElfFileHeader;
import com.google.firebase.crashlytics.buildtools.ndk.internal.elf.ElfFileIdent;
import com.google.firebase.crashlytics.buildtools.ndk.internal.elf.ElfSectionHeader;
import com.google.firebase.crashlytics.buildtools.ndk.internal.elf.ElfSectionHeaders;
import com.google.firebase.crashlytics.buildtools.ndk.internal.elf.ElfSymbol;
import com.google.firebase.crashlytics.buildtools.utils.io.ByteReader;
import com.google.firebase.crashlytics.buildtools.utils.io.RandomAccessFileInputStream;
import java.io.File;
import java.io.IOException;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

public class ElfDataParser {
    private static final String[] ARM_ARCH = new String[]{"Pre-v4", "4", "4T", "5T", "5TE", "5TEJ", "6", "6KZ", "6T2", "6K", "7", "6-M", "6S-M", "7E-M", "8"};
    private static final int SHT_ARM_ATTRIBUTES = 0x70000003;
    private static final String ARM_ATTR_PUBLIC_SECTION = "aeabi";
    private static final int ARM_ATTR_TAG_FILE_ATTRIBUTES = 1;
    private static final String ELF_NOTE_GNU = "GNU";
    private static final long NT_GNU_BUILD_ID = 3L;
    private final ByteReader _fileReader;
    private ByteOrder _byteOrder;
    private int _wordSize;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void parse(File input, ContentHandler handler, boolean featureUseDebug) throws IOException {
        try (ByteReader reader = null;){
            reader = new ByteReader(new RandomAccessFileInputStream(input));
            new ElfDataParser(reader).parse(handler, featureUseDebug);
        }
    }

    public ElfDataParser(ByteReader reader) {
        this._fileReader = reader;
    }

    public void parse(ContentHandler handler) throws IOException {
        this.parse(handler, false);
    }

    public void parse(ContentHandler handler, boolean featureUseDebug) throws IOException {
        ElfFileIdent elfFileIdent = this.initializeReader();
        if (handler == null) {
            handler = new NullContentHandler();
        }
        this.parseElf(elfFileIdent, handler, featureUseDebug);
    }

    private ElfFileIdent initializeReader() throws IOException {
        ElfFileIdent elfFileIdent = ElfDataParser.readElfFileIdent(this._fileReader);
        if (!elfFileIdent.isElf()) {
            throw new IllegalArgumentException("Input is not a valid ELF file.");
        }
        this._byteOrder = elfFileIdent.getDataEncoding() == 2 ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN;
        this._wordSize = elfFileIdent.getElfClass() == 2 ? 8 : 4;
        this._fileReader.setByteOrder(this._byteOrder);
        return elfFileIdent;
    }

    private void parseElf(ElfFileIdent ident, ContentHandler handler, boolean featureUseDebug) throws IOException {
        Optional<DebugElfSectionHeaders> debugHeaders;
        ElfFileHeader header = ElfDataParser.readElfFileHeader(this._fileReader, ident, this._wordSize);
        handler.processElfHeader(header);
        ElfSectionHeaders sectionHeaders = ElfDataParser.readElfSectionHeaders(this._fileReader, header, this._wordSize);
        handler.processSectionHeaders(sectionHeaders);
        Optional<byte[]> buildId = this.getBuildId(sectionHeaders);
        if (!buildId.isPresent()) {
            Buildtools.logD("Crashlytics could not find a build ID.");
            return;
        }
        handler.processBuildId((byte[])buildId.get());
        Optional<String> armVersion = this.readArmVersion(header, sectionHeaders);
        if (armVersion.isPresent()) {
            handler.processArmVersion((String)armVersion.get());
        }
        handler.startProcessingSymbols();
        boolean hasDebugInfo = sectionHeaders.hasDebugInfo();
        if (!featureUseDebug || !hasDebugInfo) {
            List<ElfSymbol> elfSymbols = this.readElfSymbols(sectionHeaders, ident.getElfClass());
            handler.processElfSymbols(elfSymbols);
        }
        if ((debugHeaders = DebugElfSectionHeaders.from(sectionHeaders)).isPresent()) {
            new DwarfDataParser(this._fileReader, this._byteOrder, (DebugElfSectionHeaders)debugHeaders.get(), featureUseDebug).parse(handler);
        }
        handler.endProcessingSymbols();
    }

    public Optional<byte[]> getBuildId(ElfSectionHeaders sectionHeaders) throws IOException {
        Optional<byte[]> identifier = this.getGnuBuildId(sectionHeaders);
        if (!identifier.isPresent()) {
            identifier = this.getTextSectionHash(sectionHeaders);
        }
        return identifier;
    }

    private Optional<byte[]> getGnuBuildId(ElfSectionHeaders sectionHeaders) throws IOException {
        return this.readGnuBuildIdNote(sectionHeaders).transform((Function)new Function<ElfNote, byte[]>(){

            public byte[] apply(ElfNote note) {
                return note.desc;
            }
        });
    }

    private Optional<byte[]> getTextSectionHash(ElfSectionHeaders sectionHeaders) throws IOException {
        int identifierSize = 16;
        return this.readTextPage(sectionHeaders, 16).transform((Function)new Function<byte[], byte[]>(){

            public byte[] apply(byte[] page) {
                byte[] identifier = new byte[16];
                for (int i2 = 0; i2 < page.length; ++i2) {
                    int n = i2 % identifier.length;
                    identifier[n] = (byte)(identifier[n] ^ page[i2]);
                }
                return identifier;
            }
        });
    }

    private Optional<byte[]> readTextPage(ElfSectionHeaders sectionHeaders, int identifierSize) throws IOException {
        Optional<ElfSectionHeader> textSectionHeader = sectionHeaders.findHeader(new Predicate<ElfSectionHeader>(){

            public boolean apply(ElfSectionHeader header) {
                return header.shNameString.equals(".text") && header.shType == 1;
            }
        });
        if (textSectionHeader.isPresent()) {
            ElfSectionHeader hdr = (ElfSectionHeader)textSectionHeader.get();
            this._fileReader.seek(hdr.shOffset);
            int readSize = ((int)Math.min(hdr.shSize, 4096L) + identifierSize - 1) / identifierSize * identifierSize;
            return Optional.of((Object)this._fileReader.readBytes(readSize));
        }
        return Optional.absent();
    }

    private Optional<ElfNote> readGnuBuildIdNote(ElfSectionHeaders sectionHeaders) throws IOException {
        Optional<ElfSectionHeader> buildIdSectionHeader = sectionHeaders.findHeader(new Predicate<ElfSectionHeader>(){

            public boolean apply(ElfSectionHeader header) {
                return header.shNameString.equals(".note.gnu.build-id") && header.shType == 7;
            }
        });
        if (buildIdSectionHeader.isPresent()) {
            ElfSectionHeader hdr = (ElfSectionHeader)buildIdSectionHeader.get();
            ElfNote note = this.readElfNote(hdr.shOffset);
            if (ELF_NOTE_GNU.equals(note.name) && 3L == note.type) {
                return Optional.of((Object)note);
            }
        }
        return Optional.absent();
    }

    private ElfNote readElfNote(long offset) throws IOException {
        this._fileReader.seek(offset);
        long namesz = this._fileReader.readLong(4);
        long descsz = this._fileReader.readLong(4);
        long type = this._fileReader.readLong(4);
        String name = this._fileReader.readNullTerminatedString(Charsets.UTF_8);
        this._fileReader.readBytes((int)(ElfDataParser.notePadding(namesz) - namesz));
        byte[] desc = this._fileReader.readBytes((int)descsz);
        return new ElfNote(name, type, desc);
    }

    private List<ElfSymbol> readElfSymbols(ElfSectionHeaders sectionHeaders, int alignment) throws IOException {
        LinkedList<ElfSymbol> symbols = new LinkedList<ElfSymbol>();
        for (ElfSectionHeader sh : sectionHeaders.getList()) {
            if (sh.shType != 2) continue;
            symbols.addAll(this.readElfSymbols(sh, sectionHeaders, alignment));
        }
        return symbols;
    }

    private List<ElfSymbol> readElfSymbols(ElfSectionHeader sh, ElfSectionHeaders sectionHeaders, int alignment) throws IOException {
        Optional<ElfSectionHeader> shStrings = sectionHeaders.getHeaderByIndex(sh.shLink);
        if (!shStrings.isPresent()) {
            return Collections.emptyList();
        }
        int numSymbols = (int)sh.shSize / (int)sh.shEntSize;
        return this.readSymbolTable(sh.shOffset, numSymbols, ((ElfSectionHeader)shStrings.get()).shOffset, alignment);
    }

    private List<ElfSymbol> readSymbolTable(long symTabOffset, int numSymbols, long stringsOffset, int alignment) throws IOException {
        this._fileReader.seek(symTabOffset);
        ArrayList<ElfSymbol> symbols = new ArrayList<ElfSymbol>(numSymbols);
        for (int i2 = 0; i2 < numSymbols; ++i2) {
            ElfSymbol sym = new ElfSymbol();
            switch (alignment) {
                case 2: {
                    sym.stName = this._fileReader.readInt(4);
                    sym.stInfo = this._fileReader.readByte();
                    sym.stOther = this._fileReader.readByte();
                    sym.stShndx = this._fileReader.readShort(2);
                    sym.stValue = this._fileReader.readLong(this._wordSize);
                    sym.stSize = this._fileReader.readLong(this._wordSize);
                    break;
                }
                default: {
                    sym.stName = this._fileReader.readInt(4);
                    sym.stValue = this._fileReader.readLong(this._wordSize);
                    sym.stSize = this._fileReader.readLong(this._wordSize);
                    sym.stInfo = this._fileReader.readByte();
                    sym.stOther = this._fileReader.readByte();
                    sym.stShndx = this._fileReader.readShort(2);
                }
            }
            symbols.add(sym);
        }
        this._fileReader.seek(stringsOffset);
        for (ElfSymbol symbol : symbols) {
            this._fileReader.seek(stringsOffset + (long)symbol.stName);
            symbol.stNameString = this._fileReader.readNullTerminatedString(Charsets.UTF_8);
        }
        return symbols;
    }

    private Optional<String> readArmVersion(ElfFileHeader header, ElfSectionHeaders sectionHeaders) throws IOException {
        Optional<ElfSectionHeader> armAttrSection;
        Optional<String> armVersion = Optional.absent();
        if (header.eMachine == 40 && (armAttrSection = sectionHeaders.findHeader(new Predicate<ElfSectionHeader>(){

            public boolean apply(ElfSectionHeader header) {
                return header.shType == 0x70000003;
            }
        })).isPresent()) {
            ElfSectionHeader hdr = (ElfSectionHeader)armAttrSection.get();
            armVersion = this.readArmVersion(hdr.shOffset, hdr.shSize);
        }
        return armVersion;
    }

    private Optional<String> readArmVersion(long dataOffset, long dataSize) throws IOException {
        this._fileReader.seek(dataOffset);
        byte version = this._fileReader.readByte();
        if (version != 65) {
            throw new IllegalArgumentException(String.format("Invalid data found at offset %d.", dataOffset));
        }
        long dataRemaining = dataSize - 1L;
        while (dataRemaining > 0L) {
            long sectionRemaining = this._fileReader.readInt(4);
            if (sectionRemaining > dataRemaining) {
                String errorString = "Section size %d is greater than remaining data length %d.";
                throw new IOException(String.format(errorString, sectionRemaining, dataRemaining));
            }
            dataRemaining -= sectionRemaining;
            sectionRemaining -= 4L;
            String sectionName = this._fileReader.readNullTerminatedString(Charsets.UTF_8);
            sectionRemaining -= (long)(sectionName.length() - 1);
            if (sectionName.equals(ARM_ATTR_PUBLIC_SECTION)) {
                return this.findArmVersionInSection(this._fileReader, sectionRemaining);
            }
            this._fileReader.seek(this._fileReader.getCurrentOffset() + sectionRemaining);
        }
        Buildtools.logD("Crashlytics did not find an ARM public attributes subsection.");
        return Optional.absent();
    }

    private Optional<String> findArmVersionInSection(ByteReader dataReader, long sectionRemaining) throws IOException {
        while (sectionRemaining > 0L) {
            byte tag = dataReader.readByte();
            long subSectionRemaining = dataReader.readInt(4);
            if (subSectionRemaining > sectionRemaining) {
                String errorString = "Subsection size %d is greater than parent section size %d.";
                throw new IOException(String.format(errorString, subSectionRemaining, sectionRemaining));
            }
            sectionRemaining -= subSectionRemaining;
            subSectionRemaining -= 5L;
            if (tag == 1) {
                return this.findArmVersionInSubSection(dataReader, subSectionRemaining);
            }
            dataReader.seek(dataReader.getCurrentOffset() + subSectionRemaining);
        }
        Buildtools.logD("Crashlytics did not find an ARM file attributes subsection.");
        return Optional.absent();
    }

    private Optional<String> findArmVersionInSubSection(ByteReader dataReader, long subSectionRemaining) throws IOException {
        long nextSubSection = dataReader.getCurrentOffset() + subSectionRemaining;
        block4: while (dataReader.getCurrentOffset() < nextSubSection) {
            int attrTag = dataReader.readULEB128();
            switch (attrTag) {
                case 4: 
                case 5: 
                case 32: 
                case 65: 
                case 67: {
                    dataReader.readNullTerminatedString(Charsets.UTF_8);
                    continue block4;
                }
                case 6: {
                    int attrVal = dataReader.readULEB128();
                    return Optional.of((Object)ARM_ARCH[attrVal]);
                }
            }
            dataReader.readULEB128();
        }
        Buildtools.logD("Crashlytics did not find an ARM architecture field.");
        return Optional.absent();
    }

    private static ElfFileIdent readElfFileIdent(ByteReader reader) throws IOException {
        reader.seek(0L);
        return new ElfFileIdent(reader.readBytes(16));
    }

    private static ElfFileHeader readElfFileHeader(ByteReader reader, ElfFileIdent identity, int wordSize) throws IOException {
        reader.seek(16L);
        ElfFileHeader header = new ElfFileHeader(identity);
        header.eType = reader.readInt(2);
        header.eMachine = reader.readInt(2);
        header.eVersion = reader.readLong(4);
        header.eEntry = reader.readLong(wordSize);
        header.ePhoff = reader.readLong(wordSize);
        header.eShoff = reader.readLong(wordSize);
        header.eFlags = reader.readLong(4);
        header.eEhsize = reader.readInt(2);
        header.ePhentsize = reader.readInt(2);
        header.ePhnum = reader.readInt(2);
        header.eShentsize = reader.readInt(2);
        header.eShnum = reader.readInt(2);
        header.eShstrndx = reader.readInt(2);
        return header;
    }

    private static ElfSectionHeaders readElfSectionHeaders(ByteReader reader, ElfFileHeader header, int wordSize) throws IOException {
        reader.seek(header.eShoff);
        ArrayList sectionHeaders = Lists.newArrayListWithCapacity((int)header.eShnum);
        for (int i2 = 0; i2 < header.eShnum; ++i2) {
            ElfSectionHeader sh = new ElfSectionHeader();
            sh.shName = reader.readInt(4);
            sh.shType = reader.readInt(4);
            sh.shFlags = reader.readLong(wordSize);
            sh.shAddr = reader.readLong(wordSize);
            sh.shOffset = reader.readLong(wordSize);
            sh.shSize = reader.readLong(wordSize);
            sh.shLink = reader.readInt(4);
            sh.shInfo = reader.readInt(4);
            sh.shAddrAlign = reader.readLong(wordSize);
            sh.shEntSize = reader.readLong(wordSize);
            sectionHeaders.add(sh);
        }
        ElfSectionHeader names = (ElfSectionHeader)sectionHeaders.get(header.eShstrndx);
        reader.seek(names.shOffset);
        for (ElfSectionHeader sectionHeader : sectionHeaders) {
            reader.seek(names.shOffset + (long)sectionHeader.shName);
            sectionHeader.shNameString = reader.readNullTerminatedString(Charsets.UTF_8);
        }
        return new ElfSectionHeaders(sectionHeaders);
    }

    private static long notePadding(long size) {
        return size + 3L & 0xFFFFFFFFFFFFFFFCL;
    }

    public static interface ContentHandler {
        public void processElfHeader(ElfFileHeader var1);

        public void processSectionHeaders(ElfSectionHeaders var1);

        public void processBuildId(byte[] var1);

        public void processArmVersion(String var1);

        public void startProcessingSymbols();

        public void processElfSymbols(List<ElfSymbol> var1);

        public void processDebugInfoCompilationUnit(NamedRanges var1, List<DebugLineEntry> var2);

        public void endProcessingSymbols();
    }

    private static final class NullContentHandler
    implements ContentHandler {
        private NullContentHandler() {
        }

        @Override
        public void processElfHeader(ElfFileHeader fileHeader) {
        }

        @Override
        public void processSectionHeaders(ElfSectionHeaders sectionHeaders) {
        }

        @Override
        public void processBuildId(byte[] buildId) {
        }

        @Override
        public void startProcessingSymbols() {
        }

        @Override
        public void processElfSymbols(List<ElfSymbol> elfSymbols) {
        }

        @Override
        public void processArmVersion(String armVersion) {
        }

        @Override
        public void processDebugInfoCompilationUnit(NamedRanges namedRanges, List<DebugLineEntry> debugLineEntries) {
        }

        @Override
        public void endProcessingSymbols() {
        }
    }

    private static class ElfNote {
        public final String name;
        public final long type;
        public final byte[] desc;

        public ElfNote(String name, long type, byte[] desc) {
            this.name = name;
            this.type = type;
            this.desc = desc;
        }
    }
}

