/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import com.google.common.io.BaseEncoding;
import com.google.firebase.FirebaseOptions;
import com.google.firebase.auth.GoogleOAuthAccessToken;
import com.google.firebase.internal.AuthStateListener;
import com.google.firebase.internal.FirebaseAppStore;
import com.google.firebase.internal.FirebaseExecutors;
import com.google.firebase.internal.FirebaseService;
import com.google.firebase.internal.GetTokenResult;
import com.google.firebase.internal.Log;
import com.google.firebase.internal.NonNull;
import com.google.firebase.internal.Nullable;
import com.google.firebase.tasks.Continuation;
import com.google.firebase.tasks.Task;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

public class FirebaseApp {
    private static final Map<String, FirebaseApp> instances = new HashMap<String, FirebaseApp>();
    public static final String DEFAULT_APP_NAME = "[DEFAULT]";
    private static final long TOKEN_REFRESH_INTERVAL_MILLIS = TimeUnit.MINUTES.toMillis(55L);
    static final TokenRefresher.Factory DEFAULT_TOKEN_REFRESHER_FACTORY = new TokenRefresher.Factory();
    static final Clock DEFAULT_CLOCK = new Clock();
    private static final Object appsLock = new Object();
    private final String name;
    private final FirebaseOptions options;
    private final TokenRefresher tokenRefresher;
    private final Clock clock;
    private final AtomicBoolean deleted = new AtomicBoolean();
    private final List<AuthStateListener> authStateListeners = new ArrayList<AuthStateListener>();
    private final AtomicReference<GetTokenResult> currentToken = new AtomicReference();
    private final Map<String, FirebaseService> services = new HashMap<String, FirebaseService>();
    private Task<GoogleOAuthAccessToken> previousTokenTask;
    private final Object lock = new Object();

    private FirebaseApp(String name, FirebaseOptions options, TokenRefresher.Factory factory, Clock clock) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)name) ? 1 : 0) != 0);
        this.name = name;
        this.options = (FirebaseOptions)Preconditions.checkNotNull((Object)options);
        this.tokenRefresher = ((TokenRefresher.Factory)Preconditions.checkNotNull((Object)factory)).create(this);
        this.clock = (Clock)Preconditions.checkNotNull((Object)clock);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<FirebaseApp> getApps() {
        Object object = appsLock;
        synchronized (object) {
            return ImmutableList.copyOf(instances.values());
        }
    }

    @Nullable
    public static FirebaseApp getInstance() {
        return FirebaseApp.getInstance(DEFAULT_APP_NAME);
    }

    public static FirebaseApp getInstance(@NonNull String name) {
        Object object = appsLock;
        synchronized (object) {
            FirebaseApp firebaseApp = instances.get(FirebaseApp.normalize(name));
            if (firebaseApp != null) {
                return firebaseApp;
            }
            List<String> availableAppNames = FirebaseApp.getAllAppNames();
            String availableAppNamesMessage = availableAppNames.isEmpty() ? "" : "Available app names: " + Joiner.on((String)", ").join(availableAppNames);
            String errorMessage = String.format("FirebaseApp with name %s doesn't exist. %s", name, availableAppNamesMessage);
            throw new IllegalStateException(errorMessage);
        }
    }

    public static FirebaseApp initializeApp(FirebaseOptions options) {
        return FirebaseApp.initializeApp(options, DEFAULT_APP_NAME);
    }

    public static FirebaseApp initializeApp(FirebaseOptions options, String name) {
        return FirebaseApp.initializeApp(options, name, DEFAULT_TOKEN_REFRESHER_FACTORY, DEFAULT_CLOCK);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static FirebaseApp initializeApp(FirebaseOptions options, String name, TokenRefresher.Factory tokenRefresherFactory, Clock clock) {
        FirebaseApp firebaseApp;
        FirebaseAppStore appStore = FirebaseAppStore.initialize();
        String normalizedName = FirebaseApp.normalize(name);
        Object object = appsLock;
        synchronized (object) {
            Preconditions.checkState((!instances.containsKey(normalizedName) ? 1 : 0) != 0, (Object)("FirebaseApp name " + normalizedName + " already exists!"));
            firebaseApp = new FirebaseApp(normalizedName, options, tokenRefresherFactory, clock);
            instances.put(normalizedName, firebaseApp);
        }
        appStore.persistApp(firebaseApp);
        return firebaseApp;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @VisibleForTesting
    static void clearInstancesForTest() {
        Object object = appsLock;
        synchronized (object) {
            for (FirebaseApp app : ImmutableList.copyOf(instances.values())) {
                app.delete();
            }
            instances.clear();
        }
    }

    static String getPersistenceKey(String name, FirebaseOptions options) {
        return BaseEncoding.base64Url().omitPadding().encode(name.getBytes(StandardCharsets.UTF_8));
    }

    String getPersistenceKey() {
        return FirebaseApp.getPersistenceKey(this.getName(), this.getOptions());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static List<String> getAllAppNames() {
        HashSet<String> allAppNames = new HashSet<String>();
        Object object = appsLock;
        synchronized (object) {
            for (FirebaseApp app : instances.values()) {
                allAppNames.add(app.getName());
            }
            FirebaseAppStore appStore = FirebaseAppStore.getInstance();
            if (appStore != null) {
                allAppNames.addAll(appStore.getAllPersistedAppNames());
            }
        }
        ArrayList<String> sortedNameList = new ArrayList<String>(allAppNames);
        Collections.sort(sortedNameList);
        return sortedNameList;
    }

    private static String normalize(@NonNull String name) {
        return ((String)Preconditions.checkNotNull((Object)name)).trim();
    }

    @NonNull
    public String getName() {
        this.checkNotDeleted();
        return this.name;
    }

    @NonNull
    public FirebaseOptions getOptions() {
        this.checkNotDeleted();
        return this.options;
    }

    public boolean equals(Object o) {
        if (!(o instanceof FirebaseApp)) {
            return false;
        }
        return this.name.equals(((FirebaseApp)o).getName());
    }

    public int hashCode() {
        return this.name.hashCode();
    }

    public String toString() {
        return MoreObjects.toStringHelper((Object)this).add("name", (Object)this.name).add("options", (Object)this.options).toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void delete() {
        Object object = this.lock;
        synchronized (object) {
            boolean valueChanged = this.deleted.compareAndSet(false, true);
            if (!valueChanged) {
                return;
            }
            for (FirebaseService service : this.services.values()) {
                service.destroy();
            }
            this.services.clear();
            this.authStateListeners.clear();
            this.tokenRefresher.cleanup();
        }
        object = appsLock;
        synchronized (object) {
            instances.remove(this.name);
        }
        FirebaseAppStore appStore = FirebaseAppStore.getInstance();
        if (appStore != null) {
            appStore.removeApp(this.name);
        }
    }

    private void checkNotDeleted() {
        Preconditions.checkState((!this.deleted.get() ? 1 : 0) != 0, (String)"FirebaseApp was deleted %s", (Object)this);
    }

    private boolean refreshRequired(@NonNull Task<GoogleOAuthAccessToken> previousTask, boolean forceRefresh) {
        return previousTask.isComplete() && (forceRefresh || !previousTask.isSuccessful() || previousTask.getResult().getExpiryTime() <= this.clock.now());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Task<GetTokenResult> getToken(boolean forceRefresh) {
        Object object = this.lock;
        synchronized (object) {
            this.checkNotDeleted();
            if (this.previousTokenTask == null || this.refreshRequired(this.previousTokenTask, forceRefresh)) {
                this.previousTokenTask = this.options.getCredential().getAccessToken();
            }
            return this.previousTokenTask.continueWith(new Continuation<GoogleOAuthAccessToken, GetTokenResult>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public GetTokenResult then(@NonNull Task<GoogleOAuthAccessToken> task) throws Exception {
                    GoogleOAuthAccessToken googleOAuthToken = task.getResult();
                    GetTokenResult newToken = new GetTokenResult(googleOAuthToken.getAccessToken());
                    GetTokenResult oldToken = (GetTokenResult)FirebaseApp.this.currentToken.get();
                    List listenersCopy = null;
                    if (!newToken.equals(oldToken)) {
                        Iterator iterator = FirebaseApp.this.lock;
                        synchronized (iterator) {
                            if (FirebaseApp.this.deleted.get()) {
                                return newToken;
                            }
                            if (FirebaseApp.this.currentToken.compareAndSet(oldToken, newToken)) {
                                listenersCopy = ImmutableList.copyOf((Collection)FirebaseApp.this.authStateListeners);
                                long refreshDelay = googleOAuthToken.getExpiryTime() - FirebaseApp.this.clock.now() - TimeUnit.MINUTES.toMillis(5L);
                                if (refreshDelay > 0L) {
                                    FirebaseApp.this.tokenRefresher.scheduleRefresh(refreshDelay);
                                } else {
                                    Log.w("FirebaseApp", "Token expiry (" + googleOAuthToken.getExpiryTime() + ") is less than 5 minutes in the future. Not scheduling a proactive refresh.");
                                }
                            }
                        }
                    }
                    if (listenersCopy != null) {
                        for (AuthStateListener listener : listenersCopy) {
                            listener.onAuthStateChanged(newToken);
                        }
                    }
                    return newToken;
                }
            });
        }
    }

    boolean isDefaultApp() {
        return DEFAULT_APP_NAME.equals(this.getName());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addAuthStateListener(@NonNull AuthStateListener listener) {
        GetTokenResult currentToken;
        Object object = this.lock;
        synchronized (object) {
            this.checkNotDeleted();
            this.authStateListeners.add((AuthStateListener)Preconditions.checkNotNull((Object)listener));
            currentToken = this.currentToken.get();
        }
        if (currentToken != null) {
            listener.onAuthStateChanged(currentToken);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeAuthStateListener(@NonNull AuthStateListener listener) {
        Object object = this.lock;
        synchronized (object) {
            this.checkNotDeleted();
            this.authStateListeners.remove(Preconditions.checkNotNull((Object)listener));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void addService(FirebaseService service) {
        Object object = this.lock;
        synchronized (object) {
            this.checkNotDeleted();
            Preconditions.checkArgument((!this.services.containsKey(((FirebaseService)Preconditions.checkNotNull((Object)service)).getId()) ? 1 : 0) != 0);
            this.services.put(service.getId(), service);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    FirebaseService getService(String id) {
        Object object = this.lock;
        synchronized (object) {
            Preconditions.checkArgument((!Strings.isNullOrEmpty((String)id) ? 1 : 0) != 0);
            return this.services.get(id);
        }
    }

    static class Clock {
        Clock() {
        }

        long now() {
            return System.currentTimeMillis();
        }
    }

    static class TokenRefresher {
        private final FirebaseApp firebaseApp;
        private ScheduledFuture<Task<GetTokenResult>> future;

        TokenRefresher(FirebaseApp app) {
            this.firebaseApp = (FirebaseApp)Preconditions.checkNotNull((Object)app);
        }

        final void scheduleRefresh(long delayMillis) {
            this.cancelPrevious();
            this.scheduleNext(new Callable<Task<GetTokenResult>>(){

                @Override
                public Task<GetTokenResult> call() throws Exception {
                    return TokenRefresher.this.firebaseApp.getToken(true);
                }
            }, delayMillis);
        }

        protected void cancelPrevious() {
            if (this.future != null) {
                this.future.cancel(true);
            }
        }

        protected void scheduleNext(Callable<Task<GetTokenResult>> task, long delayMillis) {
            try {
                this.future = FirebaseExecutors.DEFAULT_SCHEDULED_EXECUTOR.schedule(task, delayMillis, TimeUnit.MILLISECONDS);
            }
            catch (UnsupportedOperationException unsupportedOperationException) {
                // empty catch block
            }
        }

        protected void cleanup() {
            if (this.future != null) {
                this.future.cancel(true);
            }
        }

        static class Factory {
            Factory() {
            }

            TokenRefresher create(FirebaseApp app) {
                return new TokenRefresher(app);
            }
        }
    }
}

