/*
 * Copyright 2010-2022 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.session

import ksp.org.jetbrains.kotlin.diagnostics.rendering.BaseDiagnosticRendererFactory
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.FirSessionComponent
import ksp.org.jetbrains.kotlin.fir.SessionConfiguration
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.LanguageVersionSettingsCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.declaration.DeclarationCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.ExpressionCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.type.TypeCheckers
import ksp.org.jetbrains.kotlin.fir.analysis.checkersComponent
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.diagnosticRendererFactory
import ksp.org.jetbrains.kotlin.fir.analysis.extensions.additionalCheckers
import ksp.org.jetbrains.kotlin.fir.diagnostics.FirDiagnosticsContainer
import ksp.org.jetbrains.kotlin.fir.extensions.*
import kotlin.reflect.KClass

class FirSessionConfigurator(private val session: FirSession) {
    private val registeredExtensions: MutableList<BunchOfRegisteredExtensions> = mutableListOf(BunchOfRegisteredExtensions.empty())

    fun registerExtensions(extensions: BunchOfRegisteredExtensions) {
        registeredExtensions += extensions
    }

    @OptIn(SessionConfiguration::class)
    fun useCheckers(checkers: ExpressionCheckers) {
        session.checkersComponent.register(checkers)
    }

    @OptIn(SessionConfiguration::class)
    fun useCheckers(checkers: DeclarationCheckers) {
        session.checkersComponent.register(checkers)
    }

    @OptIn(SessionConfiguration::class)
    fun useCheckers(checkers: TypeCheckers) {
        session.checkersComponent.register(checkers)
    }

    @OptIn(SessionConfiguration::class)
    fun useCheckers(checkers: LanguageVersionSettingsCheckers) {
        session.checkersComponent.register(checkers)
    }

    @OptIn(SessionConfiguration::class)
    fun registerComponent(componentKey: KClass<out FirSessionComponent>, componentValue: FirSessionComponent) {
        session.register(componentKey, componentValue)
    }

    @OptIn(SessionConfiguration::class)
    fun registerDiagnosticContainers(vararg diagnosticContainers: FirDiagnosticsContainer) {
        session.diagnosticRendererFactory.registerFactories(diagnosticContainers.map { it.getRendererFactory() })
    }

    @OptIn(PluginServicesInitialization::class)
    @SessionConfiguration
    fun configure() {
        var extensions = registeredExtensions.reduce(BunchOfRegisteredExtensions::plus)
        if (session.kind == FirSession.Kind.Library) {
            val filteredExtensions = extensions.extensions.filterKeys { it in FirExtensionRegistrar.ALLOWED_EXTENSIONS_FOR_LIBRARY_SESSION }
            extensions = BunchOfRegisteredExtensions(filteredExtensions, diagnosticsContainers = emptyList())
        }
        session.extensionService.registerExtensions(extensions)
        if (session.kind == FirSession.Kind.Source) {
            session.extensionService.additionalCheckers.forEach(session.checkersComponent::register)
        }
    }
}
