/*
 * Copyright 2010-2021 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.syntax

import ksp.org.jetbrains.kotlin.KtFakeSourceElementKind
import ksp.org.jetbrains.kotlin.KtLightSourceElement
import ksp.org.jetbrains.kotlin.KtPsiSourceElement
import ksp.org.jetbrains.kotlin.KtSourceElement
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.nameIdentifier
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.diagnostics.typeParametersList
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.*
import ksp.org.jetbrains.kotlin.fir.declarations.FirSimpleFunction
import ksp.org.jetbrains.kotlin.psi.KtFunction
import ksp.org.jetbrains.kotlin.psi.psiUtil.startOffset

object FirFunctionTypeParametersSyntaxChecker : FirDeclarationSyntaxChecker<FirSimpleFunction, KtFunction>() {
    override fun isApplicable(element: FirSimpleFunction, source: KtSourceElement): Boolean =
        source.kind !is KtFakeSourceElementKind

    override fun checkPsi(
        element: FirSimpleFunction,
        source: KtPsiSourceElement,
        psi: KtFunction,
        context: CheckerContext,
        reporter: DiagnosticReporter
    ) {
        val typeParamsNode = psi.typeParameterList
        val nameNode = psi.nameIdentifier

        if (typeParamsNode != null && nameNode != null && typeParamsNode.startOffset > nameNode.startOffset) {
            reporter.reportOn(
                source,
                FirErrors.DEPRECATED_TYPE_PARAMETER_SYNTAX,
                context
            )
        }
    }

    override fun checkLightTree(
        element: FirSimpleFunction,
        source: KtLightSourceElement,
        context: CheckerContext,
        reporter: DiagnosticReporter
    ) {
        val typeParamsNode = source.treeStructure.typeParametersList(source.lighterASTNode)
        val nameNode = source.treeStructure.nameIdentifier(source.lighterASTNode)
        if (typeParamsNode != null && nameNode != null && typeParamsNode.startOffset > nameNode.startOffset) {
            reporter.reportOn(
                source,
                FirErrors.DEPRECATED_TYPE_PARAMETER_SYNTAX,
                context
            )
        }
    }
}
