/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.sessions

import ksp.com.intellij.openapi.project.Project
import ksp.com.intellij.psi.search.GlobalSearchScope
import ksp.org.jetbrains.kotlin.analysis.api.platform.packages.createPackagePartProvider
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaDanglingFileModule
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaModule
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaSourceModule
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.factories.LLLibrarySymbolProviderFactory
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.projectStructure.moduleData
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.symbolProviders.LLModuleWithDependenciesSymbolProvider
import ksp.org.jetbrains.kotlin.fir.SessionConfiguration
import ksp.org.jetbrains.kotlin.fir.backend.jvm.FirJvmTypeMapper
import ksp.org.jetbrains.kotlin.fir.deserialization.SingleModuleDataProvider
import ksp.org.jetbrains.kotlin.fir.java.deserialization.OptionalAnnotationClassesProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirSymbolProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.firProvider
import ksp.org.jetbrains.kotlin.fir.scopes.kotlinScopeProvider
import ksp.org.jetbrains.kotlin.platform.has
import ksp.org.jetbrains.kotlin.platform.jvm.JvmPlatform
import ksp.org.jetbrains.kotlin.utils.addIfNotNull

@OptIn(SessionConfiguration::class)
internal class LLFirCommonSessionFactory(project: Project) : LLFirAbstractSessionFactory(project) {
    override fun createSourcesSession(module: KaSourceModule): LLFirSourcesSession {
        return doCreateSourcesSession(module) { context ->
            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOfNotNull(
                        context.firProvider.symbolProvider,
                        context.switchableExtensionDeclarationsSymbolProvider,
                        context.syntheticFunctionInterfaceProvider,
                    ),
                    context.dependencyProvider,
                )
            )

            registerPlatformSpecificComponentsIfAny(module)
        }
    }

    override fun createResolvableLibrarySession(module: KaModule): LLFirLibraryOrLibrarySourceResolvableModuleSession {
        return doCreateResolvableLibrarySession(module) { context ->
            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOf(
                        context.firProvider.symbolProvider,
                    ),
                    context.dependencyProvider,
                )
            )

            registerPlatformSpecificComponentsIfAny(module)
        }
    }

    override fun createBinaryLibrarySession(module: KaModule): LLFirLibrarySession {
        return doCreateBinaryLibrarySession(module) {
            registerPlatformSpecificComponentsIfAny(module)
        }
    }

    override fun createDanglingFileSession(module: KaDanglingFileModule, contextSession: LLFirSession): LLFirSession {
        return doCreateDanglingFileSession(module, contextSession) { context ->
            register(
                FirSymbolProvider::class,
                LLModuleWithDependenciesSymbolProvider(
                    this,
                    providers = listOf(
                        firProvider.symbolProvider,
                    ),
                    context.dependencyProvider,
                )
            )

            registerPlatformSpecificComponentsIfAny(module)
        }
    }

    override fun createProjectLibraryProvidersForScope(
        session: LLFirSession,
        scope: GlobalSearchScope,
    ): List<FirSymbolProvider> {
        val moduleData = session.moduleData
        val moduleDataProvider = SingleModuleDataProvider(moduleData)
        val packagePartProvider = project.createPackagePartProvider(scope)
        return buildList {
            addAll(
                LLLibrarySymbolProviderFactory.fromSettings(project).createCommonLibrarySymbolProvider(
                    session,
                    packagePartProvider,
                    scope,
                )
            )

            addIfNotNull(
                OptionalAnnotationClassesProvider.createIfNeeded(
                    session,
                    moduleDataProvider,
                    session.kotlinScopeProvider,
                    packagePartProvider
                )
            )
        }
    }

    private fun LLFirSession.registerPlatformSpecificComponentsIfAny(module: KaModule) {
        if (module.targetPlatform.has<JvmPlatform>())
            register(FirJvmTypeMapper::class, FirJvmTypeMapper(this))
    }
}
