/*
 * Decompiled with CFR 0.152.
 */
package com.google.crypto.tink.internal;

import com.google.crypto.tink.KeyManager;
import com.google.crypto.tink.config.internal.TinkFipsUtil;
import com.google.crypto.tink.internal.KeyManagerImpl;
import com.google.crypto.tink.internal.KeyTypeManager;
import com.google.crypto.tink.internal.PrivateKeyManagerImpl;
import com.google.crypto.tink.internal.PrivateKeyTypeManager;
import com.google.crypto.tink.shaded.protobuf.MessageLite;
import java.security.GeneralSecurityException;
import java.util.Collections;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.logging.Logger;

public final class KeyManagerRegistry {
    private static final Logger logger = Logger.getLogger(KeyManagerRegistry.class.getName());
    private ConcurrentMap<String, KeyManagerContainer> keyManagerMap;
    private ConcurrentMap<String, Boolean> newKeyAllowedMap;
    private static final KeyManagerRegistry GLOBAL_INSTANCE = new KeyManagerRegistry();

    public static KeyManagerRegistry globalInstance() {
        return GLOBAL_INSTANCE;
    }

    public static void resetGlobalInstanceTestOnly() {
        KeyManagerRegistry.GLOBAL_INSTANCE.keyManagerMap = new ConcurrentHashMap<String, KeyManagerContainer>();
        KeyManagerRegistry.GLOBAL_INSTANCE.newKeyAllowedMap = new ConcurrentHashMap<String, Boolean>();
    }

    public KeyManagerRegistry(KeyManagerRegistry original) {
        this.keyManagerMap = new ConcurrentHashMap<String, KeyManagerContainer>(original.keyManagerMap);
        this.newKeyAllowedMap = new ConcurrentHashMap<String, Boolean>(original.newKeyAllowedMap);
    }

    public KeyManagerRegistry() {
        this.keyManagerMap = new ConcurrentHashMap<String, KeyManagerContainer>();
        this.newKeyAllowedMap = new ConcurrentHashMap<String, Boolean>();
    }

    private static <P> KeyManagerContainer createContainerFor(KeyManager<P> keyManager) {
        final KeyManager<P> localKeyManager = keyManager;
        return new KeyManagerContainer(){

            public <Q> KeyManager<Q> getKeyManager(Class<Q> primitiveClass) {
                if (!localKeyManager.getPrimitiveClass().equals(primitiveClass)) {
                    throw new InternalError("This should never be called, as we always first check supportedPrimitives.");
                }
                KeyManager result = localKeyManager;
                return result;
            }

            @Override
            public KeyManager<?> getUntypedKeyManager() {
                return localKeyManager;
            }

            @Override
            public Class<?> getImplementingClass() {
                return localKeyManager.getClass();
            }

            @Override
            public Set<Class<?>> supportedPrimitives() {
                return Collections.singleton(localKeyManager.getPrimitiveClass());
            }
        };
    }

    private static <KeyProtoT extends MessageLite> KeyManagerContainer createContainerFor(KeyTypeManager<KeyProtoT> keyManager) {
        final KeyTypeManager<KeyProtoT> localKeyManager = keyManager;
        return new KeyManagerContainer(){

            public <Q> KeyManager<Q> getKeyManager(Class<Q> primitiveClass) throws GeneralSecurityException {
                try {
                    return new KeyManagerImpl(localKeyManager, primitiveClass);
                }
                catch (IllegalArgumentException e) {
                    throw new GeneralSecurityException("Primitive type not supported", e);
                }
            }

            @Override
            public KeyManager<?> getUntypedKeyManager() {
                return new KeyManagerImpl(localKeyManager, localKeyManager.firstSupportedPrimitiveClass());
            }

            @Override
            public Class<?> getImplementingClass() {
                return localKeyManager.getClass();
            }

            @Override
            public Set<Class<?>> supportedPrimitives() {
                return localKeyManager.supportedPrimitives();
            }
        };
    }

    private static <KeyProtoT extends MessageLite, PublicKeyProtoT extends MessageLite> KeyManagerContainer createPrivateKeyContainerFor(PrivateKeyTypeManager<KeyProtoT, PublicKeyProtoT> privateKeyTypeManager, KeyTypeManager<PublicKeyProtoT> publicKeyTypeManager) {
        final PrivateKeyTypeManager<KeyProtoT, PublicKeyProtoT> localPrivateKeyManager = privateKeyTypeManager;
        final KeyTypeManager<PublicKeyProtoT> localPublicKeyManager = publicKeyTypeManager;
        return new KeyManagerContainer(){

            public <Q> KeyManager<Q> getKeyManager(Class<Q> primitiveClass) throws GeneralSecurityException {
                try {
                    return new PrivateKeyManagerImpl(localPrivateKeyManager, localPublicKeyManager, primitiveClass);
                }
                catch (IllegalArgumentException e) {
                    throw new GeneralSecurityException("Primitive type not supported", e);
                }
            }

            @Override
            public KeyManager<?> getUntypedKeyManager() {
                return new PrivateKeyManagerImpl(localPrivateKeyManager, localPublicKeyManager, localPrivateKeyManager.firstSupportedPrimitiveClass());
            }

            @Override
            public Class<?> getImplementingClass() {
                return localPrivateKeyManager.getClass();
            }

            @Override
            public Set<Class<?>> supportedPrimitives() {
                return localPrivateKeyManager.supportedPrimitives();
            }
        };
    }

    private synchronized KeyManagerContainer getKeyManagerContainerOrThrow(String typeUrl) throws GeneralSecurityException {
        if (!this.keyManagerMap.containsKey(typeUrl)) {
            throw new GeneralSecurityException("No key manager found for key type " + typeUrl);
        }
        return (KeyManagerContainer)this.keyManagerMap.get(typeUrl);
    }

    private synchronized void registerKeyManagerContainer(KeyManagerContainer containerToInsert, boolean forceOverwrite, boolean newKeyAllowed) throws GeneralSecurityException {
        String typeUrl = containerToInsert.getUntypedKeyManager().getKeyType();
        if (newKeyAllowed && this.newKeyAllowedMap.containsKey(typeUrl) && !((Boolean)this.newKeyAllowedMap.get(typeUrl)).booleanValue()) {
            throw new GeneralSecurityException("New keys are already disallowed for key type " + typeUrl);
        }
        KeyManagerContainer container = (KeyManagerContainer)this.keyManagerMap.get(typeUrl);
        if (container != null && !container.getImplementingClass().equals(containerToInsert.getImplementingClass())) {
            logger.warning("Attempted overwrite of a registered key manager for key type " + typeUrl);
            throw new GeneralSecurityException(String.format("typeUrl (%s) is already registered with %s, cannot be re-registered with %s", typeUrl, container.getImplementingClass().getName(), containerToInsert.getImplementingClass().getName()));
        }
        if (!forceOverwrite) {
            this.keyManagerMap.putIfAbsent(typeUrl, containerToInsert);
        } else {
            this.keyManagerMap.put(typeUrl, containerToInsert);
        }
        this.newKeyAllowedMap.put(typeUrl, newKeyAllowed);
    }

    public synchronized <P> void registerKeyManager(KeyManager<P> manager, boolean newKeyAllowed) throws GeneralSecurityException {
        this.registerKeyManagerWithFipsCompatibility(manager, TinkFipsUtil.AlgorithmFipsCompatibility.ALGORITHM_NOT_FIPS, newKeyAllowed);
    }

    public synchronized <P> void registerKeyManagerWithFipsCompatibility(KeyManager<P> manager, TinkFipsUtil.AlgorithmFipsCompatibility compatibility, boolean newKeyAllowed) throws GeneralSecurityException {
        if (!compatibility.isCompatible()) {
            throw new GeneralSecurityException("Cannot register key manager: FIPS compatibility insufficient");
        }
        this.registerKeyManagerContainer(KeyManagerRegistry.createContainerFor(manager), false, newKeyAllowed);
    }

    public synchronized <KeyProtoT extends MessageLite> void registerKeyManager(KeyTypeManager<KeyProtoT> manager, boolean newKeyAllowed) throws GeneralSecurityException {
        if (!manager.fipsStatus().isCompatible()) {
            throw new GeneralSecurityException("failed to register key manager " + manager.getClass() + " as it is not FIPS compatible.");
        }
        this.registerKeyManagerContainer(KeyManagerRegistry.createContainerFor(manager), false, newKeyAllowed);
    }

    public synchronized <KeyProtoT extends MessageLite, PublicKeyProtoT extends MessageLite> void registerAsymmetricKeyManagers(PrivateKeyTypeManager<KeyProtoT, PublicKeyProtoT> privateKeyTypeManager, KeyTypeManager<PublicKeyProtoT> publicKeyTypeManager, boolean newKeyAllowed) throws GeneralSecurityException {
        TinkFipsUtil.AlgorithmFipsCompatibility fipsStatusPrivateKey = privateKeyTypeManager.fipsStatus();
        TinkFipsUtil.AlgorithmFipsCompatibility fipsStatusPublicKey = publicKeyTypeManager.fipsStatus();
        if (!fipsStatusPrivateKey.isCompatible()) {
            throw new GeneralSecurityException("failed to register key manager " + privateKeyTypeManager.getClass() + " as it is not FIPS compatible.");
        }
        if (!fipsStatusPublicKey.isCompatible()) {
            throw new GeneralSecurityException("failed to register key manager " + publicKeyTypeManager.getClass() + " as it is not FIPS compatible.");
        }
        this.registerKeyManagerContainer(KeyManagerRegistry.createPrivateKeyContainerFor(privateKeyTypeManager, publicKeyTypeManager), true, newKeyAllowed);
        this.registerKeyManagerContainer(KeyManagerRegistry.createContainerFor(publicKeyTypeManager), false, false);
    }

    public boolean typeUrlExists(String typeUrl) {
        return this.keyManagerMap.containsKey(typeUrl);
    }

    private static String toCommaSeparatedString(Set<Class<?>> setOfClasses) {
        StringBuilder b = new StringBuilder();
        boolean first = true;
        for (Class<?> clazz : setOfClasses) {
            if (!first) {
                b.append(", ");
            }
            b.append(clazz.getCanonicalName());
            first = false;
        }
        return b.toString();
    }

    public <P> KeyManager<P> getKeyManager(String typeUrl, Class<P> primitiveClass) throws GeneralSecurityException {
        KeyManagerContainer container = this.getKeyManagerContainerOrThrow(typeUrl);
        if (container.supportedPrimitives().contains(primitiveClass)) {
            return container.getKeyManager(primitiveClass);
        }
        throw new GeneralSecurityException("Primitive type " + primitiveClass.getName() + " not supported by key manager of type " + container.getImplementingClass() + ", supported primitives: " + KeyManagerRegistry.toCommaSeparatedString(container.supportedPrimitives()));
    }

    public KeyManager<?> getUntypedKeyManager(String typeUrl) throws GeneralSecurityException {
        KeyManagerContainer container = this.getKeyManagerContainerOrThrow(typeUrl);
        return container.getUntypedKeyManager();
    }

    public boolean isNewKeyAllowed(String typeUrl) {
        return (Boolean)this.newKeyAllowedMap.get(typeUrl);
    }

    public boolean isEmpty() {
        return this.keyManagerMap.isEmpty();
    }

    public synchronized void restrictToFipsIfEmptyAndGlobalInstance() throws GeneralSecurityException {
        if (this != KeyManagerRegistry.globalInstance()) {
            throw new GeneralSecurityException("Only the global instance can be restricted to FIPS.");
        }
        if (TinkFipsUtil.useOnlyFips()) {
            return;
        }
        if (!this.isEmpty()) {
            throw new GeneralSecurityException("Could not enable FIPS mode as Registry is not empty.");
        }
        TinkFipsUtil.setFipsRestricted();
    }

    private static interface KeyManagerContainer {
        public <P> KeyManager<P> getKeyManager(Class<P> var1) throws GeneralSecurityException;

        public KeyManager<?> getUntypedKeyManager();

        public Class<?> getImplementingClass();

        public Set<Class<?>> supportedPrimitives();
    }
}

