/*
 * Decompiled with CFR 0.152.
 */
package com.google.crypto.tink.subtle;

import com.google.crypto.tink.subtle.EngineFactory;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECField;
import java.security.spec.ECFieldFp;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPrivateKeySpec;
import java.security.spec.ECPublicKeySpec;
import java.security.spec.EllipticCurve;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import javax.crypto.KeyAgreement;

public final class EllipticCurves {
    public static ECParameterSpec getNistP256Params() {
        return EllipticCurves.getNistCurveSpec("115792089210356248762697446949407573530086143415290314195533631308867097853951", "115792089210356248762697446949407573529996955224135760342422259061068512044369", "5ac635d8aa3a93e7b3ebbd55769886bc651d06b0cc53b0f63bce3c3e27d2604b", "6b17d1f2e12c4247f8bce6e563a440f277037d812deb33a0f4a13945d898c296", "4fe342e2fe1a7f9b8ee7eb4a7c0f9e162bce33576b315ececbb6406837bf51f5");
    }

    public static ECParameterSpec getNistP384Params() {
        return EllipticCurves.getNistCurveSpec("39402006196394479212279040100143613805079739270465446667948293404245721771496870329047266088258938001861606973112319", "39402006196394479212279040100143613805079739270465446667946905279627659399113263569398956308152294913554433653942643", "b3312fa7e23ee7e4988e056be3f82d19181d9c6efe8141120314088f5013875ac656398d8a2ed19d2a85c8edd3ec2aef", "aa87ca22be8b05378eb1c71ef320ad746e1d3b628ba79b9859f741e082542a385502f25dbf55296c3a545e3872760ab7", "3617de4a96262c6f5d9e98bf9292dc29f8f41dbd289a147ce9da3113b5f0b8c00a60b1ce1d7e819d7a431d7c90ea0e5f");
    }

    public static ECParameterSpec getNistP521Params() {
        return EllipticCurves.getNistCurveSpec("6864797660130609714981900799081393217269435300143305409394463459185543183397656052122559640661454554977296311391480858037121987999716643812574028291115057151", "6864797660130609714981900799081393217269435300143305409394463459185543183397655394245057746333217197532963996371363321113864768612440380340372808892707005449", "051953eb9618e1c9a1f929a21a0b68540eea2da725b99b315f3b8b489918ef109e156193951ec7e937b1652c0bd3bb1bf073573df883d2c34f1ef451fd46b503f00", "c6858e06b70404e9cd9e3ecb662395b4429c648139053fb521f828af606b4d3dbaa14b5e77efe75928fe1dc127a2ffa8de3348b3c1856a429bf97e7e31c2e5bd66", "11839296a789a3bc0045c8a5fb42c7d1bd998f54449579b446817afbd17273e662c97ee72995ef42640c550b9013fad0761353c7086a272c24088be94769fd16650");
    }

    static void checkPointOnCurve(ECPoint point, EllipticCurve ec) throws GeneralSecurityException {
        BigInteger rhs;
        BigInteger p = EllipticCurves.getModulus(ec);
        BigInteger x = point.getAffineX();
        BigInteger y = point.getAffineY();
        if (x == null || y == null) {
            throw new GeneralSecurityException("point is at infinity");
        }
        if (x.signum() == -1 || x.compareTo(p) != -1) {
            throw new GeneralSecurityException("x is out of range");
        }
        if (y.signum() == -1 || y.compareTo(p) != -1) {
            throw new GeneralSecurityException("y is out of range");
        }
        BigInteger lhs = y.multiply(y).mod(p);
        if (!lhs.equals(rhs = x.multiply(x).add(ec.getA()).multiply(x).add(ec.getB()).mod(p))) {
            throw new GeneralSecurityException("Point is not on curve");
        }
    }

    static void checkPublicKey(ECPublicKey key) throws GeneralSecurityException {
        EllipticCurves.checkPointOnCurve(key.getW(), key.getParams().getCurve());
    }

    public static void validatePublicKey(ECPublicKey publicKey, ECPrivateKey privateKey) throws GeneralSecurityException {
        EllipticCurves.validatePublicKeySpec(publicKey, privateKey);
        EllipticCurves.checkPointOnCurve(publicKey.getW(), privateKey.getParams().getCurve());
    }

    static void validatePublicKeySpec(ECPublicKey publicKey, ECPrivateKey privateKey) throws GeneralSecurityException {
        try {
            ECParameterSpec publicKeySpec = publicKey.getParams();
            ECParameterSpec privateKeySpec = privateKey.getParams();
            if (!(publicKeySpec.getCurve().equals(privateKeySpec.getCurve()) && publicKeySpec.getGenerator().equals(privateKeySpec.getGenerator()) && publicKeySpec.getOrder().equals(privateKeySpec.getOrder()) && publicKeySpec.getCofactor() == privateKeySpec.getCofactor())) {
                throw new GeneralSecurityException("invalid public key spec");
            }
        }
        catch (IllegalArgumentException | NullPointerException ex) {
            throw new GeneralSecurityException(ex.toString());
        }
    }

    public static BigInteger getModulus(EllipticCurve curve) throws GeneralSecurityException {
        ECField field = curve.getField();
        if (field instanceof ECFieldFp) {
            return ((ECFieldFp)field).getP();
        }
        throw new GeneralSecurityException("Only curves over prime order fields are supported");
    }

    private static int fieldSizeInBits(EllipticCurve curve) throws GeneralSecurityException {
        return EllipticCurves.getModulus(curve).subtract(BigInteger.ONE).bitLength();
    }

    public static int fieldSizeInBytes(EllipticCurve curve) throws GeneralSecurityException {
        return (EllipticCurves.fieldSizeInBits(curve) + 7) / 8;
    }

    private static ECParameterSpec getNistCurveSpec(String decimalP, String decimalN, String hexB, String hexGX, String hexGY) {
        BigInteger p = new BigInteger(decimalP);
        BigInteger n = new BigInteger(decimalN);
        BigInteger three = new BigInteger("3");
        BigInteger a = p.subtract(three);
        BigInteger b = new BigInteger(hexB, 16);
        BigInteger gx = new BigInteger(hexGX, 16);
        BigInteger gy = new BigInteger(hexGY, 16);
        boolean h = true;
        ECFieldFp fp = new ECFieldFp(p);
        EllipticCurve curveSpec = new EllipticCurve(fp, a, b);
        ECPoint g = new ECPoint(gx, gy);
        ECParameterSpec ecSpec = new ECParameterSpec(curveSpec, g, n, 1);
        return ecSpec;
    }

    protected static BigInteger modSqrt(BigInteger x, BigInteger p) throws GeneralSecurityException {
        if (p.signum() != 1) {
            throw new InvalidAlgorithmParameterException("p must be positive");
        }
        x = x.mod(p);
        BigInteger squareRoot = null;
        if (x.equals(BigInteger.ZERO)) {
            return BigInteger.ZERO;
        }
        if (p.testBit(0) && p.testBit(1)) {
            BigInteger q = p.add(BigInteger.ONE).shiftRight(2);
            squareRoot = x.modPow(q, p);
        } else if (p.testBit(0) && !p.testBit(1)) {
            BigInteger d;
            BigInteger a;
            block10: {
                a = BigInteger.ONE;
                d = null;
                BigInteger q1 = p.subtract(BigInteger.ONE).shiftRight(1);
                int tries = 0;
                do {
                    if ((d = a.multiply(a).subtract(x).mod(p)).equals(BigInteger.ZERO)) {
                        return a;
                    }
                    BigInteger t = d.modPow(q1, p);
                    if (t.add(BigInteger.ONE).equals(p)) break block10;
                    if (!t.equals(BigInteger.ONE)) {
                        throw new InvalidAlgorithmParameterException("p is not prime");
                    }
                    a = a.add(BigInteger.ONE);
                } while (++tries != 128 || p.isProbablePrime(80));
                throw new InvalidAlgorithmParameterException("p is not prime");
            }
            BigInteger q = p.add(BigInteger.ONE).shiftRight(1);
            BigInteger u = a;
            BigInteger v = BigInteger.ONE;
            for (int bit = q.bitLength() - 2; bit >= 0; --bit) {
                BigInteger tmp = u.multiply(v);
                u = u.multiply(u).add(v.multiply(v).mod(p).multiply(d)).mod(p);
                v = tmp.add(tmp).mod(p);
                if (!q.testBit(bit)) continue;
                tmp = u.multiply(a).add(v.multiply(d)).mod(p);
                v = a.multiply(v).add(u).mod(p);
                u = tmp;
            }
            squareRoot = u;
        }
        if (squareRoot != null && squareRoot.multiply(squareRoot).mod(p).compareTo(x) != 0) {
            throw new GeneralSecurityException("Could not find a modular square root");
        }
        return squareRoot;
    }

    public static BigInteger getY(BigInteger x, boolean lsb, EllipticCurve curve) throws GeneralSecurityException {
        BigInteger p = EllipticCurves.getModulus(curve);
        BigInteger a = curve.getA();
        BigInteger b = curve.getB();
        BigInteger rhs = x.multiply(x).add(a).multiply(x).add(b).mod(p);
        BigInteger y = EllipticCurves.modSqrt(rhs, p);
        if (lsb != y.testBit(0)) {
            y = p.subtract(y).mod(p);
        }
        return y;
    }

    public static int encodingSizeInBytes(EllipticCurve curve, PointFormatType format) throws GeneralSecurityException {
        int coordinateSize = EllipticCurves.fieldSizeInBytes(curve);
        switch (format) {
            case UNCOMPRESSED: {
                return 2 * coordinateSize + 1;
            }
            case COMPRESSED: {
                return coordinateSize + 1;
            }
        }
        throw new GeneralSecurityException("unknown EC point format");
    }

    @Deprecated
    public static ECPoint ecPointDecode(EllipticCurve curve, PointFormatType format, byte[] encoded) throws GeneralSecurityException {
        return EllipticCurves.pointDecode(curve, format, encoded);
    }

    public static ECPoint pointDecode(CurveType curveType, PointFormatType format, byte[] encoded) throws GeneralSecurityException {
        return EllipticCurves.pointDecode(EllipticCurves.getCurveSpec(curveType).getCurve(), format, encoded);
    }

    public static ECPoint pointDecode(EllipticCurve curve, PointFormatType format, byte[] encoded) throws GeneralSecurityException {
        int coordinateSize = EllipticCurves.fieldSizeInBytes(curve);
        switch (format) {
            case UNCOMPRESSED: {
                if (encoded.length != 2 * coordinateSize + 1) {
                    throw new GeneralSecurityException("invalid point size");
                }
                if (encoded[0] != 4) {
                    throw new GeneralSecurityException("invalid point format");
                }
                BigInteger x = new BigInteger(1, Arrays.copyOfRange(encoded, 1, coordinateSize + 1));
                BigInteger y = new BigInteger(1, Arrays.copyOfRange(encoded, coordinateSize + 1, encoded.length));
                ECPoint point = new ECPoint(x, y);
                EllipticCurves.checkPointOnCurve(point, curve);
                return point;
            }
            case COMPRESSED: {
                boolean lsb;
                BigInteger p = EllipticCurves.getModulus(curve);
                if (encoded.length != coordinateSize + 1) {
                    throw new GeneralSecurityException("compressed point has wrong length");
                }
                if (encoded[0] == 2) {
                    lsb = false;
                } else if (encoded[0] == 3) {
                    lsb = true;
                } else {
                    throw new GeneralSecurityException("invalid format");
                }
                BigInteger x = new BigInteger(1, Arrays.copyOfRange(encoded, 1, encoded.length));
                if (x.signum() == -1 || x.compareTo(p) != -1) {
                    throw new GeneralSecurityException("x is out of range");
                }
                BigInteger y = EllipticCurves.getY(x, lsb, curve);
                return new ECPoint(x, y);
            }
        }
        throw new GeneralSecurityException("invalid format:" + (Object)((Object)format));
    }

    public static byte[] pointEncode(CurveType curveType, PointFormatType format, ECPoint point) throws GeneralSecurityException {
        return EllipticCurves.pointEncode(EllipticCurves.getCurveSpec(curveType).getCurve(), format, point);
    }

    public static byte[] pointEncode(EllipticCurve curve, PointFormatType format, ECPoint point) throws GeneralSecurityException {
        EllipticCurves.checkPointOnCurve(point, curve);
        int coordinateSize = EllipticCurves.fieldSizeInBytes(curve);
        switch (format) {
            case UNCOMPRESSED: {
                byte[] encoded = new byte[2 * coordinateSize + 1];
                byte[] x = point.getAffineX().toByteArray();
                byte[] y = point.getAffineY().toByteArray();
                System.arraycopy(y, 0, encoded, 1 + 2 * coordinateSize - y.length, y.length);
                System.arraycopy(x, 0, encoded, 1 + coordinateSize - x.length, x.length);
                encoded[0] = 4;
                return encoded;
            }
            case COMPRESSED: {
                byte[] encoded = new byte[coordinateSize + 1];
                byte[] x = point.getAffineX().toByteArray();
                System.arraycopy(x, 0, encoded, 1 + coordinateSize - x.length, x.length);
                encoded[0] = (byte)(point.getAffineY().testBit(0) ? 3 : 2);
                return encoded;
            }
        }
        throw new GeneralSecurityException("invalid format:" + (Object)((Object)format));
    }

    public static ECParameterSpec getCurveSpec(CurveType curve) throws NoSuchAlgorithmException {
        switch (curve) {
            case NIST_P256: {
                return EllipticCurves.getNistP256Params();
            }
            case NIST_P384: {
                return EllipticCurves.getNistP384Params();
            }
            case NIST_P521: {
                return EllipticCurves.getNistP521Params();
            }
        }
        throw new NoSuchAlgorithmException("curve not implemented:" + (Object)((Object)curve));
    }

    public static ECPublicKey getEcPublicKey(byte[] x509PublicKey) throws GeneralSecurityException {
        KeyFactory kf = EngineFactory.KEY_FACTORY.getInstance("EC");
        return (ECPublicKey)kf.generatePublic(new X509EncodedKeySpec(x509PublicKey));
    }

    public static ECPublicKey getEcPublicKey(CurveType curve, PointFormatType pointFormat, byte[] publicKey) throws GeneralSecurityException {
        return EllipticCurves.getEcPublicKey(EllipticCurves.getCurveSpec(curve), pointFormat, publicKey);
    }

    public static ECPublicKey getEcPublicKey(ECParameterSpec spec, PointFormatType pointFormat, byte[] publicKey) throws GeneralSecurityException {
        ECPoint point = EllipticCurves.pointDecode(spec.getCurve(), pointFormat, publicKey);
        ECPublicKeySpec pubSpec = new ECPublicKeySpec(point, spec);
        KeyFactory kf = EngineFactory.KEY_FACTORY.getInstance("EC");
        return (ECPublicKey)kf.generatePublic(pubSpec);
    }

    public static ECPublicKey getEcPublicKey(CurveType curve, byte[] x, byte[] y) throws GeneralSecurityException {
        ECParameterSpec ecParams = EllipticCurves.getCurveSpec(curve);
        BigInteger pubX = new BigInteger(1, x);
        BigInteger pubY = new BigInteger(1, y);
        ECPoint w = new ECPoint(pubX, pubY);
        EllipticCurves.checkPointOnCurve(w, ecParams.getCurve());
        ECPublicKeySpec spec = new ECPublicKeySpec(w, ecParams);
        KeyFactory kf = EngineFactory.KEY_FACTORY.getInstance("EC");
        return (ECPublicKey)kf.generatePublic(spec);
    }

    public static ECPrivateKey getEcPrivateKey(byte[] pkcs8PrivateKey) throws GeneralSecurityException {
        KeyFactory kf = EngineFactory.KEY_FACTORY.getInstance("EC");
        return (ECPrivateKey)kf.generatePrivate(new PKCS8EncodedKeySpec(pkcs8PrivateKey));
    }

    public static ECPrivateKey getEcPrivateKey(CurveType curve, byte[] keyValue) throws GeneralSecurityException {
        ECParameterSpec ecParams = EllipticCurves.getCurveSpec(curve);
        BigInteger privValue = new BigInteger(1, keyValue);
        ECPrivateKeySpec spec = new ECPrivateKeySpec(privValue, ecParams);
        KeyFactory kf = EngineFactory.KEY_FACTORY.getInstance("EC");
        return (ECPrivateKey)kf.generatePrivate(spec);
    }

    public static KeyPair generateKeyPair(CurveType curve) throws GeneralSecurityException {
        return EllipticCurves.generateKeyPair(EllipticCurves.getCurveSpec(curve));
    }

    public static KeyPair generateKeyPair(ECParameterSpec spec) throws GeneralSecurityException {
        KeyPairGenerator keyGen = EngineFactory.KEY_PAIR_GENERATOR.getInstance("EC");
        keyGen.initialize(spec);
        return keyGen.generateKeyPair();
    }

    private static void validateSharedSecret(byte[] secret, ECPrivateKey privateKey) throws GeneralSecurityException {
        EllipticCurve privateKeyCurve = privateKey.getParams().getCurve();
        BigInteger x = new BigInteger(1, secret);
        if (x.signum() == -1 || x.compareTo(EllipticCurves.getModulus(privateKeyCurve)) != -1) {
            throw new GeneralSecurityException("shared secret is out of range");
        }
        EllipticCurves.getY(x, true, privateKeyCurve);
    }

    public static byte[] computeSharedSecret(ECPrivateKey myPrivateKey, ECPublicKey peerPublicKey) throws GeneralSecurityException {
        EllipticCurves.validatePublicKeySpec(peerPublicKey, myPrivateKey);
        return EllipticCurves.computeSharedSecret(myPrivateKey, peerPublicKey.getW());
    }

    public static byte[] computeSharedSecret(ECPrivateKey myPrivateKey, ECPoint publicPoint) throws GeneralSecurityException {
        EllipticCurves.checkPointOnCurve(publicPoint, myPrivateKey.getParams().getCurve());
        ECParameterSpec privSpec = myPrivateKey.getParams();
        EllipticCurve privCurve = privSpec.getCurve();
        ECPublicKeySpec publicKeySpec = new ECPublicKeySpec(publicPoint, privSpec);
        KeyFactory kf = KeyFactory.getInstance("EC");
        PublicKey publicKey = kf.generatePublic(publicKeySpec);
        KeyAgreement ka = EngineFactory.KEY_AGREEMENT.getInstance("ECDH");
        ka.init(myPrivateKey);
        try {
            ka.doPhase(publicKey, true);
            byte[] secret = ka.generateSecret();
            EllipticCurves.validateSharedSecret(secret, myPrivateKey);
            return secret;
        }
        catch (IllegalStateException ex) {
            throw new GeneralSecurityException(ex.toString());
        }
    }

    public static enum CurveType {
        NIST_P256,
        NIST_P384,
        NIST_P521;

    }

    public static enum PointFormatType {
        UNCOMPRESSED,
        COMPRESSED;

    }
}

