/*
 * Decompiled with CFR 0.152.
 */
package com.google.crypto.tink.subtle;

import com.google.crypto.tink.annotations.Alpha;
import com.google.crypto.tink.subtle.ImmutableByteArray;
import com.google.crypto.tink.subtle.IndCpaCipher;
import com.google.crypto.tink.subtle.Random;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.IntBuffer;
import java.security.GeneralSecurityException;
import java.util.Arrays;

@Alpha
public abstract class DjbCipher
implements IndCpaCipher {
    static final int BLOCK_SIZE_IN_INTS = 16;
    public static final int BLOCK_SIZE_IN_BYTES = 64;
    static final int KEY_SIZE_IN_INTS = 8;
    public static final int KEY_SIZE_IN_BYTES = 32;
    private static final byte[] ZERO_16_BYTES = new byte[16];
    static final int[] SIGMA = DjbCipher.toIntArray(ByteBuffer.wrap(new byte[]{101, 120, 112, 97, 110, 100, 32, 51, 50, 45, 98, 121, 116, 101, 32, 107}));
    final ImmutableByteArray key;

    DjbCipher(byte[] key) {
        if (key.length != 32) {
            throw new IllegalArgumentException("The key length in bytes must be 32.");
        }
        this.key = ImmutableByteArray.of(key);
    }

    static DjbCipher chaCha20(byte[] key) {
        return new ChaCha20(key);
    }

    static DjbCipher xChaCha20(byte[] key) {
        return new XChaCha20(key);
    }

    static DjbCipher xSalsa20(byte[] key) {
        return new XSalsa20(key);
    }

    static int rotateLeft(int x, int y) {
        return x << y | x >>> -y;
    }

    static int[] toIntArray(ByteBuffer in) {
        IntBuffer intBuffer = in.order(ByteOrder.LITTLE_ENDIAN).asIntBuffer();
        int[] ret = new int[intBuffer.remaining()];
        intBuffer.get(ret);
        return ret;
    }

    int[] shuffleAdd(int[] state) {
        int[] x = Arrays.copyOf(state, state.length);
        this.shuffle(x);
        for (int i = 0; i < state.length; ++i) {
            int n = i;
            x[n] = x[n] + state[i];
        }
        return x;
    }

    byte[] getAuthenticatorKey(byte[] nonce) {
        return new KeyStream(this, nonce, 0).first(32);
    }

    abstract void shuffle(int[] var1);

    abstract int[] initialState(byte[] var1, int var2);

    abstract void incrementCounter(int[] var1);

    abstract int nonceSizeInBytes();

    abstract KeyStream getKeyStream(byte[] var1);

    private void process(ByteBuffer output, ByteBuffer input, KeyStream keyStream) {
        ByteBuffer buf = ByteBuffer.allocate(64).order(ByteOrder.LITTLE_ENDIAN);
        while (input.hasRemaining()) {
            int todo = input.remaining() < 64 ? input.remaining() : 64;
            buf.asIntBuffer().put(keyStream.next());
            for (int j = 0; j < todo; ++j) {
                output.put((byte)(input.get() ^ buf.get(j)));
            }
        }
    }

    void process(ByteBuffer output, ByteBuffer input, byte[] nonce, int counter) {
        this.process(output, input, new KeyStream(this, nonce, counter));
    }

    @Override
    public byte[] encrypt(byte[] plaintext) throws GeneralSecurityException {
        ByteBuffer ciphertext = ByteBuffer.allocate(this.nonceSizeInBytes() + plaintext.length);
        this.encrypt(ciphertext, plaintext);
        return ciphertext.array();
    }

    void encrypt(ByteBuffer output, byte[] plaintext) throws GeneralSecurityException {
        if (plaintext.length > Integer.MAX_VALUE - this.nonceSizeInBytes()) {
            throw new GeneralSecurityException("plaintext too long");
        }
        if (output.remaining() < plaintext.length + this.nonceSizeInBytes()) {
            throw new IllegalArgumentException("Given ByteBuffer output is too small");
        }
        byte[] nonce = Random.randBytes(this.nonceSizeInBytes());
        output.put(nonce);
        this.process(output, ByteBuffer.wrap(plaintext), this.getKeyStream(nonce));
    }

    byte[] decrypt(ByteBuffer ciphertext) throws GeneralSecurityException {
        if (ciphertext.remaining() < this.nonceSizeInBytes()) {
            throw new GeneralSecurityException("ciphertext too short");
        }
        byte[] nonce = new byte[this.nonceSizeInBytes()];
        ciphertext.get(nonce);
        ByteBuffer plaintext = ByteBuffer.allocate(ciphertext.remaining());
        this.process(plaintext, ciphertext, this.getKeyStream(nonce));
        return plaintext.array();
    }

    @Override
    public byte[] decrypt(byte[] ciphertext) throws GeneralSecurityException {
        return this.decrypt(ByteBuffer.wrap(ciphertext));
    }

    static class XSalsa20
    extends DjbCipher {
        private XSalsa20(byte[] key) {
            super(key);
        }

        static void quarterRound(int[] x, int a, int b, int c, int d) {
            int n = b;
            x[n] = x[n] ^ XSalsa20.rotateLeft(x[a] + x[d], 7);
            int n2 = c;
            x[n2] = x[n2] ^ XSalsa20.rotateLeft(x[b] + x[a], 9);
            int n3 = d;
            x[n3] = x[n3] ^ XSalsa20.rotateLeft(x[c] + x[b], 13);
            int n4 = a;
            x[n4] = x[n4] ^ XSalsa20.rotateLeft(x[d] + x[c], 18);
        }

        static void columnRound(int[] state) {
            XSalsa20.quarterRound(state, 0, 4, 8, 12);
            XSalsa20.quarterRound(state, 5, 9, 13, 1);
            XSalsa20.quarterRound(state, 10, 14, 2, 6);
            XSalsa20.quarterRound(state, 15, 3, 7, 11);
        }

        static void rowRound(int[] state) {
            XSalsa20.quarterRound(state, 0, 1, 2, 3);
            XSalsa20.quarterRound(state, 5, 6, 7, 4);
            XSalsa20.quarterRound(state, 10, 11, 8, 9);
            XSalsa20.quarterRound(state, 15, 12, 13, 14);
        }

        private static void shuffleInternal(int[] state) {
            for (int i = 0; i < 10; ++i) {
                XSalsa20.columnRound(state);
                XSalsa20.rowRound(state);
            }
        }

        @Override
        void shuffle(int[] state) {
            XSalsa20.shuffleInternal(state);
        }

        private static void setSigma(int[] state) {
            state[0] = SIGMA[0];
            state[5] = SIGMA[1];
            state[10] = SIGMA[2];
            state[15] = SIGMA[3];
        }

        private static void setKey(int[] state, byte[] key) {
            int[] keyInt = XSalsa20.toIntArray(ByteBuffer.wrap(key));
            System.arraycopy(keyInt, 0, state, 1, 4);
            System.arraycopy(keyInt, 4, state, 11, 4);
        }

        static byte[] hSalsa20(byte[] key) {
            return XSalsa20.hSalsa20(key, ZERO_16_BYTES);
        }

        private static byte[] hSalsa20(byte[] key, byte[] nonce) {
            int[] state = new int[16];
            XSalsa20.setSigma(state);
            XSalsa20.setKey(state, key);
            int[] nonceInt = XSalsa20.toIntArray(ByteBuffer.wrap(nonce));
            state[6] = nonceInt[0];
            state[7] = nonceInt[1];
            state[8] = nonceInt[2];
            state[9] = nonceInt[3];
            XSalsa20.shuffleInternal(state);
            state[1] = state[5];
            state[2] = state[10];
            state[3] = state[15];
            state[4] = state[6];
            state[5] = state[7];
            state[6] = state[8];
            state[7] = state[9];
            ByteBuffer buf = ByteBuffer.allocate(32).order(ByteOrder.LITTLE_ENDIAN);
            buf.asIntBuffer().put(state, 0, 8);
            return buf.array();
        }

        @Override
        int[] initialState(byte[] nonce, int counter) {
            int[] state = new int[16];
            XSalsa20.setSigma(state);
            XSalsa20.setKey(state, XSalsa20.hSalsa20(this.key.getBytes(), nonce));
            int[] nonceInt = XSalsa20.toIntArray(ByteBuffer.wrap(nonce));
            state[6] = nonceInt[4];
            state[7] = nonceInt[5];
            state[8] = counter;
            state[9] = 0;
            return state;
        }

        @Override
        void incrementCounter(int[] state) {
            state[8] = state[8] + 1;
            if (state[8] == 0) {
                state[9] = state[9] + 1;
            }
        }

        @Override
        int nonceSizeInBytes() {
            return 24;
        }

        @Override
        KeyStream getKeyStream(byte[] nonce) {
            KeyStream keyStream = new KeyStream(this, nonce, 0);
            keyStream.first(32);
            return keyStream;
        }
    }

    static class XChaCha20
    extends ChaCha20Base {
        private XChaCha20(byte[] key) {
            super(key);
        }

        @Override
        int[] initialState(byte[] nonce, int counter) {
            int[] state = new int[16];
            ChaCha20Base.setSigma(state);
            ChaCha20Base.setKey(state, XChaCha20.hChaCha20(this.key.getBytes(), nonce));
            int[] nonceInt = XChaCha20.toIntArray(ByteBuffer.wrap(nonce));
            state[14] = nonceInt[4];
            state[15] = nonceInt[5];
            state[12] = counter;
            state[13] = 0;
            return state;
        }

        @Override
        void incrementCounter(int[] state) {
            state[12] = state[12] + 1;
            if (state[12] == 0) {
                state[13] = state[13] + 1;
            }
        }

        @Override
        int nonceSizeInBytes() {
            return 24;
        }

        @Override
        KeyStream getKeyStream(byte[] nonce) {
            return new KeyStream(this, nonce, 1);
        }
    }

    static class ChaCha20
    extends ChaCha20Base {
        private ChaCha20(byte[] key) {
            super(key);
        }

        @Override
        int[] initialState(byte[] nonce, int counter) {
            int[] state = new int[16];
            ChaCha20Base.setSigma(state);
            ChaCha20Base.setKey(state, this.key.getBytes());
            state[12] = counter;
            System.arraycopy(ChaCha20.toIntArray(ByteBuffer.wrap(nonce)), 0, state, 13, this.nonceSizeInBytes() / 4);
            return state;
        }

        @Override
        void incrementCounter(int[] state) {
            state[12] = state[12] + 1;
        }

        @Override
        int nonceSizeInBytes() {
            return 12;
        }

        @Override
        KeyStream getKeyStream(byte[] nonce) {
            return new KeyStream(this, nonce, 1);
        }
    }

    static abstract class ChaCha20Base
    extends DjbCipher {
        private ChaCha20Base(byte[] key) {
            super(key);
        }

        static void quarterRound(int[] x, int a, int b, int c, int d) {
            int n = a;
            x[n] = x[n] + x[b];
            x[d] = ChaCha20Base.rotateLeft(x[d] ^ x[a], 16);
            int n2 = c;
            x[n2] = x[n2] + x[d];
            x[b] = ChaCha20Base.rotateLeft(x[b] ^ x[c], 12);
            int n3 = a;
            x[n3] = x[n3] + x[b];
            x[d] = ChaCha20Base.rotateLeft(x[d] ^ x[a], 8);
            int n4 = c;
            x[n4] = x[n4] + x[d];
            x[b] = ChaCha20Base.rotateLeft(x[b] ^ x[c], 7);
        }

        static void shuffleInternal(int[] state) {
            for (int i = 0; i < 10; ++i) {
                ChaCha20Base.quarterRound(state, 0, 4, 8, 12);
                ChaCha20Base.quarterRound(state, 1, 5, 9, 13);
                ChaCha20Base.quarterRound(state, 2, 6, 10, 14);
                ChaCha20Base.quarterRound(state, 3, 7, 11, 15);
                ChaCha20Base.quarterRound(state, 0, 5, 10, 15);
                ChaCha20Base.quarterRound(state, 1, 6, 11, 12);
                ChaCha20Base.quarterRound(state, 2, 7, 8, 13);
                ChaCha20Base.quarterRound(state, 3, 4, 9, 14);
            }
        }

        @Override
        void shuffle(int[] state) {
            ChaCha20Base.shuffleInternal(state);
        }

        private static void setSigma(int[] state) {
            System.arraycopy(SIGMA, 0, state, 0, SIGMA.length);
        }

        private static void setKey(int[] state, byte[] key) {
            int[] keyInt = ChaCha20Base.toIntArray(ByteBuffer.wrap(key));
            System.arraycopy(keyInt, 0, state, 4, keyInt.length);
        }

        static byte[] hChaCha20(byte[] key) {
            return ChaCha20Base.hChaCha20(key, ZERO_16_BYTES);
        }

        static byte[] hChaCha20(byte[] key, byte[] nonce) {
            int[] state = new int[16];
            ChaCha20Base.setSigma(state);
            ChaCha20Base.setKey(state, key);
            int[] nonceInt = ChaCha20Base.toIntArray(ByteBuffer.wrap(nonce));
            state[12] = nonceInt[0];
            state[13] = nonceInt[1];
            state[14] = nonceInt[2];
            state[15] = nonceInt[3];
            ChaCha20Base.shuffleInternal(state);
            state[4] = state[12];
            state[5] = state[13];
            state[6] = state[14];
            state[7] = state[15];
            ByteBuffer buf = ByteBuffer.allocate(32).order(ByteOrder.LITTLE_ENDIAN);
            buf.asIntBuffer().put(state, 0, 8);
            return buf.array();
        }
    }

    static class KeyStream {
        private DjbCipher djbCipher;
        private int[] state;
        private int[] keyStreamBlock;
        private int[] keyStreamBlockReturn;
        private int currentPosInBlock;
        private boolean readCalled;

        KeyStream(DjbCipher djbCipher, byte[] nonce, int counter) {
            this.djbCipher = djbCipher;
            this.keyStreamBlockReturn = new int[16];
            this.currentPosInBlock = 0;
            this.state = djbCipher.initialState(nonce, counter);
            this.keyStreamBlock = djbCipher.shuffleAdd(this.state);
            this.readCalled = false;
        }

        byte[] first(int byteLength) {
            if (this.readCalled) {
                throw new IllegalStateException("first can only be called once and before next().");
            }
            if (byteLength >= 64) {
                throw new IllegalArgumentException(String.format("length must be less than 64. length: %d", byteLength));
            }
            if (byteLength % 4 != 0) {
                throw new IllegalArgumentException(String.format("length must be a multiple of 4. length: %d", byteLength));
            }
            this.readCalled = true;
            this.currentPosInBlock = byteLength / 4;
            ByteBuffer out = ByteBuffer.allocate(byteLength).order(ByteOrder.LITTLE_ENDIAN);
            out.asIntBuffer().put(this.keyStreamBlock, 0, byteLength / 4);
            return out.array();
        }

        int[] next() {
            this.readCalled = true;
            System.arraycopy(this.keyStreamBlock, this.currentPosInBlock, this.keyStreamBlockReturn, 0, 16 - this.currentPosInBlock);
            this.djbCipher.incrementCounter(this.state);
            this.keyStreamBlock = this.djbCipher.shuffleAdd(this.state);
            System.arraycopy(this.keyStreamBlock, 0, this.keyStreamBlockReturn, 16 - this.currentPosInBlock, this.currentPosInBlock);
            return this.keyStreamBlockReturn;
        }
    }
}

