/*
 * Decompiled with CFR 0.152.
 */
package com.google.crypto.tink.subtle;

import com.google.crypto.tink.StreamingAead;
import com.google.crypto.tink.annotations.Alpha;
import com.google.crypto.tink.subtle.Bytes;
import com.google.crypto.tink.subtle.EngineFactory;
import com.google.crypto.tink.subtle.Hkdf;
import com.google.crypto.tink.subtle.Random;
import com.google.crypto.tink.subtle.StreamSegmentDecrypter;
import com.google.crypto.tink.subtle.StreamSegmentEncrypter;
import com.google.crypto.tink.subtle.StreamingAeadDecryptingChannel;
import com.google.crypto.tink.subtle.StreamingAeadEncryptingChannel;
import com.google.crypto.tink.subtle.StreamingAeadSeekableDecryptingChannel;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.SeekableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.security.GeneralSecurityException;
import java.security.InvalidAlgorithmParameterException;
import java.security.Key;
import java.util.Arrays;
import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

@Alpha
public final class AesCtrHmacStreaming
implements StreamingAead {
    private static final int NONCE_SIZE_IN_BYTES = 16;
    private static final int NONCE_PREFIX_IN_BYTES = 7;
    private static final String HKDF_ALGORITHM = "HMACSHA256";
    private static final String TAG_ALGORITHM = "HMACSHA256";
    private static final int HMAC_KEY_SIZE_IN_BYTES = 32;
    private int keySizeInBytes;
    private int tagSizeInBytes;
    private int ciphertextSegmentSize;
    private int plaintextSegmentSize;
    private int firstSegmentOffset;
    private byte[] ikm;

    public AesCtrHmacStreaming(byte[] ikm, int keySizeInBytes, int tagSizeInBytes, int ciphertextSegmentSize, int firstSegmentOffset) throws InvalidAlgorithmParameterException {
        AesCtrHmacStreaming.validateParameters(ikm.length, keySizeInBytes, tagSizeInBytes, ciphertextSegmentSize, firstSegmentOffset);
        this.ikm = Arrays.copyOf(ikm, ikm.length);
        this.keySizeInBytes = keySizeInBytes;
        this.tagSizeInBytes = tagSizeInBytes;
        this.ciphertextSegmentSize = ciphertextSegmentSize;
        this.firstSegmentOffset = firstSegmentOffset;
        this.plaintextSegmentSize = ciphertextSegmentSize - tagSizeInBytes;
    }

    private static void validateParameters(int ikmSize, int keySizeInBytes, int tagSizeInBytes, int ciphertextSegmentSize, int firstSegmentOffset) throws InvalidAlgorithmParameterException {
        boolean isValidKeySize;
        if (ikmSize < keySizeInBytes) {
            throw new InvalidAlgorithmParameterException("ikm to short");
        }
        boolean bl = isValidKeySize = keySizeInBytes == 16 || keySizeInBytes == 24 || keySizeInBytes == 32;
        if (!isValidKeySize) {
            throw new InvalidAlgorithmParameterException("Invalid key size");
        }
        if (tagSizeInBytes < 12 || tagSizeInBytes > 32) {
            throw new InvalidAlgorithmParameterException("Invalid tag size " + tagSizeInBytes);
        }
        int firstPlaintextSegment = ciphertextSegmentSize - firstSegmentOffset - tagSizeInBytes - keySizeInBytes - 7 - 1;
        if (firstPlaintextSegment <= 0) {
            throw new InvalidAlgorithmParameterException("ciphertextSegmentSize too small");
        }
    }

    public int getFirstSegmentOffset() {
        return this.firstSegmentOffset;
    }

    public int headerLength() {
        return 1 + this.keySizeInBytes + 7;
    }

    private int ciphertextOffset() {
        return this.headerLength() + this.firstSegmentOffset;
    }

    private int ciphertextOverhead() {
        return this.tagSizeInBytes;
    }

    public long expectedCiphertextSize(long plaintextSize) {
        long offset = this.ciphertextOffset();
        long fullSegments = (plaintextSize + offset) / (long)this.plaintextSegmentSize;
        long ciphertextSize = fullSegments * (long)this.ciphertextSegmentSize;
        long lastSegmentSize = (plaintextSize + offset) % (long)this.plaintextSegmentSize;
        if (lastSegmentSize > 0L) {
            ciphertextSize += lastSegmentSize + (long)this.tagSizeInBytes;
        }
        return ciphertextSize;
    }

    private static Cipher cipherInstance() throws GeneralSecurityException {
        return EngineFactory.CIPHER.getInstance("AES/CTR/NoPadding");
    }

    private static Mac macInstance() throws GeneralSecurityException {
        return EngineFactory.MAC.getInstance("HMACSHA256");
    }

    private byte[] randomSalt() {
        return Random.randBytes(this.keySizeInBytes);
    }

    public int getCiphertextSegmentSize() {
        return this.ciphertextSegmentSize;
    }

    private byte[] nonceForSegment(byte[] prefix, int segmentNr, boolean last) {
        ByteBuffer nonce = ByteBuffer.allocate(16);
        nonce.order(ByteOrder.BIG_ENDIAN);
        nonce.put(prefix);
        nonce.putInt(segmentNr);
        nonce.put((byte)(last ? 1 : 0));
        nonce.putInt(0);
        return nonce.array();
    }

    private byte[] randomNonce() {
        return Random.randBytes(7);
    }

    private byte[] deriveKeyMaterial(byte[] salt, byte[] aad) throws GeneralSecurityException {
        int keyMaterialSize = this.keySizeInBytes + 32;
        return Hkdf.computeHkdf("HMACSHA256", this.ikm, salt, aad, keyMaterialSize);
    }

    private SecretKeySpec deriveKeySpec(byte[] keyMaterial) throws GeneralSecurityException {
        return new SecretKeySpec(keyMaterial, 0, this.keySizeInBytes, "AES");
    }

    private SecretKeySpec deriveHmacKeySpec(byte[] keyMaterial) throws GeneralSecurityException {
        return new SecretKeySpec(keyMaterial, this.keySizeInBytes, 32, "HMACSHA256");
    }

    @Override
    public WritableByteChannel newEncryptingChannel(WritableByteChannel ciphertextChannel, byte[] associatedData) throws GeneralSecurityException, IOException {
        AesCtrHmacStreamEncrypter encrypter = new AesCtrHmacStreamEncrypter(associatedData);
        return new StreamingAeadEncryptingChannel(encrypter, ciphertextChannel, this.plaintextSegmentSize, this.ciphertextSegmentSize, this.ciphertextOffset());
    }

    @Override
    public ReadableByteChannel newDecryptingChannel(ReadableByteChannel ciphertextChannel, byte[] associatedData) throws GeneralSecurityException, IOException {
        return new StreamingAeadDecryptingChannel(new AesCtrHmacStreamDecrypter(), ciphertextChannel, associatedData, this.plaintextSegmentSize, this.ciphertextSegmentSize, this.ciphertextOffset(), this.headerLength());
    }

    @Override
    public SeekableByteChannel newSeekableDecryptingChannel(SeekableByteChannel ciphertextSource, byte[] associatedData) throws GeneralSecurityException, IOException {
        return new StreamingAeadSeekableDecryptingChannel(new AesCtrHmacStreamDecrypter(), ciphertextSource, associatedData, this.plaintextSegmentSize, this.ciphertextSegmentSize, this.ciphertextOffset(), this.ciphertextOverhead(), this.headerLength());
    }

    class AesCtrHmacStreamDecrypter
    implements StreamSegmentDecrypter {
        private SecretKeySpec keySpec;
        private SecretKeySpec hmacKeySpec;
        private Cipher cipher;
        private Mac mac;
        private byte[] noncePrefix;

        AesCtrHmacStreamDecrypter() {
        }

        @Override
        public synchronized void init(ByteBuffer header, byte[] aad) throws GeneralSecurityException {
            if (header.remaining() != AesCtrHmacStreaming.this.headerLength()) {
                throw new InvalidAlgorithmParameterException("Invalid header length");
            }
            byte firstByte = header.get();
            if (firstByte != AesCtrHmacStreaming.this.headerLength()) {
                throw new GeneralSecurityException("Invalid ciphertext");
            }
            this.noncePrefix = new byte[7];
            byte[] salt = new byte[AesCtrHmacStreaming.this.keySizeInBytes];
            header.get(salt);
            header.get(this.noncePrefix);
            byte[] keymaterial = AesCtrHmacStreaming.this.deriveKeyMaterial(salt, aad);
            this.keySpec = AesCtrHmacStreaming.this.deriveKeySpec(keymaterial);
            this.hmacKeySpec = AesCtrHmacStreaming.this.deriveHmacKeySpec(keymaterial);
            this.cipher = AesCtrHmacStreaming.cipherInstance();
            this.mac = AesCtrHmacStreaming.macInstance();
        }

        @Override
        public synchronized void decryptSegment(ByteBuffer ciphertext, int segmentNr, boolean isLastSegment, ByteBuffer plaintext) throws GeneralSecurityException {
            int position = ciphertext.position();
            byte[] nonce = AesCtrHmacStreaming.this.nonceForSegment(this.noncePrefix, segmentNr, isLastSegment);
            int ctLength = ciphertext.remaining();
            if (ctLength < AesCtrHmacStreaming.this.tagSizeInBytes) {
                throw new GeneralSecurityException("Ciphertext too short");
            }
            int ptLength = ctLength - AesCtrHmacStreaming.this.tagSizeInBytes;
            int startOfTag = position + ptLength;
            ByteBuffer ct = ciphertext.duplicate();
            ct.limit(startOfTag);
            ByteBuffer tagBuffer = ciphertext.duplicate();
            tagBuffer.position(startOfTag);
            assert (this.mac != null);
            assert (this.hmacKeySpec != null);
            this.mac.init(this.hmacKeySpec);
            this.mac.update(nonce);
            this.mac.update(ct);
            byte[] tag = this.mac.doFinal();
            tag = Arrays.copyOf(tag, AesCtrHmacStreaming.this.tagSizeInBytes);
            byte[] expectedTag = new byte[AesCtrHmacStreaming.this.tagSizeInBytes];
            assert (tagBuffer.remaining() == AesCtrHmacStreaming.this.tagSizeInBytes);
            tagBuffer.get(expectedTag);
            assert (expectedTag.length == tag.length);
            if (!Bytes.equal(expectedTag, tag)) {
                throw new GeneralSecurityException("Tag mismatch");
            }
            ciphertext.limit(startOfTag);
            this.cipher.init(1, (Key)this.keySpec, new IvParameterSpec(nonce));
            this.cipher.doFinal(ciphertext, plaintext);
        }
    }

    class AesCtrHmacStreamEncrypter
    implements StreamSegmentEncrypter {
        private final SecretKeySpec keySpec;
        private final SecretKeySpec hmacKeySpec;
        private final Cipher cipher = AesCtrHmacStreaming.access$000();
        private final Mac mac = AesCtrHmacStreaming.access$100();
        private final byte[] noncePrefix;
        private ByteBuffer header;
        private int encryptedSegments = 0;

        public AesCtrHmacStreamEncrypter(byte[] aad) throws GeneralSecurityException {
            byte[] salt = AesCtrHmacStreaming.this.randomSalt();
            this.noncePrefix = AesCtrHmacStreaming.this.randomNonce();
            this.header = ByteBuffer.allocate(AesCtrHmacStreaming.this.headerLength());
            this.header.put((byte)AesCtrHmacStreaming.this.headerLength());
            this.header.put(salt);
            this.header.put(this.noncePrefix);
            this.header.flip();
            byte[] keymaterial = AesCtrHmacStreaming.this.deriveKeyMaterial(salt, aad);
            this.keySpec = AesCtrHmacStreaming.this.deriveKeySpec(keymaterial);
            this.hmacKeySpec = AesCtrHmacStreaming.this.deriveHmacKeySpec(keymaterial);
        }

        @Override
        public ByteBuffer getHeader() {
            return this.header.asReadOnlyBuffer();
        }

        @Override
        public synchronized void encryptSegment(ByteBuffer plaintext, boolean isLastSegment, ByteBuffer ciphertext) throws GeneralSecurityException {
            int position = ciphertext.position();
            byte[] nonce = AesCtrHmacStreaming.this.nonceForSegment(this.noncePrefix, this.encryptedSegments, isLastSegment);
            this.cipher.init(1, (Key)this.keySpec, new IvParameterSpec(nonce));
            ++this.encryptedSegments;
            this.cipher.doFinal(plaintext, ciphertext);
            ByteBuffer ctCopy = ciphertext.duplicate();
            ctCopy.flip();
            ctCopy.position(position);
            this.mac.init(this.hmacKeySpec);
            this.mac.update(nonce);
            this.mac.update(ctCopy);
            byte[] tag = this.mac.doFinal();
            ciphertext.put(tag, 0, AesCtrHmacStreaming.this.tagSizeInBytes);
        }

        @Override
        public synchronized void encryptSegment(ByteBuffer part1, ByteBuffer part2, boolean isLastSegment, ByteBuffer ciphertext) throws GeneralSecurityException {
            int position = ciphertext.position();
            byte[] nonce = AesCtrHmacStreaming.this.nonceForSegment(this.noncePrefix, this.encryptedSegments, isLastSegment);
            this.cipher.init(1, (Key)this.keySpec, new IvParameterSpec(nonce));
            ++this.encryptedSegments;
            this.cipher.update(part1, ciphertext);
            this.cipher.doFinal(part2, ciphertext);
            ByteBuffer ctCopy = ciphertext.duplicate();
            ctCopy.flip();
            ctCopy.position(position);
            this.mac.init(this.hmacKeySpec);
            this.mac.update(nonce);
            this.mac.update(ctCopy);
            byte[] tag = this.mac.doFinal();
            ciphertext.put(tag, 0, AesCtrHmacStreaming.this.tagSizeInBytes);
        }

        @Override
        public int getEncryptedSegments() {
            return this.encryptedSegments;
        }
    }
}

