/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigtable.data.v2.stub.metrics;

import com.google.api.MonitoredResource;
import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutureCallback;
import com.google.api.core.ApiFutures;
import com.google.api.core.InternalApi;
import com.google.api.gax.core.CredentialsProvider;
import com.google.api.gax.core.FixedCredentialsProvider;
import com.google.api.gax.core.NoCredentialsProvider;
import com.google.api.gax.rpc.PermissionDeniedException;
import com.google.auth.Credentials;
import com.google.cloud.bigtable.data.v2.stub.metrics.BigtableExporterUtils;
import com.google.cloud.monitoring.v3.MetricServiceClient;
import com.google.cloud.monitoring.v3.MetricServiceSettings;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.monitoring.v3.CreateTimeSeriesRequest;
import com.google.monitoring.v3.ProjectName;
import com.google.monitoring.v3.TimeSeries;
import com.google.protobuf.Empty;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.metrics.InstrumentType;
import io.opentelemetry.sdk.metrics.data.AggregationTemporality;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.export.MetricExporter;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.Nullable;

@InternalApi
public final class BigtableCloudMonitoringExporter
implements MetricExporter {
    private static final Logger logger = Logger.getLogger(BigtableCloudMonitoringExporter.class.getName());
    @Deprecated
    @Nullable
    private static final String MONITORING_ENDPOINT_OVERRIDE_SYS_PROP = System.getProperty("bigtable.test-monitoring-endpoint");
    private static final String APPLICATION_RESOURCE_PROJECT_ID = "project_id";
    private static final int EXPORT_BATCH_SIZE_LIMIT = 200;
    private final String exporterName;
    private final MetricServiceClient client;
    private final TimeSeriesConverter timeSeriesConverter;
    private final AtomicBoolean isShutdown = new AtomicBoolean(false);
    private CompletableResultCode lastExportCode;
    private final AtomicBoolean exportFailureLogged = new AtomicBoolean(false);

    static BigtableCloudMonitoringExporter create(String exporterName, @Nullable Credentials credentials, @Nullable String endpoint, String universeDomain, TimeSeriesConverter converter) throws IOException {
        Preconditions.checkNotNull((Object)universeDomain);
        MetricServiceSettings.Builder settingsBuilder = MetricServiceSettings.newBuilder();
        CredentialsProvider credentialsProvider = Optional.ofNullable(credentials).map(FixedCredentialsProvider::create).orElse((CredentialsProvider)NoCredentialsProvider.create());
        settingsBuilder.setCredentialsProvider(credentialsProvider);
        settingsBuilder.setUniverseDomain(universeDomain);
        if (MONITORING_ENDPOINT_OVERRIDE_SYS_PROP != null) {
            logger.warning("Setting the monitoring endpoint through system variable will be removed in future versions");
            settingsBuilder.setEndpoint(MONITORING_ENDPOINT_OVERRIDE_SYS_PROP);
        }
        if (endpoint != null) {
            settingsBuilder.setEndpoint(endpoint);
        }
        Duration timeout = Duration.ofMinutes(1L);
        settingsBuilder.createServiceTimeSeriesSettings().setSimpleTimeoutNoRetriesDuration(timeout);
        return new BigtableCloudMonitoringExporter(exporterName, MetricServiceClient.create((MetricServiceSettings)settingsBuilder.build()), converter);
    }

    @VisibleForTesting
    BigtableCloudMonitoringExporter(String exporterName, MetricServiceClient client, TimeSeriesConverter converter) {
        this.exporterName = exporterName;
        this.client = client;
        this.timeSeriesConverter = converter;
    }

    public CompletableResultCode export(Collection<MetricData> metricData) {
        Preconditions.checkState((!this.isShutdown.get() ? 1 : 0) != 0, (Object)"Exporter is shutting down");
        this.lastExportCode = this.doExport(metricData);
        return this.lastExportCode;
    }

    private CompletableResultCode doExport(Collection<MetricData> metricData) {
        Map<ProjectName, List<TimeSeries>> bigtableTimeSeries;
        try {
            bigtableTimeSeries = this.timeSeriesConverter.convert(metricData);
        }
        catch (Throwable t) {
            logger.log(Level.WARNING, String.format("Failed to convert %s metric data to cloud monitoring timeseries.", this.exporterName), t);
            return CompletableResultCode.ofFailure();
        }
        if (bigtableTimeSeries.isEmpty()) {
            return CompletableResultCode.ofSuccess();
        }
        final CompletableResultCode exportCode = new CompletableResultCode();
        bigtableTimeSeries.forEach((projectName, ts) -> {
            ApiFuture<List<Empty>> future = this.exportTimeSeries((ProjectName)projectName, (List<TimeSeries>)ts);
            ApiFutures.addCallback(future, (ApiFutureCallback)new ApiFutureCallback<List<Empty>>(){

                public void onFailure(Throwable throwable) {
                    if (BigtableCloudMonitoringExporter.this.exportFailureLogged.compareAndSet(false, true)) {
                        String msg = String.format("createServiceTimeSeries request failed for %s.", BigtableCloudMonitoringExporter.this.exporterName);
                        if (throwable instanceof PermissionDeniedException) {
                            msg = msg + String.format(" Need monitoring metric writer permission on project=%s. Follow https://cloud.google.com/bigtable/docs/client-side-metrics-setup to set up permissions.", projectName.getProject());
                        }
                        logger.log(Level.WARNING, msg, throwable);
                    }
                    exportCode.fail();
                }

                public void onSuccess(List<Empty> emptyList) {
                    BigtableCloudMonitoringExporter.this.exportFailureLogged.set(false);
                    exportCode.succeed();
                }
            }, (Executor)MoreExecutors.directExecutor());
        });
        return exportCode;
    }

    private ApiFuture<List<Empty>> exportTimeSeries(ProjectName projectName, List<TimeSeries> timeSeries) {
        ArrayList<ApiFuture> batchResults = new ArrayList<ApiFuture>();
        for (List batch : Iterables.partition(timeSeries, (int)200)) {
            CreateTimeSeriesRequest req = CreateTimeSeriesRequest.newBuilder().setName(projectName.toString()).addAllTimeSeries((Iterable)batch).build();
            ApiFuture f = this.client.createServiceTimeSeriesCallable().futureCall((Object)req);
            batchResults.add(f);
        }
        return ApiFutures.allAsList(batchResults);
    }

    public CompletableResultCode flush() {
        if (this.lastExportCode != null) {
            return this.lastExportCode;
        }
        return CompletableResultCode.ofSuccess();
    }

    public CompletableResultCode shutdown() {
        if (!this.isShutdown.compareAndSet(false, true)) {
            logger.log(Level.WARNING, "shutdown is called multiple times");
            return CompletableResultCode.ofSuccess();
        }
        CompletableResultCode flushResult = this.flush();
        CompletableResultCode shutdownResult = new CompletableResultCode();
        flushResult.whenComplete(() -> {
            Throwable throwable = null;
            try {
                this.client.shutdown();
            }
            catch (Throwable e) {
                logger.log(Level.WARNING, "failed to shutdown the monitoring client", e);
                throwable = e;
            }
            if (throwable != null) {
                shutdownResult.fail();
            } else {
                shutdownResult.succeed();
            }
        });
        return CompletableResultCode.ofAll(Arrays.asList(flushResult, shutdownResult));
    }

    public AggregationTemporality getAggregationTemporality(InstrumentType instrumentType) {
        return AggregationTemporality.CUMULATIVE;
    }

    static interface TimeSeriesConverter {
        public Map<ProjectName, List<TimeSeries>> convert(Collection<MetricData> var1);
    }

    static class InternalTimeSeriesConverter
    implements TimeSeriesConverter {
        private static final ImmutableList<String> APPLICATION_METRICS = (ImmutableList)ImmutableSet.of((Object)"per_connection_error_count").stream().map(m -> "bigtable.googleapis.com/internal/client/" + m).collect(ImmutableList.toImmutableList());
        private final Supplier<MonitoredResource> monitoredResource;

        InternalTimeSeriesConverter(Supplier<MonitoredResource> monitoredResource) {
            this.monitoredResource = monitoredResource;
        }

        @Override
        public Map<ProjectName, List<TimeSeries>> convert(Collection<MetricData> metricData) {
            MonitoredResource monitoredResource = (MonitoredResource)this.monitoredResource.get();
            if (monitoredResource == null) {
                return ImmutableMap.of();
            }
            return ImmutableMap.of((Object)ProjectName.of((String)monitoredResource.getLabelsOrThrow(BigtableCloudMonitoringExporter.APPLICATION_RESOURCE_PROJECT_ID)), BigtableExporterUtils.convertToApplicationResourceTimeSeries(metricData, monitoredResource));
        }
    }

    static class PublicTimeSeriesConverter
    implements TimeSeriesConverter {
        private static final ImmutableList<String> BIGTABLE_TABLE_METRICS = (ImmutableList)ImmutableSet.of((Object)"operation_latencies", (Object)"attempt_latencies", (Object)"attempt_latencies2", (Object)"server_latencies", (Object)"first_response_latencies", (Object)"throttling_latencies", (Object[])new String[]{"application_latencies", "retry_count", "connectivity_error_count", "remaining_deadline"}).stream().map(m -> "bigtable.googleapis.com/internal/client/" + m).collect(ImmutableList.toImmutableList());
        private final String taskId;

        PublicTimeSeriesConverter() {
            this((String)BigtableExporterUtils.DEFAULT_TASK_VALUE.get());
        }

        PublicTimeSeriesConverter(String taskId) {
            this.taskId = taskId;
        }

        @Override
        public Map<ProjectName, List<TimeSeries>> convert(Collection<MetricData> metricData) {
            List<MetricData> relevantData = metricData.stream().filter(md -> BIGTABLE_TABLE_METRICS.contains((Object)md.getName())).collect(Collectors.toList());
            if (relevantData.isEmpty()) {
                return ImmutableMap.of();
            }
            return BigtableExporterUtils.convertToBigtableTimeSeries(relevantData, this.taskId);
        }
    }
}

