/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.hadoop.fs.gcs;

import com.google.cloud.hadoop.fs.gcs.GhfsGlobalStorageStatistics;
import com.google.cloud.hadoop.fs.gcs.GhfsInputStreamStatistics;
import com.google.cloud.hadoop.fs.gcs.GhfsStatistic;
import com.google.cloud.hadoop.fs.gcs.VectoredReadOptions;
import com.google.cloud.hadoop.gcsio.FileInfo;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageFileSystem;
import com.google.cloud.hadoop.gcsio.GoogleCloudStorageReadOptions;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.flogger.GoogleLogger;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.Closeable;
import java.io.EOFException;
import java.io.IOException;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.function.IntFunction;
import javax.annotation.Nonnull;
import org.apache.hadoop.fs.FileRange;
import org.apache.hadoop.fs.VectoredReadUtils;
import org.apache.hadoop.fs.impl.CombinedFileRange;

@VisibleForTesting
public class VectoredIOImpl
implements Closeable {
    private static final GoogleLogger logger = GoogleLogger.forEnclosingClass();
    private final BlockingQueue taskQueue = new LinkedBlockingQueue();
    private final VectoredReadOptions vectoredReadOptions;
    private final GhfsGlobalStorageStatistics storageStatistics;
    private ExecutorService boundedThreadPool;

    public VectoredIOImpl(VectoredReadOptions vectoredReadOptions, GhfsGlobalStorageStatistics storageStatistics) {
        this.vectoredReadOptions = vectoredReadOptions;
        this.boundedThreadPool = new ThreadPoolExecutor(vectoredReadOptions.getReadThreads(), vectoredReadOptions.getReadThreads(), 0L, TimeUnit.MILLISECONDS, (BlockingQueue<Runnable>)this.taskQueue, new ThreadFactoryBuilder().setNameFormat("vectoredRead-range-pool-%d").setDaemon(true).build());
        this.storageStatistics = storageStatistics;
    }

    public void readVectored(List<? extends FileRange> ranges, IntFunction<ByteBuffer> allocate, GoogleCloudStorageFileSystem gcsFs, FileInfo fileInfo, @Nonnull URI gcsPath, GhfsInputStreamStatistics streamStatistics) throws IOException {
        VectoredReadChannel vectoredReadChannel = new VectoredReadChannel(gcsFs, fileInfo, gcsPath, streamStatistics);
        vectoredReadChannel.readVectored(ranges, allocate);
    }

    @VisibleForTesting
    public List<? extends FileRange> validateNonOverlappingAndReturnSortedRanges(List<? extends FileRange> input) {
        if (input.size() == 1) {
            return input;
        }
        FileRange[] sortedRanges = VectoredReadUtils.sortRanges(input);
        FileRange prev = null;
        for (FileRange current : sortedRanges) {
            if (prev != null && current.getOffset() < prev.getOffset() + (long)prev.getLength()) {
                throw new IllegalArgumentException(String.format("Overlapping ranges not supported, overlapping range: %s, %s", prev, current));
            }
            prev = current;
        }
        return Arrays.asList(sortedRanges);
    }

    @Override
    public void close() {
        try {
            if (this.boundedThreadPool != null) {
                this.boundedThreadPool.shutdown();
            }
        }
        finally {
            this.boundedThreadPool = null;
        }
    }

    class VectoredReadChannel {
        private final GhfsInputStreamStatistics streamStatistics;
        private final ReadChannelProvider channelProvider;

        public VectoredReadChannel(GoogleCloudStorageFileSystem gcsFs, FileInfo fileInfo, URI gcsPath, GhfsInputStreamStatistics streamStatistics) {
            this.channelProvider = new ReadChannelProvider(gcsFs, fileInfo, gcsPath);
            this.streamStatistics = streamStatistics;
        }

        private void readVectored(List<? extends FileRange> ranges, IntFunction<ByteBuffer> allocate) throws IOException {
            List<? extends FileRange> sortedRanges = VectoredIOImpl.this.validateNonOverlappingAndReturnSortedRanges(ranges);
            for (FileRange fileRange : ranges) {
                VectoredReadUtils.validateRangeRequest((FileRange)fileRange);
                CompletableFuture result = new CompletableFuture();
                fileRange.setData(result);
            }
            if (this.shouldMergeRanges(ranges)) {
                this.updateRangeSizeCounters(sortedRanges.size(), sortedRanges.size());
                for (FileRange fileRange : sortedRanges) {
                    long startTimer = System.currentTimeMillis();
                    VectoredIOImpl.this.boundedThreadPool.submit(() -> {
                        ((GoogleLogger.Api)logger.atFiner()).log("Submitting range %s for execution.", (Object)sortedRange);
                        this.readSingleRange(sortedRange, allocate, this.channelProvider);
                        long endTimer = System.currentTimeMillis();
                        VectoredIOImpl.this.storageStatistics.updateStats(GhfsStatistic.STREAM_READ_VECTORED_READ_RANGE_DURATION, endTimer - startTimer, (Object)this.channelProvider.gcsPath);
                    });
                }
            } else {
                List<CombinedFileRange> combinedFileRanges = this.getCombinedFileRange(sortedRanges);
                this.updateRangeSizeCounters(sortedRanges.size(), combinedFileRanges.size());
                for (CombinedFileRange combinedFileRange : combinedFileRanges) {
                    CompletableFuture result = new CompletableFuture();
                    combinedFileRange.setData(result);
                    long startTimer = System.currentTimeMillis();
                    VectoredIOImpl.this.boundedThreadPool.submit(() -> {
                        ((GoogleLogger.Api)logger.atFiner()).log("Submitting combinedRange %s for execution.", (Object)combinedFileRange);
                        this.readCombinedRange(combinedFileRange, allocate, this.channelProvider);
                        long endTimer = System.currentTimeMillis();
                        VectoredIOImpl.this.storageStatistics.updateStats(GhfsStatistic.STREAM_READ_VECTORED_READ_RANGE_DURATION, endTimer - startTimer, (Object)this.channelProvider.gcsPath);
                    });
                }
            }
        }

        private void updateRangeSizeCounters(int incomingRangeSize, int combinedRangeSize) {
            VectoredIOImpl.this.storageStatistics.incrementCounter(GhfsStatistic.STREAM_READ_VECTORED_READ_INCOMING_RANGES, (long)incomingRangeSize);
            VectoredIOImpl.this.storageStatistics.incrementCounter(GhfsStatistic.STREAM_READ_VECTORED_READ_COMBINED_RANGES, (long)combinedRangeSize);
        }

        private void updateBytesRead(int readBytes) {
            this.streamStatistics.bytesRead(readBytes);
            VectoredIOImpl.this.storageStatistics.streamReadBytes(readBytes);
        }

        private List<CombinedFileRange> getCombinedFileRange(List<? extends FileRange> sortedRanges) {
            return VectoredReadUtils.mergeSortedRanges(sortedRanges, (int)1, (int)VectoredIOImpl.this.vectoredReadOptions.getMinSeekVectoredReadSize(), (int)VectoredIOImpl.this.vectoredReadOptions.getMergeRangeMaxSize());
        }

        private void readCombinedRange(CombinedFileRange combinedFileRange, IntFunction<ByteBuffer> allocate, ReadChannelProvider channelProvider) {
            try (SeekableByteChannel channel = channelProvider.getReadChannel();){
                channel.position(combinedFileRange.getOffset());
                ByteBuffer readContent = allocate.apply(combinedFileRange.getLength());
                int numRead = channel.read(readContent);
                readContent.flip();
                ((GoogleLogger.Api)logger.atFiner()).log("Read combinedFileRange completed from range: %s, path: %s, readBytes: %d", (Object)combinedFileRange, (Object)channelProvider.gcsPath, (Object)numRead);
                if (numRead < 0) {
                    throw new EOFException(String.format("EOF reached while reading combinedFileRange, range: %s, path: %s, numRead: %d", combinedFileRange, channelProvider.gcsPath, numRead));
                }
                long currentPosition = combinedFileRange.getOffset();
                long totalBytesRead = 0L;
                for (FileRange child : combinedFileRange.getUnderlying()) {
                    ((GoogleLogger.Api)logger.atFiner()).log("Populating childRange: %s from combinedRange:%s", (Object)child, (Object)combinedFileRange);
                    int discardedBytes = (int)(child.getOffset() - currentPosition);
                    ((GoogleLogger.Api)logger.atFiner()).log("Discarding %d bytes at offset: %d from read combinedRange %s while updating childRange: %s", (Object)discardedBytes, (Object)currentPosition, (Object)combinedFileRange, (Object)child);
                    currentPosition = child.getOffset() + (long)child.getLength();
                    VectoredIOImpl.this.storageStatistics.incrementCounter(GhfsStatistic.STREAM_READ_VECTORED_EXTRA_READ_BYTES, (long)discardedBytes);
                    if ((long)numRead >= (totalBytesRead += (long)(discardedBytes + child.getLength()))) {
                        ByteBuffer childBuffer = VectoredReadUtils.sliceTo((ByteBuffer)readContent, (long)combinedFileRange.getOffset(), (FileRange)child);
                        child.getData().complete(childBuffer);
                        this.updateBytesRead(child.getLength());
                        continue;
                    }
                    throw new EOFException(String.format("EOF reached before all child ranges can be populated, combinedFileRange: %s, expected length: %s, readBytes: %s, path: %s", combinedFileRange, combinedFileRange.getLength(), numRead, channelProvider.gcsPath));
                }
                combinedFileRange.getData().complete(readContent);
            }
            catch (Exception e) {
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause((Throwable)e)).log("Exception while reading combinedFileRange:%s for path: %s", (Object)combinedFileRange, (Object)channelProvider.gcsPath);
                combinedFileRange.getData().completeExceptionally(e);
                this.completeExceptionally(combinedFileRange, e);
            }
        }

        private void completeExceptionally(CombinedFileRange combinedFileRange, Throwable e) {
            for (FileRange child : combinedFileRange.getUnderlying()) {
                if (child.getData().isDone()) continue;
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atFiner()).withCause(e)).log("Marking child:%s as `completeExceptionally` of combinedRange:%s", (Object)child, (Object)combinedFileRange);
                child.getData().completeExceptionally(new IOException(String.format("Error while populating childRange: %s from combinedRange: %s", child, combinedFileRange), e));
            }
        }

        private void readSingleRange(FileRange range, IntFunction<ByteBuffer> allocate, ReadChannelProvider channelProvider) {
            try (SeekableByteChannel channel = channelProvider.getReadChannel();){
                channel.position(range.getOffset());
                ByteBuffer dst = allocate.apply(range.getLength());
                int numRead = channel.read(dst.duplicate());
                if (numRead < range.getLength()) {
                    throw new EOFException(String.format("EOF reached before whole range can be read, range: %s, path: %s", range, channelProvider.gcsPath));
                }
                range.getData().complete(dst);
                this.updateBytesRead(range.getLength());
                ((GoogleLogger.Api)logger.atFiner()).log("Read single range completed from range: %s, path: %s", (Object)range, (Object)channelProvider.gcsPath);
            }
            catch (Exception e) {
                ((GoogleLogger.Api)((GoogleLogger.Api)logger.atWarning()).withCause((Throwable)e)).log("Exception while reading range:%s for path: %s", (Object)range, (Object)channelProvider.gcsPath);
                range.getData().completeExceptionally(e);
            }
        }

        private boolean shouldMergeRanges(List<? extends FileRange> ranges) {
            return VectoredReadUtils.isOrderedDisjoint(ranges, (int)1, (int)VectoredIOImpl.this.vectoredReadOptions.getMinSeekVectoredReadSize());
        }

        private class ReadChannelProvider {
            private final GoogleCloudStorageFileSystem gcsFs;
            private final FileInfo fileInfo;
            private final URI gcsPath;

            public ReadChannelProvider(GoogleCloudStorageFileSystem gcsFS, FileInfo fileInfo, URI gcsPath) {
                this.gcsFs = gcsFS;
                this.fileInfo = fileInfo;
                this.gcsPath = gcsPath;
            }

            public SeekableByteChannel getReadChannel() throws IOException {
                GoogleCloudStorageReadOptions options = this.channelReadOptions(this.gcsFs.getOptions().getCloudStorageOptions().getReadChannelOptions());
                if (this.fileInfo != null) {
                    return this.gcsFs.open(this.fileInfo, options);
                }
                return this.gcsFs.open(this.gcsPath, options);
            }

            private GoogleCloudStorageReadOptions channelReadOptions(GoogleCloudStorageReadOptions readOptions) {
                GoogleCloudStorageReadOptions.Builder builder = readOptions.toBuilder();
                builder.setReadExactRequestedBytesEnabled(true);
                return builder.build();
            }
        }
    }
}

