/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.store;

import com.google.bitcoin.core.Block;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ProtocolException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.StoredTransactionOutput;
import com.google.bitcoin.core.StoredUndoableBlock;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionOutputChanges;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.store.BlockStoreException;
import com.google.bitcoin.store.FullPrunedBlockStore;
import com.google.common.collect.Lists;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.LinkedList;
import java.util.List;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class H2FullPrunedBlockStore
implements FullPrunedBlockStore {
    private static final Logger log = LoggerFactory.getLogger(H2FullPrunedBlockStore.class);
    private Sha256Hash chainHeadHash;
    private StoredBlock chainHeadBlock;
    private Sha256Hash verifiedChainHeadHash;
    private StoredBlock verifiedChainHeadBlock;
    private NetworkParameters params;
    private ThreadLocal<Connection> conn;
    private List<Connection> allConnections;
    private String connectionURL;
    private int fullStoreDepth;
    static final String driver = "org.h2.Driver";
    static final String CREATE_SETTINGS_TABLE = "CREATE TABLE settings ( name VARCHAR(32) NOT NULL CONSTRAINT settings_pk PRIMARY KEY,value BLOB)";
    static final String CHAIN_HEAD_SETTING = "chainhead";
    static final String VERIFIED_CHAIN_HEAD_SETTING = "verifiedchainhead";
    static final String VERSION_SETTING = "version";
    static final String CREATE_HEADERS_TABLE = "CREATE TABLE headers ( hash BINARY(28) NOT NULL CONSTRAINT headers_pk PRIMARY KEY,chainWork BLOB NOT NULL,height INT NOT NULL,header BLOB NOT NULL,wasUndoable BOOL NOT NULL)";
    static final String CREATE_UNDOABLE_TABLE = "CREATE TABLE undoableBlocks ( hash BINARY(28) NOT NULL CONSTRAINT undoableBlocks_pk PRIMARY KEY,height INT NOT NULL,txOutChanges BLOB,transactions BLOB)";
    static final String CREATE_UNDOABLE_TABLE_INDEX = "CREATE INDEX heightIndex ON undoableBlocks (height)";
    static final String CREATE_OPEN_OUTPUT_TABLE = "CREATE TABLE openOutputs (hash BINARY(32) NOT NULL,index INT NOT NULL,height INT NOT NULL,value BLOB NOT NULL,scriptBytes BLOB NOT NULL,PRIMARY KEY (hash, index),)";

    public H2FullPrunedBlockStore(NetworkParameters params, String dbName, int fullStoreDepth) throws BlockStoreException {
        this.params = params;
        this.fullStoreDepth = fullStoreDepth;
        this.connectionURL = "jdbc:h2:" + dbName + ";create=true;LOCK_TIMEOUT=60000";
        this.conn = new ThreadLocal();
        this.allConnections = new LinkedList<Connection>();
        try {
            Class.forName(driver);
            log.info("org.h2.Driver loaded. ");
        }
        catch (ClassNotFoundException e) {
            log.error("check CLASSPATH for H2 jar ", (Throwable)e);
        }
        this.maybeConnect();
        try {
            if (!this.tableExists("settings")) {
                this.createTables();
            }
            this.initFromDatabase();
        }
        catch (SQLException e) {
            throw new BlockStoreException(e);
        }
    }

    public H2FullPrunedBlockStore(NetworkParameters params, String dbName, int fullStoreDepth, int cacheSize) throws BlockStoreException {
        this(params, dbName, fullStoreDepth);
        try {
            Statement s = this.conn.get().createStatement();
            s.executeUpdate("SET CACHE_SIZE " + cacheSize);
            s.close();
        }
        catch (SQLException e) {
            throw new BlockStoreException(e);
        }
    }

    private synchronized void maybeConnect() throws BlockStoreException {
        try {
            if (this.conn.get() != null) {
                return;
            }
            this.conn.set(DriverManager.getConnection(this.connectionURL));
            this.allConnections.add(this.conn.get());
            log.info("Made a new connection to database " + this.connectionURL);
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
    }

    @Override
    public synchronized void close() {
        for (Connection conn : this.allConnections) {
            try {
                conn.rollback();
            }
            catch (SQLException ex) {
                throw new RuntimeException(ex);
            }
        }
        this.allConnections.clear();
    }

    public void resetStore() throws BlockStoreException {
        this.maybeConnect();
        try {
            Statement s = this.conn.get().createStatement();
            s.executeUpdate("DROP TABLE settings");
            s.executeUpdate("DROP TABLE headers");
            s.executeUpdate("DROP TABLE undoableBlocks");
            s.executeUpdate("DROP TABLE openOutputs");
            s.close();
            this.createTables();
            this.initFromDatabase();
        }
        catch (SQLException ex) {
            throw new RuntimeException(ex);
        }
    }

    private void createTables() throws SQLException, BlockStoreException {
        Statement s = this.conn.get().createStatement();
        log.debug("H2FullPrunedBlockStore : CREATE headers table");
        s.executeUpdate(CREATE_HEADERS_TABLE);
        log.debug("H2FullPrunedBlockStore : CREATE settings table");
        s.executeUpdate(CREATE_SETTINGS_TABLE);
        log.debug("H2FullPrunedBlockStore : CREATE undoable block table");
        s.executeUpdate(CREATE_UNDOABLE_TABLE);
        log.debug("H2FullPrunedBlockStore : CREATE undoable block index");
        s.executeUpdate(CREATE_UNDOABLE_TABLE_INDEX);
        log.debug("H2FullPrunedBlockStore : CREATE open output table");
        s.executeUpdate(CREATE_OPEN_OUTPUT_TABLE);
        s.executeUpdate("INSERT INTO settings(name, value) VALUES('chainhead', NULL)");
        s.executeUpdate("INSERT INTO settings(name, value) VALUES('verifiedchainhead', NULL)");
        s.executeUpdate("INSERT INTO settings(name, value) VALUES('version', '03')");
        s.close();
        this.createNewStore(this.params);
    }

    private void initFromDatabase() throws SQLException, BlockStoreException {
        Statement s = this.conn.get().createStatement();
        ResultSet rs = s.executeQuery("SHOW TABLES");
        while (rs.next()) {
            if (!rs.getString(1).equalsIgnoreCase("openOutputsIndex")) continue;
            throw new BlockStoreException("Attempted to open a H2 database with an old schema, please reset database.");
        }
        rs = s.executeQuery("SELECT value FROM settings WHERE name = 'chainhead'");
        if (!rs.next()) {
            throw new BlockStoreException("corrupt H2 block store - no chain head pointer");
        }
        Sha256Hash hash = new Sha256Hash(rs.getBytes(1));
        rs.close();
        this.chainHeadBlock = this.get(hash);
        this.chainHeadHash = hash;
        if (this.chainHeadBlock == null) {
            throw new BlockStoreException("corrupt H2 block store - head block not found");
        }
        rs = s.executeQuery("SELECT value FROM settings WHERE name = 'verifiedchainhead'");
        if (!rs.next()) {
            throw new BlockStoreException("corrupt H2 block store - no verified chain head pointer");
        }
        hash = new Sha256Hash(rs.getBytes(1));
        rs.close();
        s.close();
        this.verifiedChainHeadBlock = this.get(hash);
        this.verifiedChainHeadHash = hash;
        if (this.verifiedChainHeadBlock == null) {
            throw new BlockStoreException("corrupt H2 block store - verified head block not found");
        }
    }

    private void createNewStore(NetworkParameters params) throws BlockStoreException {
        try {
            StoredBlock storedGenesisHeader = new StoredBlock(params.getGenesisBlock().cloneAsHeader(), params.getGenesisBlock().getWork(), 0);
            LinkedList genesisTransactions = Lists.newLinkedList();
            StoredUndoableBlock storedGenesis = new StoredUndoableBlock(params.getGenesisBlock().getHash(), genesisTransactions);
            this.put(storedGenesisHeader, storedGenesis);
            this.setChainHead(storedGenesisHeader);
            this.setVerifiedChainHead(storedGenesisHeader);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean tableExists(String table) throws SQLException {
        Statement s = this.conn.get().createStatement();
        try {
            ResultSet results = s.executeQuery("SELECT * FROM " + table + " WHERE 1 = 2");
            results.close();
            boolean bl = true;
            return bl;
        }
        catch (SQLException ex) {
            boolean bl = false;
            return bl;
        }
        finally {
            s.close();
        }
    }

    public void dumpSizes() throws SQLException, BlockStoreException {
        this.maybeConnect();
        Statement s = this.conn.get().createStatement();
        long size = 0L;
        long totalSize = 0L;
        int count = 0;
        ResultSet rs = s.executeQuery("SELECT name, value FROM settings");
        while (rs.next()) {
            size += (long)rs.getString(1).length();
            size += (long)rs.getBytes(2).length;
            ++count;
        }
        rs.close();
        System.out.printf("Settings size: %d, count: %d, average size: %f%n", size, count, (double)size / (double)count);
        totalSize += size;
        size = 0L;
        count = 0;
        rs = s.executeQuery("SELECT chainWork, header FROM headers");
        while (rs.next()) {
            size += 28L;
            size += (long)rs.getBytes(1).length;
            size += 4L;
            size += (long)rs.getBytes(2).length;
            ++count;
        }
        rs.close();
        System.out.printf("Headers size: %d, count: %d, average size: %f%n", size, count, (double)size / (double)count);
        totalSize += size;
        size = 0L;
        count = 0;
        rs = s.executeQuery("SELECT txOutChanges, transactions FROM undoableBlocks");
        while (rs.next()) {
            size += 28L;
            size += 4L;
            byte[] txOutChanges = rs.getBytes(1);
            byte[] transactions = rs.getBytes(2);
            size = txOutChanges == null ? (size += (long)transactions.length) : (size += (long)txOutChanges.length);
            ++count;
        }
        rs.close();
        System.out.printf("Undoable Blocks size: %d, count: %d, average size: %f%n", size, count, (double)size / (double)count);
        totalSize += size;
        size = 0L;
        count = 0;
        long scriptSize = 0L;
        rs = s.executeQuery("SELECT value, scriptBytes FROM openOutputs");
        while (rs.next()) {
            size += 32L;
            size += 4L;
            size += 4L;
            size += (long)rs.getBytes(1).length;
            size += (long)rs.getBytes(2).length;
            scriptSize += (long)rs.getBytes(2).length;
            ++count;
        }
        rs.close();
        System.out.printf("Open Outputs size: %d, count: %d, average size: %f, average script size: %f (%d in id indexes)%n", size, count, (double)size / (double)count, (double)scriptSize / (double)count, count * 8);
        System.out.println("Total Size: " + (totalSize += size));
        s.close();
    }

    private void putUpdateStoredBlock(StoredBlock storedBlock, boolean wasUndoable) throws SQLException {
        try {
            PreparedStatement s = this.conn.get().prepareStatement("INSERT INTO headers(hash, chainWork, height, header, wasUndoable) VALUES(?, ?, ?, ?, ?)");
            byte[] hashBytes = new byte[28];
            System.arraycopy(storedBlock.getHeader().getHash().getBytes(), 3, hashBytes, 0, 28);
            s.setBytes(1, hashBytes);
            s.setBytes(2, storedBlock.getChainWork().toByteArray());
            s.setInt(3, storedBlock.getHeight());
            s.setBytes(4, storedBlock.getHeader().unsafeBitcoinSerialize());
            s.setBoolean(5, wasUndoable);
            s.executeUpdate();
            s.close();
        }
        catch (SQLException e) {
            if (e.getErrorCode() != 23505 || !wasUndoable) {
                throw e;
            }
            PreparedStatement s = this.conn.get().prepareStatement("UPDATE headers SET wasUndoable=? WHERE hash=?");
            s.setBoolean(1, true);
            byte[] hashBytes = new byte[28];
            System.arraycopy(storedBlock.getHeader().getHash().getBytes(), 3, hashBytes, 0, 28);
            s.setBytes(2, hashBytes);
            s.executeUpdate();
            s.close();
        }
    }

    @Override
    public void put(StoredBlock storedBlock) throws BlockStoreException {
        this.maybeConnect();
        try {
            this.putUpdateStoredBlock(storedBlock, false);
        }
        catch (SQLException e) {
            throw new BlockStoreException(e);
        }
    }

    @Override
    public void put(StoredBlock storedBlock, StoredUndoableBlock undoableBlock) throws BlockStoreException {
        this.maybeConnect();
        byte[] hashBytes = new byte[28];
        System.arraycopy(storedBlock.getHeader().getHash().getBytes(), 3, hashBytes, 0, 28);
        int height = storedBlock.getHeight();
        byte[] transactions = null;
        byte[] txOutChanges = null;
        try {
            ByteArrayOutputStream bos = new ByteArrayOutputStream();
            if (undoableBlock.getTxOutChanges() != null) {
                undoableBlock.getTxOutChanges().serializeToStream(bos);
                txOutChanges = bos.toByteArray();
            } else {
                int numTxn = undoableBlock.getTransactions().size();
                bos.write(0xFF & numTxn >> 0);
                bos.write(0xFF & numTxn >> 8);
                bos.write(0xFF & numTxn >> 16);
                bos.write(0xFF & numTxn >> 24);
                for (Transaction tx : undoableBlock.getTransactions()) {
                    tx.bitcoinSerialize(bos);
                }
                transactions = bos.toByteArray();
            }
            bos.close();
        }
        catch (IOException e) {
            throw new BlockStoreException(e);
        }
        try {
            try {
                PreparedStatement s = this.conn.get().prepareStatement("INSERT INTO undoableBlocks(hash, height, txOutChanges, transactions) VALUES(?, ?, ?, ?)");
                s.setBytes(1, hashBytes);
                s.setInt(2, height);
                if (transactions == null) {
                    s.setBytes(3, txOutChanges);
                    s.setNull(4, 2004);
                } else {
                    s.setNull(3, 2004);
                    s.setBytes(4, transactions);
                }
                s.executeUpdate();
                s.close();
                try {
                    this.putUpdateStoredBlock(storedBlock, true);
                }
                catch (SQLException e) {
                    throw new BlockStoreException(e);
                }
            }
            catch (SQLException e) {
                if (e.getErrorCode() != 23505) {
                    throw new BlockStoreException(e);
                }
                PreparedStatement s = this.conn.get().prepareStatement("UPDATE undoableBlocks SET txOutChanges=?, transactions=? WHERE hash = ?");
                s.setBytes(3, hashBytes);
                if (transactions == null) {
                    s.setBytes(1, txOutChanges);
                    s.setNull(2, 2004);
                } else {
                    s.setNull(1, 2004);
                    s.setBytes(2, transactions);
                }
                s.executeUpdate();
                s.close();
            }
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
    }

    @Nullable
    public StoredBlock get(Sha256Hash hash, boolean wasUndoableOnly) throws BlockStoreException {
        if (this.chainHeadHash != null && this.chainHeadHash.equals(hash)) {
            return this.chainHeadBlock;
        }
        if (this.verifiedChainHeadHash != null && this.verifiedChainHeadHash.equals(hash)) {
            return this.verifiedChainHeadBlock;
        }
        this.maybeConnect();
        PreparedStatement s = null;
        try {
            s = this.conn.get().prepareStatement("SELECT chainWork, height, header, wasUndoable FROM headers WHERE hash = ?");
            byte[] hashBytes = new byte[28];
            System.arraycopy(hash.getBytes(), 3, hashBytes, 0, 28);
            s.setBytes(1, hashBytes);
            ResultSet results = s.executeQuery();
            if (!results.next()) {
                StoredBlock storedBlock = null;
                return storedBlock;
            }
            if (wasUndoableOnly && !results.getBoolean(4)) {
                StoredBlock storedBlock = null;
                return storedBlock;
            }
            BigInteger chainWork = new BigInteger(results.getBytes(1));
            int height = results.getInt(2);
            Block b = new Block(this.params, results.getBytes(3));
            b.verifyHeader();
            StoredBlock storedBlock = new StoredBlock(b, chainWork, height);
            return storedBlock;
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
        catch (ProtocolException e) {
            throw new BlockStoreException(e);
        }
        catch (VerificationException e) {
            throw new BlockStoreException(e);
        }
        finally {
            if (s != null) {
                try {
                    s.close();
                }
                catch (SQLException e) {
                    throw new BlockStoreException("Failed to close PreparedStatement");
                }
            }
        }
    }

    @Override
    @Nullable
    public StoredBlock get(Sha256Hash hash) throws BlockStoreException {
        return this.get(hash, false);
    }

    @Override
    @Nullable
    public StoredBlock getOnceUndoableStoredBlock(Sha256Hash hash) throws BlockStoreException {
        return this.get(hash, true);
    }

    @Override
    @Nullable
    public StoredUndoableBlock getUndoBlock(Sha256Hash hash) throws BlockStoreException {
        this.maybeConnect();
        PreparedStatement s = null;
        try {
            StoredUndoableBlock block;
            s = this.conn.get().prepareStatement("SELECT txOutChanges, transactions FROM undoableBlocks WHERE hash = ?");
            byte[] hashBytes = new byte[28];
            System.arraycopy(hash.getBytes(), 3, hashBytes, 0, 28);
            s.setBytes(1, hashBytes);
            ResultSet results = s.executeQuery();
            if (!results.next()) {
                StoredUndoableBlock storedUndoableBlock = null;
                return storedUndoableBlock;
            }
            byte[] txOutChanges = results.getBytes(1);
            byte[] transactions = results.getBytes(2);
            if (txOutChanges == null) {
                int offset = 0;
                int numTxn = (transactions[offset++] & 0xFF) << 0 | (transactions[offset++] & 0xFF) << 8 | (transactions[offset++] & 0xFF) << 16 | (transactions[offset++] & 0xFF) << 24;
                LinkedList<Transaction> transactionList = new LinkedList<Transaction>();
                for (int i = 0; i < numTxn; ++i) {
                    Transaction tx = new Transaction(this.params, transactions, offset);
                    transactionList.add(tx);
                    offset += tx.getMessageSize();
                }
                block = new StoredUndoableBlock(hash, transactionList);
            } else {
                TransactionOutputChanges outChangesObject = new TransactionOutputChanges(new ByteArrayInputStream(txOutChanges));
                block = new StoredUndoableBlock(hash, outChangesObject);
            }
            StoredUndoableBlock storedUndoableBlock = block;
            return storedUndoableBlock;
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
        catch (NullPointerException e) {
            throw new BlockStoreException(e);
        }
        catch (ClassCastException e) {
            throw new BlockStoreException(e);
        }
        catch (ProtocolException e) {
            throw new BlockStoreException(e);
        }
        catch (IOException e) {
            throw new BlockStoreException(e);
        }
        finally {
            if (s != null) {
                try {
                    s.close();
                }
                catch (SQLException e) {
                    throw new BlockStoreException("Failed to close PreparedStatement");
                }
            }
        }
    }

    @Override
    public StoredBlock getChainHead() throws BlockStoreException {
        return this.chainHeadBlock;
    }

    @Override
    public void setChainHead(StoredBlock chainHead) throws BlockStoreException {
        Sha256Hash hash;
        this.chainHeadHash = hash = chainHead.getHeader().getHash();
        this.chainHeadBlock = chainHead;
        this.maybeConnect();
        try {
            PreparedStatement s = this.conn.get().prepareStatement("UPDATE settings SET value = ? WHERE name = ?");
            s.setString(2, CHAIN_HEAD_SETTING);
            s.setBytes(1, hash.getBytes());
            s.executeUpdate();
            s.close();
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
    }

    @Override
    public StoredBlock getVerifiedChainHead() throws BlockStoreException {
        return this.verifiedChainHeadBlock;
    }

    @Override
    public void setVerifiedChainHead(StoredBlock chainHead) throws BlockStoreException {
        Sha256Hash hash;
        this.verifiedChainHeadHash = hash = chainHead.getHeader().getHash();
        this.verifiedChainHeadBlock = chainHead;
        this.maybeConnect();
        try {
            PreparedStatement s = this.conn.get().prepareStatement("UPDATE settings SET value = ? WHERE name = ?");
            s.setString(2, VERIFIED_CHAIN_HEAD_SETTING);
            s.setBytes(1, hash.getBytes());
            s.executeUpdate();
            s.close();
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
        if (this.chainHeadBlock.getHeight() < chainHead.getHeight()) {
            this.setChainHead(chainHead);
        }
        this.removeUndoableBlocksWhereHeightIsLessThan(chainHead.getHeight() - this.fullStoreDepth);
    }

    private void removeUndoableBlocksWhereHeightIsLessThan(int height) throws BlockStoreException {
        try {
            PreparedStatement s = this.conn.get().prepareStatement("DELETE FROM undoableBlocks WHERE height <= ?");
            s.setInt(1, height);
            s.executeUpdate();
            s.close();
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
    }

    @Override
    @Nullable
    public StoredTransactionOutput getTransactionOutput(Sha256Hash hash, long index) throws BlockStoreException {
        this.maybeConnect();
        PreparedStatement s = null;
        try {
            s = this.conn.get().prepareStatement("SELECT height, value, scriptBytes FROM openOutputs WHERE hash = ? AND index = ?");
            s.setBytes(1, hash.getBytes());
            s.setInt(2, (int)index);
            ResultSet results = s.executeQuery();
            if (!results.next()) {
                StoredTransactionOutput storedTransactionOutput = null;
                return storedTransactionOutput;
            }
            int height = results.getInt(1);
            BigInteger value = new BigInteger(results.getBytes(2));
            StoredTransactionOutput storedTransactionOutput = new StoredTransactionOutput(hash, index, value, height, true, results.getBytes(3));
            return storedTransactionOutput;
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
        finally {
            if (s != null) {
                try {
                    s.close();
                }
                catch (SQLException e) {
                    throw new BlockStoreException("Failed to close PreparedStatement");
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void addUnspentTransactionOutput(StoredTransactionOutput out) throws BlockStoreException {
        this.maybeConnect();
        PreparedStatement s = null;
        try {
            s = this.conn.get().prepareStatement("INSERT INTO openOutputs (hash, index, height, value, scriptBytes) VALUES (?, ?, ?, ?, ?)");
            s.setBytes(1, out.getHash().getBytes());
            s.setInt(2, (int)out.getIndex());
            s.setInt(3, out.getHeight());
            s.setBytes(4, out.getValue().toByteArray());
            s.setBytes(5, out.getScriptBytes());
            s.executeUpdate();
            s.close();
        }
        catch (SQLException e) {
            if (e.getErrorCode() != 23505) {
                throw new BlockStoreException(e);
            }
        }
        finally {
            if (s != null) {
                try {
                    s.close();
                }
                catch (SQLException e) {
                    throw new BlockStoreException(e);
                }
            }
        }
    }

    @Override
    public void removeUnspentTransactionOutput(StoredTransactionOutput out) throws BlockStoreException {
        this.maybeConnect();
        try {
            PreparedStatement s = this.conn.get().prepareStatement("DELETE FROM openOutputs WHERE hash = ? AND index = ?");
            s.setBytes(1, out.getHash().getBytes());
            s.setInt(2, (int)out.getIndex());
            s.executeUpdate();
            int updateCount = s.getUpdateCount();
            s.close();
            if (updateCount == 0) {
                throw new BlockStoreException("Tried to remove a StoredTransactionOutput from H2FullPrunedBlockStore that it didn't have!");
            }
        }
        catch (SQLException e) {
            throw new BlockStoreException(e);
        }
    }

    @Override
    public void beginDatabaseBatchWrite() throws BlockStoreException {
        this.maybeConnect();
        try {
            this.conn.get().setAutoCommit(false);
        }
        catch (SQLException e) {
            throw new BlockStoreException(e);
        }
    }

    @Override
    public void commitDatabaseBatchWrite() throws BlockStoreException {
        this.maybeConnect();
        try {
            this.conn.get().commit();
            this.conn.get().setAutoCommit(true);
        }
        catch (SQLException e) {
            throw new BlockStoreException(e);
        }
    }

    @Override
    public void abortDatabaseBatchWrite() throws BlockStoreException {
        this.maybeConnect();
        try {
            this.conn.get().rollback();
            this.conn.get().setAutoCommit(true);
        }
        catch (SQLException e) {
            throw new BlockStoreException(e);
        }
    }

    @Override
    public boolean hasUnspentOutputs(Sha256Hash hash, int numOutputs) throws BlockStoreException {
        this.maybeConnect();
        PreparedStatement s = null;
        try {
            s = this.conn.get().prepareStatement("SELECT COUNT(*) FROM openOutputs WHERE hash = ?");
            s.setBytes(1, hash.getBytes());
            ResultSet results = s.executeQuery();
            if (!results.next()) {
                throw new BlockStoreException("Got no results from a COUNT(*) query");
            }
            int count = results.getInt(1);
            boolean bl = count != 0;
            return bl;
        }
        catch (SQLException ex) {
            throw new BlockStoreException(ex);
        }
        finally {
            if (s != null) {
                try {
                    s.close();
                }
                catch (SQLException e) {
                    throw new BlockStoreException("Failed to close PreparedStatement");
                }
            }
        }
    }
}

