/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.protocols.channels;

import com.google.bitcoin.core.AbstractWalletEventListener;
import com.google.bitcoin.core.ECKey;
import com.google.bitcoin.core.InsufficientMoneyException;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ScriptException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.Transaction;
import com.google.bitcoin.core.TransactionConfidence;
import com.google.bitcoin.core.TransactionInput;
import com.google.bitcoin.core.TransactionOutput;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.core.Wallet;
import com.google.bitcoin.crypto.TransactionSignature;
import com.google.bitcoin.protocols.channels.StoredClientChannel;
import com.google.bitcoin.protocols.channels.StoredPaymentChannelClientStates;
import com.google.bitcoin.protocols.channels.ValueOutOfRangeException;
import com.google.bitcoin.script.Script;
import com.google.bitcoin.script.ScriptBuilder;
import com.google.bitcoin.utils.Threading;
import com.google.bitcoin.wallet.AllowUnconfirmedCoinSelector;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.math.BigInteger;
import java.util.ArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PaymentChannelClientState {
    private static final Logger log = LoggerFactory.getLogger(PaymentChannelClientState.class);
    private static final int CONFIRMATIONS_FOR_DELETE = 3;
    private final Wallet wallet;
    private final ECKey myKey;
    private final ECKey serverMultisigKey;
    private final BigInteger totalValue;
    private final long expiryTime;
    private Transaction refundTx;
    private BigInteger refundFees;
    private Transaction multisigContract;
    private Script multisigScript;
    private BigInteger valueToMe;
    private State state;
    private StoredClientChannel storedChannel;

    PaymentChannelClientState(StoredClientChannel storedClientChannel, Wallet wallet) throws VerificationException {
        this.wallet = (Wallet)Preconditions.checkNotNull((Object)wallet);
        this.multisigContract = (Transaction)Preconditions.checkNotNull((Object)storedClientChannel.contract);
        this.multisigScript = this.multisigContract.getOutput(0).getScriptPubKey();
        this.refundTx = (Transaction)Preconditions.checkNotNull((Object)storedClientChannel.refund);
        this.refundFees = (BigInteger)Preconditions.checkNotNull((Object)storedClientChannel.refundFees);
        this.expiryTime = this.refundTx.getLockTime();
        this.myKey = (ECKey)Preconditions.checkNotNull((Object)storedClientChannel.myKey);
        this.serverMultisigKey = null;
        this.totalValue = this.multisigContract.getOutput(0).getValue();
        this.valueToMe = (BigInteger)Preconditions.checkNotNull((Object)storedClientChannel.valueToMe);
        this.storedChannel = storedClientChannel;
        this.state = State.READY;
        this.initWalletListeners();
    }

    public synchronized boolean isSettlementTransaction(Transaction tx) {
        try {
            tx.verify();
            tx.getInput(0).verify(this.multisigContract.getOutput(0));
            return true;
        }
        catch (VerificationException e) {
            return false;
        }
    }

    public PaymentChannelClientState(Wallet wallet, ECKey myKey, ECKey serverMultisigKey, BigInteger value, long expiryTimeInSeconds) throws VerificationException {
        Preconditions.checkArgument((value.compareTo(BigInteger.ZERO) > 0 ? 1 : 0) != 0);
        this.wallet = (Wallet)Preconditions.checkNotNull((Object)wallet);
        this.initWalletListeners();
        this.serverMultisigKey = (ECKey)Preconditions.checkNotNull((Object)serverMultisigKey);
        if (!myKey.isPubKeyCanonical() || !serverMultisigKey.isPubKeyCanonical()) {
            throw new VerificationException("Pubkey was not canonical (ie non-standard)");
        }
        this.myKey = (ECKey)Preconditions.checkNotNull((Object)myKey);
        this.valueToMe = this.totalValue = (BigInteger)Preconditions.checkNotNull((Object)value);
        this.expiryTime = expiryTimeInSeconds;
        this.state = State.NEW;
    }

    private synchronized void initWalletListeners() {
        if (this.storedChannel != null && this.storedChannel.close != null) {
            this.watchCloseConfirmations();
        }
        this.wallet.addEventListener(new AbstractWalletEventListener(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void onCoinsReceived(Wallet wallet, Transaction tx, BigInteger prevBalance, BigInteger newBalance) {
                PaymentChannelClientState paymentChannelClientState = PaymentChannelClientState.this;
                synchronized (paymentChannelClientState) {
                    if (PaymentChannelClientState.this.multisigContract == null) {
                        return;
                    }
                    if (PaymentChannelClientState.this.isSettlementTransaction(tx)) {
                        log.info("Close: transaction {} closed contract {}", (Object)tx.getHash(), (Object)PaymentChannelClientState.this.multisigContract.getHash());
                        PaymentChannelClientState.this.state = State.CLOSED;
                        if (PaymentChannelClientState.this.storedChannel == null) {
                            return;
                        }
                        ((PaymentChannelClientState)PaymentChannelClientState.this).storedChannel.close = tx;
                        PaymentChannelClientState.this.updateChannelInWallet();
                        PaymentChannelClientState.this.watchCloseConfirmations();
                    }
                }
            }
        }, Threading.SAME_THREAD);
    }

    private void watchCloseConfirmations() {
        TransactionConfidence confidence = this.storedChannel.close.getConfidence();
        ListenableFuture<Transaction> future = confidence.getDepthFuture(3, Threading.SAME_THREAD);
        Futures.addCallback(future, (FutureCallback)new FutureCallback<Transaction>(){

            public void onSuccess(Transaction result) {
                PaymentChannelClientState.this.deleteChannelFromWallet();
            }

            public void onFailure(Throwable t) {
                Throwables.propagate((Throwable)t);
            }
        });
    }

    private synchronized void deleteChannelFromWallet() {
        log.info("Close tx has confirmed, deleting channel from wallet: {}", (Object)this.storedChannel);
        StoredPaymentChannelClientStates channels = (StoredPaymentChannelClientStates)this.wallet.getExtensions().get(StoredPaymentChannelClientStates.EXTENSION_ID);
        channels.removeChannel(this.storedChannel);
        this.wallet.addOrUpdateExtension(channels);
        this.storedChannel = null;
    }

    public synchronized State getState() {
        return this.state;
    }

    public synchronized void initiate() throws ValueOutOfRangeException, InsufficientMoneyException {
        NetworkParameters params = this.wallet.getParams();
        Transaction template = new Transaction(params);
        ArrayList keys = Lists.newArrayList((Object[])new ECKey[]{this.myKey, this.serverMultisigKey});
        TransactionOutput multisigOutput = template.addOutput(this.totalValue, ScriptBuilder.createMultiSigOutputScript(2, keys));
        if (multisigOutput.getMinNonDustValue().compareTo(this.totalValue) > 0) {
            throw new ValueOutOfRangeException("totalValue too small to use");
        }
        Wallet.SendRequest req = Wallet.SendRequest.forTx(template);
        req.coinSelector = AllowUnconfirmedCoinSelector.get();
        this.editContractSendRequest(req);
        this.wallet.completeTx(req);
        BigInteger multisigFee = req.fee;
        this.multisigContract = req.tx;
        this.refundTx = new Transaction(params);
        this.refundTx.addInput(multisigOutput).setSequenceNumber(0L);
        this.refundTx.setLockTime(this.expiryTime);
        if (this.totalValue.compareTo(Utils.CENT) < 0) {
            BigInteger valueAfterFee = this.totalValue.subtract(Transaction.REFERENCE_DEFAULT_MIN_TX_FEE);
            if (Transaction.MIN_NONDUST_OUTPUT.compareTo(valueAfterFee) > 0) {
                throw new ValueOutOfRangeException("totalValue too small to use");
            }
            this.refundTx.addOutput(valueAfterFee, this.myKey.toAddress(params));
            this.refundFees = multisigFee.add(Transaction.REFERENCE_DEFAULT_MIN_TX_FEE);
        } else {
            this.refundTx.addOutput(this.totalValue, this.myKey.toAddress(params));
            this.refundFees = multisigFee;
        }
        this.refundTx.getConfidence().setSource(TransactionConfidence.Source.SELF);
        log.info("initiated channel with multi-sig contract {}, refund {}", (Object)this.multisigContract.getHashAsString(), (Object)this.refundTx.getHashAsString());
        this.state = State.INITIATED;
    }

    protected void editContractSendRequest(Wallet.SendRequest req) {
    }

    public synchronized Transaction getMultisigContract() {
        Preconditions.checkState((this.multisigContract != null ? 1 : 0) != 0);
        if (this.state == State.PROVIDE_MULTISIG_CONTRACT_TO_SERVER) {
            this.state = State.READY;
        }
        return this.multisigContract;
    }

    public synchronized Transaction getIncompleteRefundTransaction() {
        Preconditions.checkState((this.refundTx != null ? 1 : 0) != 0);
        if (this.state == State.INITIATED) {
            this.state = State.WAITING_FOR_SIGNED_REFUND;
        }
        return this.refundTx;
    }

    public synchronized void provideRefundSignature(byte[] theirSignature) throws VerificationException {
        Preconditions.checkNotNull((Object)theirSignature);
        Preconditions.checkState((this.state == State.WAITING_FOR_SIGNED_REFUND ? 1 : 0) != 0);
        TransactionSignature theirSig = TransactionSignature.decodeFromBitcoin(theirSignature, true);
        if (theirSig.sigHashMode() != Transaction.SigHash.NONE || !theirSig.anyoneCanPay()) {
            throw new VerificationException("Refund signature was not SIGHASH_NONE|SIGHASH_ANYONECANPAY");
        }
        TransactionOutput multisigContractOutput = this.multisigContract.getOutput(0);
        try {
            this.multisigScript = multisigContractOutput.getScriptPubKey();
        }
        catch (ScriptException e) {
            throw new RuntimeException(e);
        }
        TransactionSignature ourSignature = this.refundTx.calculateSignature(0, this.myKey, this.multisigScript, Transaction.SigHash.ALL, false);
        Script scriptSig = ScriptBuilder.createMultiSigInputScript(ourSignature, theirSig);
        log.info("Refund scriptSig: {}", (Object)scriptSig);
        log.info("Multi-sig contract scriptPubKey: {}", (Object)this.multisigScript);
        TransactionInput refundInput = this.refundTx.getInput(0);
        refundInput.setScriptSig(scriptSig);
        refundInput.verify(multisigContractOutput);
        this.state = State.SAVE_STATE_IN_WALLET;
    }

    private synchronized Transaction makeUnsignedChannelContract(BigInteger valueToMe) throws ValueOutOfRangeException {
        Transaction tx = new Transaction(this.wallet.getParams());
        tx.addInput(this.multisigContract.getOutput(0));
        tx.addOutput(valueToMe, this.myKey.toAddress(this.wallet.getParams()));
        return tx;
    }

    public synchronized void checkNotExpired() {
        if (Utils.currentTimeMillis() / 1000L > this.expiryTime) {
            this.state = State.EXPIRED;
            this.disconnectFromChannel();
            throw new IllegalStateException("Channel expired");
        }
    }

    public synchronized IncrementedPayment incrementPaymentBy(BigInteger size) throws ValueOutOfRangeException {
        Preconditions.checkState((this.state == State.READY ? 1 : 0) != 0);
        this.checkNotExpired();
        Preconditions.checkNotNull((Object)size);
        if (size.compareTo(BigInteger.ZERO) < 0) {
            throw new ValueOutOfRangeException("Tried to decrement payment");
        }
        BigInteger newValueToMe = this.valueToMe.subtract(size);
        if (newValueToMe.compareTo(Transaction.MIN_NONDUST_OUTPUT) < 0 && newValueToMe.compareTo(BigInteger.ZERO) > 0) {
            log.info("New value being sent back as change was smaller than minimum nondust output, sending all");
            size = this.valueToMe;
            newValueToMe = BigInteger.ZERO;
        }
        if (newValueToMe.compareTo(BigInteger.ZERO) < 0) {
            throw new ValueOutOfRangeException("Channel has too little money to pay " + size + " satoshis");
        }
        Transaction tx = this.makeUnsignedChannelContract(newValueToMe);
        log.info("Signing new payment tx {}", (Object)tx);
        Transaction.SigHash mode = newValueToMe.equals(BigInteger.ZERO) ? Transaction.SigHash.NONE : Transaction.SigHash.SINGLE;
        TransactionSignature sig = tx.calculateSignature(0, this.myKey, this.multisigScript, mode, true);
        this.valueToMe = newValueToMe;
        this.updateChannelInWallet();
        IncrementedPayment payment = new IncrementedPayment();
        payment.signature = sig;
        payment.amount = size;
        return payment;
    }

    private synchronized void updateChannelInWallet() {
        if (this.storedChannel == null) {
            return;
        }
        this.storedChannel.valueToMe = this.valueToMe;
        StoredPaymentChannelClientStates channels = (StoredPaymentChannelClientStates)this.wallet.getExtensions().get(StoredPaymentChannelClientStates.EXTENSION_ID);
        this.wallet.addOrUpdateExtension(channels);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void disconnectFromChannel() {
        if (this.storedChannel == null) {
            return;
        }
        StoredClientChannel storedClientChannel = this.storedChannel;
        synchronized (storedClientChannel) {
            this.storedChannel.active = false;
        }
    }

    @VisibleForTesting
    synchronized void fakeSave() {
        try {
            this.wallet.commitTx(this.multisigContract);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
        this.state = State.PROVIDE_MULTISIG_CONTRACT_TO_SERVER;
    }

    @VisibleForTesting
    synchronized void doStoreChannelInWallet(Sha256Hash id) {
        StoredPaymentChannelClientStates channels = (StoredPaymentChannelClientStates)this.wallet.getExtensions().get(StoredPaymentChannelClientStates.EXTENSION_ID);
        Preconditions.checkNotNull((Object)channels, (Object)"You have not added the StoredPaymentChannelClientStates extension to the wallet.");
        Preconditions.checkState((channels.getChannel(id, this.multisigContract.getHash()) == null ? 1 : 0) != 0);
        this.storedChannel = new StoredClientChannel(id, this.multisigContract, this.refundTx, this.myKey, this.valueToMe, this.refundFees, true);
        channels.putChannel(this.storedChannel);
        this.wallet.addOrUpdateExtension(channels);
    }

    public synchronized void storeChannelInWallet(Sha256Hash id) {
        Preconditions.checkState((this.state == State.SAVE_STATE_IN_WALLET && id != null ? 1 : 0) != 0);
        if (this.storedChannel != null) {
            Preconditions.checkState((boolean)this.storedChannel.id.equals(id));
            return;
        }
        this.doStoreChannelInWallet(id);
        try {
            this.wallet.commitTx(this.multisigContract);
        }
        catch (VerificationException e) {
            throw new RuntimeException(e);
        }
        this.state = State.PROVIDE_MULTISIG_CONTRACT_TO_SERVER;
    }

    public synchronized BigInteger getRefundTxFees() {
        Preconditions.checkState((this.state.compareTo(State.NEW) > 0 ? 1 : 0) != 0);
        return this.refundFees;
    }

    public synchronized Transaction getCompletedRefundTransaction() {
        Preconditions.checkState((this.state.compareTo(State.WAITING_FOR_SIGNED_REFUND) > 0 ? 1 : 0) != 0);
        return this.refundTx;
    }

    public BigInteger getTotalValue() {
        return this.totalValue;
    }

    public synchronized BigInteger getValueRefunded() {
        Preconditions.checkState((this.state == State.READY ? 1 : 0) != 0);
        return this.valueToMe;
    }

    public synchronized BigInteger getValueSpent() {
        return this.getTotalValue().subtract(this.getValueRefunded());
    }

    public static class IncrementedPayment {
        public TransactionSignature signature;
        public BigInteger amount;
    }

    public static enum State {
        NEW,
        INITIATED,
        WAITING_FOR_SIGNED_REFUND,
        SAVE_STATE_IN_WALLET,
        PROVIDE_MULTISIG_CONTRACT_TO_SERVER,
        READY,
        EXPIRED,
        CLOSED;

    }
}

