/*
 * Decompiled with CFR 0.152.
 */
package com.google.bitcoin.core;

import com.google.bitcoin.core.AbstractBlockChain;
import com.google.bitcoin.core.Address;
import com.google.bitcoin.core.ChildMessage;
import com.google.bitcoin.core.ECKey;
import com.google.bitcoin.core.Message;
import com.google.bitcoin.core.NetworkParameters;
import com.google.bitcoin.core.ProtocolException;
import com.google.bitcoin.core.ScriptException;
import com.google.bitcoin.core.Sha256Hash;
import com.google.bitcoin.core.StoredBlock;
import com.google.bitcoin.core.TransactionConfidence;
import com.google.bitcoin.core.TransactionInput;
import com.google.bitcoin.core.TransactionOutPoint;
import com.google.bitcoin.core.TransactionOutput;
import com.google.bitcoin.core.UnsafeByteArrayOutputStream;
import com.google.bitcoin.core.Utils;
import com.google.bitcoin.core.VarInt;
import com.google.bitcoin.core.VerificationException;
import com.google.bitcoin.core.Wallet;
import com.google.bitcoin.crypto.TransactionSignature;
import com.google.bitcoin.script.Script;
import com.google.bitcoin.script.ScriptBuilder;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.math.BigInteger;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.crypto.params.KeyParameter;

public class Transaction
extends ChildMessage
implements Serializable {
    private static final Logger log = LoggerFactory.getLogger(Transaction.class);
    private static final long serialVersionUID = -8567546957352643140L;
    public static final int LOCKTIME_THRESHOLD = 500000000;
    public static final int MAX_STANDARD_TX_SIZE = 102400;
    public static final BigInteger REFERENCE_DEFAULT_MIN_TX_FEE = BigInteger.valueOf(10000L);
    public static final BigInteger MIN_NONDUST_OUTPUT = BigInteger.valueOf(5460L);
    private long version;
    private ArrayList<TransactionInput> inputs;
    private ArrayList<TransactionOutput> outputs;
    private long lockTime;
    private Date updatedAt;
    private transient Sha256Hash hash;
    private TransactionConfidence confidence;
    private Map<Sha256Hash, Integer> appearsInHashes;
    private transient int optimalEncodingMessageSize;
    private Purpose purpose = Purpose.UNKNOWN;
    public static final byte SIGHASH_ANYONECANPAY_VALUE = -128;

    public Transaction(NetworkParameters params) {
        super(params);
        this.version = 1L;
        this.inputs = new ArrayList();
        this.outputs = new ArrayList();
        this.length = 8;
    }

    public Transaction(NetworkParameters params, int version, Sha256Hash hash) {
        super(params);
        this.version = (long)version & 0xFFFFFFFFL;
        this.inputs = new ArrayList();
        this.outputs = new ArrayList();
        this.hash = hash;
        this.length = 8;
    }

    public Transaction(NetworkParameters params, byte[] payloadBytes) throws ProtocolException {
        super(params, payloadBytes, 0);
    }

    public Transaction(NetworkParameters params, byte[] payload, int offset) throws ProtocolException {
        super(params, payload, offset);
    }

    public Transaction(NetworkParameters params, byte[] msg, int offset, @Nullable Message parent, boolean parseLazy, boolean parseRetain, int length) throws ProtocolException {
        super(params, msg, offset, parent, parseLazy, parseRetain, length);
    }

    public Transaction(NetworkParameters params, byte[] msg, @Nullable Message parent, boolean parseLazy, boolean parseRetain, int length) throws ProtocolException {
        super(params, msg, 0, parent, parseLazy, parseRetain, length);
    }

    @Override
    public Sha256Hash getHash() {
        if (this.hash == null) {
            byte[] bits = this.bitcoinSerialize();
            this.hash = new Sha256Hash(Utils.reverseBytes(Utils.doubleDigest(bits)));
        }
        return this.hash;
    }

    void setHash(Sha256Hash hash) {
        this.hash = hash;
    }

    public String getHashAsString() {
        return this.getHash().toString();
    }

    BigInteger getValueSentToMe(Wallet wallet, boolean includeSpent) {
        this.maybeParse();
        BigInteger v = BigInteger.ZERO;
        for (TransactionOutput o : this.outputs) {
            if (!o.isMineOrWatched(wallet) || !includeSpent && !o.isAvailableForSpending()) continue;
            v = v.add(o.getValue());
        }
        return v;
    }

    boolean isConsistent(Wallet wallet, boolean isSpent) {
        boolean isActuallySpent = true;
        for (TransactionOutput o : this.outputs) {
            if (o.isAvailableForSpending()) {
                if (o.isMineOrWatched(wallet)) {
                    isActuallySpent = false;
                }
                if (o.getSpentBy() == null) continue;
                log.error("isAvailableForSpending != spentBy");
                return false;
            }
            if (o.getSpentBy() != null) continue;
            log.error("isAvailableForSpending != spentBy");
            return false;
        }
        return isActuallySpent == isSpent;
    }

    public BigInteger getValueSentToMe(Wallet wallet) {
        return this.getValueSentToMe(wallet, true);
    }

    @Nullable
    public Map<Sha256Hash, Integer> getAppearsInHashes() {
        return this.appearsInHashes != null ? ImmutableMap.copyOf(this.appearsInHashes) : null;
    }

    public boolean isPending() {
        return this.getConfidence().getConfidenceType() == TransactionConfidence.ConfidenceType.PENDING;
    }

    public void setBlockAppearance(StoredBlock block, boolean bestChain, int relativityOffset) {
        long blockTime = block.getHeader().getTimeSeconds() * 1000L;
        if (bestChain && (this.updatedAt == null || this.updatedAt.getTime() == 0L || this.updatedAt.getTime() > blockTime)) {
            this.updatedAt = new Date(blockTime);
        }
        this.addBlockAppearance(block.getHeader().getHash(), relativityOffset);
        if (bestChain) {
            TransactionConfidence transactionConfidence = this.getConfidence();
            try {
                transactionConfidence.setWorkDone(block.getHeader().getWork());
            }
            catch (VerificationException e) {
                throw new RuntimeException(e);
            }
            transactionConfidence.setAppearedAtChainHeight(block.getHeight());
        }
    }

    public void addBlockAppearance(Sha256Hash blockHash, int relativityOffset) {
        if (this.appearsInHashes == null) {
            this.appearsInHashes = new TreeMap<Sha256Hash, Integer>();
        }
        this.appearsInHashes.put(blockHash, relativityOffset);
    }

    public BigInteger getValueSentFromMe(Wallet wallet) throws ScriptException {
        this.maybeParse();
        BigInteger v = BigInteger.ZERO;
        for (TransactionInput input : this.inputs) {
            TransactionOutput connected = input.getConnectedOutput(wallet.unspent);
            if (connected == null) {
                connected = input.getConnectedOutput(wallet.spent);
            }
            if (connected == null) {
                connected = input.getConnectedOutput(wallet.pending);
            }
            if (connected == null || !connected.isMineOrWatched(wallet)) continue;
            v = v.add(connected.getValue());
        }
        return v;
    }

    public BigInteger getValue(Wallet wallet) throws ScriptException {
        return this.getValueSentToMe(wallet).subtract(this.getValueSentFromMe(wallet));
    }

    boolean disconnectInputs() {
        boolean disconnected = false;
        this.maybeParse();
        for (TransactionInput input : this.inputs) {
            disconnected |= input.disconnect();
        }
        return disconnected;
    }

    public boolean isEveryOutputSpent() {
        this.maybeParse();
        for (TransactionOutput output : this.outputs) {
            if (!output.isAvailableForSpending()) continue;
            return false;
        }
        return true;
    }

    public boolean isAnyOutputSpent() {
        this.maybeParse();
        for (TransactionOutput output : this.outputs) {
            if (output.isAvailableForSpending()) continue;
            return true;
        }
        return false;
    }

    public boolean isEveryOwnedOutputSpent(Wallet wallet) {
        this.maybeParse();
        for (TransactionOutput output : this.outputs) {
            if (!output.isAvailableForSpending() || !output.isMineOrWatched(wallet)) continue;
            return false;
        }
        return true;
    }

    public Date getUpdateTime() {
        if (this.updatedAt == null) {
            this.updatedAt = new Date(0L);
        }
        return this.updatedAt;
    }

    public void setUpdateTime(Date updatedAt) {
        this.updatedAt = updatedAt;
    }

    @Override
    protected void unCache() {
        super.unCache();
        this.hash = null;
    }

    @Override
    protected void parseLite() throws ProtocolException {
        if (this.parseLazy && this.length == Integer.MIN_VALUE) {
            this.length = Transaction.calcLength(this.bytes, this.offset);
            this.cursor = this.offset + this.length;
        }
    }

    protected static int calcLength(byte[] buf, int offset) {
        long scriptLen;
        int cursor = offset + 4;
        VarInt varint = new VarInt(buf, cursor);
        long txInCount = varint.value;
        cursor += varint.getOriginalSizeInBytes();
        int i = 0;
        while ((long)i < txInCount) {
            varint = new VarInt(buf, cursor += 36);
            scriptLen = varint.value;
            cursor = (int)((long)cursor + (scriptLen + 4L + (long)varint.getOriginalSizeInBytes()));
            ++i;
        }
        varint = new VarInt(buf, cursor);
        long txOutCount = varint.value;
        cursor += varint.getOriginalSizeInBytes();
        i = 0;
        while ((long)i < txOutCount) {
            varint = new VarInt(buf, cursor += 8);
            scriptLen = varint.value;
            cursor = (int)((long)cursor + (scriptLen + (long)varint.getOriginalSizeInBytes()));
            ++i;
        }
        return cursor - offset + 4;
    }

    @Override
    void parse() throws ProtocolException {
        if (this.parsed) {
            return;
        }
        this.cursor = this.offset;
        this.version = this.readUint32();
        this.optimalEncodingMessageSize = 4;
        long numInputs = this.readVarInt();
        this.optimalEncodingMessageSize += VarInt.sizeOf(numInputs);
        this.inputs = new ArrayList((int)numInputs);
        for (long i = 0L; i < numInputs; ++i) {
            TransactionInput input = new TransactionInput(this.params, this, this.bytes, this.cursor, this.parseLazy, this.parseRetain);
            this.inputs.add(input);
            long scriptLen = this.readVarInt(36);
            this.optimalEncodingMessageSize = (int)((long)this.optimalEncodingMessageSize + ((long)(36 + VarInt.sizeOf(scriptLen)) + scriptLen + 4L));
            this.cursor = (int)((long)this.cursor + (scriptLen + 4L));
        }
        long numOutputs = this.readVarInt();
        this.optimalEncodingMessageSize += VarInt.sizeOf(numOutputs);
        this.outputs = new ArrayList((int)numOutputs);
        for (long i = 0L; i < numOutputs; ++i) {
            TransactionOutput output = new TransactionOutput(this.params, this, this.bytes, this.cursor, this.parseLazy, this.parseRetain);
            this.outputs.add(output);
            long scriptLen = this.readVarInt(8);
            this.optimalEncodingMessageSize = (int)((long)this.optimalEncodingMessageSize + ((long)(8 + VarInt.sizeOf(scriptLen)) + scriptLen));
            this.cursor = (int)((long)this.cursor + scriptLen);
        }
        this.lockTime = this.readUint32();
        this.optimalEncodingMessageSize += 4;
        this.length = this.cursor - this.offset;
    }

    public int getOptimalEncodingMessageSize() {
        if (this.optimalEncodingMessageSize != 0) {
            return this.optimalEncodingMessageSize;
        }
        this.maybeParse();
        if (this.optimalEncodingMessageSize != 0) {
            return this.optimalEncodingMessageSize;
        }
        this.optimalEncodingMessageSize = this.getMessageSize();
        return this.optimalEncodingMessageSize;
    }

    public boolean isCoinBase() {
        this.maybeParse();
        return this.inputs.size() == 1 && this.inputs.get(0).isCoinBase();
    }

    public boolean isMature() {
        if (!this.isCoinBase()) {
            return true;
        }
        if (this.getConfidence().getConfidenceType() != TransactionConfidence.ConfidenceType.BUILDING) {
            return false;
        }
        return this.getConfidence().getDepthInBlocks() >= this.params.getSpendableCoinbaseDepth();
    }

    public String toString() {
        return this.toString(null);
    }

    public String toString(@Nullable AbstractBlockChain chain) {
        StringBuilder s = new StringBuilder();
        s.append(String.format("  %s: %s%n", this.getHashAsString(), this.getConfidence()));
        if (this.isTimeLocked()) {
            String time;
            if (this.lockTime < 500000000L) {
                time = "block " + this.lockTime;
                if (chain != null) {
                    time = time + " (estimated to be reached at " + chain.estimateBlockTime((int)this.lockTime).toString() + ")";
                }
            } else {
                time = new Date(this.lockTime * 1000L).toString();
            }
            s.append(String.format("  time locked until %s%n", time));
        }
        if (this.inputs.size() == 0) {
            s.append(String.format("  INCOMPLETE: No inputs!%n", new Object[0]));
            return s.toString();
        }
        if (this.isCoinBase()) {
            String script2;
            String script;
            try {
                script = this.inputs.get(0).getScriptSig().toString();
                script2 = this.outputs.get(0).getScriptPubKey().toString();
            }
            catch (ScriptException e) {
                script = "???";
                script2 = "???";
            }
            s.append("     == COINBASE TXN (scriptSig " + script + ")  (scriptPubKey " + script2 + ")\n");
            return s.toString();
        }
        for (TransactionInput in : this.inputs) {
            s.append("     ");
            s.append("in   ");
            try {
                Script scriptSig = in.getScriptSig();
                s.append(scriptSig);
                s.append("\n          ");
                s.append("outpoint:");
                TransactionOutPoint outpoint = in.getOutpoint();
                s.append(outpoint.toString());
                TransactionOutput connectedOutput = outpoint.getConnectedOutput();
                if (connectedOutput != null) {
                    s.append(" hash160:");
                    s.append(Utils.bytesToHexString(connectedOutput.getScriptPubKey().getPubKeyHash()));
                }
            }
            catch (Exception e) {
                s.append("[exception: ").append(e.getMessage()).append("]");
            }
            s.append(String.format("%n", new Object[0]));
        }
        for (TransactionOutput out : this.outputs) {
            s.append("     ");
            s.append("out  ");
            try {
                Script scriptPubKey = out.getScriptPubKey();
                s.append(scriptPubKey);
                s.append(" ");
                s.append(Utils.bitcoinValueToFriendlyString(out.getValue()));
                s.append(" BTC");
                if (!out.isAvailableForSpending()) {
                    s.append(" Spent");
                }
                if (out.getSpentBy() != null) {
                    s.append(" by ");
                    s.append(out.getSpentBy().getParentTransaction().getHashAsString());
                }
            }
            catch (Exception e) {
                s.append("[exception: ").append(e.getMessage()).append("]");
            }
            s.append(String.format("%n", new Object[0]));
        }
        return s.toString();
    }

    public void clearInputs() {
        this.unCache();
        for (TransactionInput input : this.inputs) {
            input.setParent(null);
        }
        this.inputs.clear();
        this.length = this.bitcoinSerialize().length;
    }

    public TransactionInput addInput(TransactionOutput from) {
        return this.addInput(new TransactionInput(this.params, this, from));
    }

    public TransactionInput addInput(TransactionInput input) {
        this.unCache();
        input.setParent(this);
        this.inputs.add(input);
        this.adjustLength(this.inputs.size(), input.length);
        return input;
    }

    public TransactionInput addSignedInput(TransactionOutPoint prevOut, Script scriptPubKey, ECKey sigKey, SigHash sigHash, boolean anyoneCanPay) throws ScriptException {
        TransactionInput input = new TransactionInput(this.params, this, new byte[0], prevOut);
        this.addInput(input);
        Sha256Hash hash = this.hashForSignature(this.inputs.size() - 1, scriptPubKey, sigHash, anyoneCanPay);
        ECKey.ECDSASignature ecSig = sigKey.sign(hash);
        TransactionSignature txSig = new TransactionSignature(ecSig, sigHash, anyoneCanPay);
        if (scriptPubKey.isSentToRawPubKey()) {
            input.setScriptSig(ScriptBuilder.createInputScript(txSig));
        } else if (scriptPubKey.isSentToAddress()) {
            input.setScriptSig(ScriptBuilder.createInputScript(txSig, sigKey));
        } else {
            throw new ScriptException("Don't know how to sign for this kind of scriptPubKey: " + scriptPubKey);
        }
        return input;
    }

    public TransactionInput addSignedInput(TransactionOutPoint prevOut, Script scriptPubKey, ECKey sigKey) throws ScriptException {
        return this.addSignedInput(prevOut, scriptPubKey, sigKey, SigHash.ALL, false);
    }

    public void clearOutputs() {
        this.unCache();
        for (TransactionOutput output : this.outputs) {
            output.setParent(null);
        }
        this.outputs.clear();
        this.length = this.bitcoinSerialize().length;
    }

    public TransactionOutput addOutput(TransactionOutput to) {
        this.unCache();
        to.setParent(this);
        this.outputs.add(to);
        this.adjustLength(this.outputs.size(), to.length);
        return to;
    }

    public TransactionOutput addOutput(BigInteger value, Address address) {
        return this.addOutput(new TransactionOutput(this.params, this, value, address));
    }

    public TransactionOutput addOutput(BigInteger value, ECKey pubkey) {
        return this.addOutput(new TransactionOutput(this.params, this, value, pubkey));
    }

    public TransactionOutput addOutput(BigInteger value, Script script) {
        return this.addOutput(new TransactionOutput(this.params, this, value, script.getProgram()));
    }

    public synchronized void signInputs(SigHash hashType, Wallet wallet) throws ScriptException {
        this.signInputs(hashType, wallet, null);
    }

    public synchronized void signInputs(SigHash hashType, Wallet wallet, @Nullable KeyParameter aesKey) throws ScriptException {
        TransactionInput input;
        int i;
        Preconditions.checkState((this.inputs.size() > 0 ? 1 : 0) != 0);
        Preconditions.checkState((this.outputs.size() > 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((hashType == SigHash.ALL ? 1 : 0) != 0, (Object)"Only SIGHASH_ALL is currently supported");
        TransactionSignature[] signatures = new TransactionSignature[this.inputs.size()];
        ECKey[] signingKeys = new ECKey[this.inputs.size()];
        for (i = 0; i < this.inputs.size(); ++i) {
            input = this.inputs.get(i);
            if (input.getOutpoint().getConnectedOutput() == null) {
                log.warn("Missing connected output, assuming input {} is already signed.", (Object)i);
                continue;
            }
            try {
                input.getScriptSig().correctlySpends(this, i, input.getOutpoint().getConnectedOutput().getScriptPubKey(), true);
                log.warn("Input {} already correctly spends output, assuming SIGHASH type used will be safe and skipping signing.", (Object)i);
                continue;
            }
            catch (ScriptException e) {
                if (input.getScriptBytes().length != 0) {
                    log.warn("Re-signing an already signed transaction! Be sure this is what you want.");
                }
                ECKey key = input.getOutpoint().getConnectedKey(wallet);
                Preconditions.checkNotNull((Object)key, (String)"Transaction exists in wallet that we cannot redeem: %s", (Object[])new Object[]{input.getOutpoint().getHash()});
                signingKeys[i] = key;
                boolean anyoneCanPay = false;
                byte[] connectedPubKeyScript = input.getOutpoint().getConnectedPubKeyScript();
                signatures[i] = key.hasPrivKey() || key.isEncrypted() ? this.calculateSignature(i, key, aesKey, connectedPubKeyScript, hashType, anyoneCanPay) : TransactionSignature.dummy();
            }
        }
        for (i = 0; i < this.inputs.size(); ++i) {
            if (signatures[i] == null) continue;
            input = this.inputs.get(i);
            TransactionOutput connectedOutput = input.getOutpoint().getConnectedOutput();
            Preconditions.checkNotNull((Object)connectedOutput);
            Script scriptPubKey = connectedOutput.getScriptPubKey();
            if (scriptPubKey.isSentToAddress()) {
                input.setScriptSig(ScriptBuilder.createInputScript(signatures[i], signingKeys[i]));
                continue;
            }
            if (scriptPubKey.isSentToRawPubKey()) {
                input.setScriptSig(ScriptBuilder.createInputScript(signatures[i]));
                continue;
            }
            throw new RuntimeException("Do not understand script type: " + scriptPubKey);
        }
    }

    public synchronized TransactionSignature calculateSignature(int inputIndex, ECKey key, @Nullable KeyParameter aesKey, byte[] connectedPubKeyScript, SigHash hashType, boolean anyoneCanPay) {
        Sha256Hash hash = this.hashForSignature(inputIndex, connectedPubKeyScript, hashType, anyoneCanPay);
        return new TransactionSignature(key.sign(hash, aesKey), hashType, anyoneCanPay);
    }

    public synchronized TransactionSignature calculateSignature(int inputIndex, ECKey key, Script connectedPubKeyScript, SigHash hashType, boolean anyoneCanPay) {
        Sha256Hash hash = this.hashForSignature(inputIndex, connectedPubKeyScript.getProgram(), hashType, anyoneCanPay);
        return new TransactionSignature(key.sign(hash), hashType, anyoneCanPay);
    }

    public synchronized Sha256Hash hashForSignature(int inputIndex, byte[] connectedScript, SigHash type, boolean anyoneCanPay) {
        byte sigHashType = (byte)TransactionSignature.calcSigHashValue(type, anyoneCanPay);
        return this.hashForSignature(inputIndex, connectedScript, sigHashType);
    }

    public synchronized Sha256Hash hashForSignature(int inputIndex, Script connectedScript, SigHash type, boolean anyoneCanPay) {
        int sigHash = TransactionSignature.calcSigHashValue(type, anyoneCanPay);
        return this.hashForSignature(inputIndex, connectedScript.getProgram(), (byte)sigHash);
    }

    public synchronized Sha256Hash hashForSignature(int inputIndex, byte[] connectedScript, byte sigHashType) {
        try {
            int i;
            byte[][] inputScripts = new byte[this.inputs.size()][];
            long[] inputSequenceNumbers = new long[this.inputs.size()];
            for (int i2 = 0; i2 < this.inputs.size(); ++i2) {
                inputScripts[i2] = this.inputs.get(i2).getScriptBytes();
                inputSequenceNumbers[i2] = this.inputs.get(i2).getSequenceNumber();
                this.inputs.get(i2).setScriptBytes(TransactionInput.EMPTY_ARRAY);
            }
            connectedScript = Script.removeAllInstancesOfOp(connectedScript, 171);
            TransactionInput input = this.inputs.get(inputIndex);
            input.setScriptBytes(connectedScript);
            ArrayList<TransactionOutput> outputs = this.outputs;
            if ((sigHashType & 0x1F) == SigHash.NONE.ordinal() + 1) {
                this.outputs = new ArrayList(0);
                for (i = 0; i < this.inputs.size(); ++i) {
                    if (i == inputIndex) continue;
                    this.inputs.get(i).setSequenceNumber(0L);
                }
            } else if ((sigHashType & 0x1F) == SigHash.SINGLE.ordinal() + 1) {
                if (inputIndex >= this.outputs.size()) {
                    for (int i3 = 0; i3 < this.inputs.size(); ++i3) {
                        this.inputs.get(i3).setScriptBytes(inputScripts[i3]);
                        this.inputs.get(i3).setSequenceNumber(inputSequenceNumbers[i3]);
                    }
                    this.outputs = outputs;
                    return new Sha256Hash("0100000000000000000000000000000000000000000000000000000000000000");
                }
                this.outputs = new ArrayList<TransactionOutput>(this.outputs.subList(0, inputIndex + 1));
                for (i = 0; i < inputIndex; ++i) {
                    this.outputs.set(i, new TransactionOutput(this.params, this, Utils.NEGATIVE_ONE, new byte[0]));
                }
                for (i = 0; i < this.inputs.size(); ++i) {
                    if (i == inputIndex) continue;
                    this.inputs.get(i).setSequenceNumber(0L);
                }
            }
            ArrayList<TransactionInput> inputs = this.inputs;
            if ((sigHashType & 0xFFFFFF80) == -128) {
                this.inputs = new ArrayList();
                this.inputs.add(input);
            }
            UnsafeByteArrayOutputStream bos = new UnsafeByteArrayOutputStream(this.length == Integer.MIN_VALUE ? 256 : this.length + 4);
            this.bitcoinSerialize(bos);
            Utils.uint32ToByteStreamLE(0xFF & sigHashType, bos);
            Sha256Hash hash = new Sha256Hash(Utils.doubleDigest(((ByteArrayOutputStream)bos).toByteArray()));
            bos.close();
            this.inputs = inputs;
            for (int i4 = 0; i4 < inputs.size(); ++i4) {
                inputs.get(i4).setScriptBytes(inputScripts[i4]);
                inputs.get(i4).setSequenceNumber(inputSequenceNumbers[i4]);
            }
            this.outputs = outputs;
            return hash;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    protected void bitcoinSerializeToStream(OutputStream stream) throws IOException {
        Utils.uint32ToByteStreamLE(this.version, stream);
        stream.write(new VarInt(this.inputs.size()).encode());
        for (TransactionInput in : this.inputs) {
            in.bitcoinSerialize(stream);
        }
        stream.write(new VarInt(this.outputs.size()).encode());
        for (TransactionOutput out : this.outputs) {
            out.bitcoinSerialize(stream);
        }
        Utils.uint32ToByteStreamLE(this.lockTime, stream);
    }

    public long getLockTime() {
        this.maybeParse();
        return this.lockTime;
    }

    public void setLockTime(long lockTime) {
        this.unCache();
        this.lockTime = lockTime;
    }

    public long getVersion() {
        this.maybeParse();
        return this.version;
    }

    public List<TransactionInput> getInputs() {
        this.maybeParse();
        return Collections.unmodifiableList(this.inputs);
    }

    public List<TransactionOutput> getOutputs() {
        this.maybeParse();
        return Collections.unmodifiableList(this.outputs);
    }

    public TransactionInput getInput(int index) {
        this.maybeParse();
        return this.inputs.get(index);
    }

    public TransactionOutput getOutput(int index) {
        this.maybeParse();
        return this.outputs.get(index);
    }

    public synchronized TransactionConfidence getConfidence() {
        if (this.confidence == null) {
            this.confidence = new TransactionConfidence(this);
        }
        return this.confidence;
    }

    public synchronized boolean hasConfidence() {
        return this.confidence != null && this.confidence.getConfidenceType() != TransactionConfidence.ConfidenceType.UNKNOWN;
    }

    public boolean equals(Object other) {
        if (!(other instanceof Transaction)) {
            return false;
        }
        Transaction t = (Transaction)other;
        return t.getHash().equals(this.getHash());
    }

    public int hashCode() {
        return this.getHash().hashCode();
    }

    private void writeObject(ObjectOutputStream out) throws IOException {
        this.maybeParse();
        out.defaultWriteObject();
    }

    public int getSigOpCount() throws ScriptException {
        this.maybeParse();
        int sigOps = 0;
        for (TransactionInput input : this.inputs) {
            sigOps += Script.getSigOpCount(input.getScriptBytes());
        }
        for (TransactionOutput output : this.outputs) {
            sigOps += Script.getSigOpCount(output.getScriptBytes());
        }
        return sigOps;
    }

    public void verify() throws VerificationException {
        this.maybeParse();
        if (this.inputs.size() == 0 || this.outputs.size() == 0) {
            throw new VerificationException("Transaction had no inputs or no outputs.");
        }
        if (this.getMessageSize() > 1000000) {
            throw new VerificationException("Transaction larger than MAX_BLOCK_SIZE");
        }
        BigInteger valueOut = BigInteger.ZERO;
        for (TransactionOutput output : this.outputs) {
            if (output.getValue().compareTo(BigInteger.ZERO) < 0) {
                throw new VerificationException("Transaction output negative");
            }
            valueOut = valueOut.add(output.getValue());
        }
        if (valueOut.compareTo(NetworkParameters.MAX_MONEY) > 0) {
            throw new VerificationException("Total transaction output value greater than possible");
        }
        if (this.isCoinBase()) {
            if (this.inputs.get(0).getScriptBytes().length < 2 || this.inputs.get(0).getScriptBytes().length > 100) {
                throw new VerificationException("Coinbase script size out of range");
            }
        } else {
            for (TransactionInput input : this.inputs) {
                if (!input.isCoinBase()) continue;
                throw new VerificationException("Coinbase input as input in non-coinbase transaction");
            }
        }
    }

    public boolean isTimeLocked() {
        if (this.getLockTime() == 0L) {
            return false;
        }
        for (TransactionInput input : this.getInputs()) {
            if (!input.hasSequence()) continue;
            return true;
        }
        return false;
    }

    public boolean isFinal(int height, long blockTimeSeconds) {
        long time;
        if (time < ((time = this.getLockTime()) < 500000000L ? (long)height : blockTimeSeconds)) {
            return true;
        }
        return !this.isTimeLocked();
    }

    public static long parseLockTimeStr(String lockTimeStr) throws ParseException {
        if (lockTimeStr.indexOf("/") != -1) {
            SimpleDateFormat format = new SimpleDateFormat("yyyy/MM/dd");
            Date date = format.parse(lockTimeStr);
            return date.getTime() / 1000L;
        }
        return Long.parseLong(lockTimeStr);
    }

    public Date estimateLockTime(AbstractBlockChain chain) {
        if (this.lockTime < 500000000L) {
            return chain.estimateBlockTime((int)this.getLockTime());
        }
        return new Date(this.getLockTime() * 1000L);
    }

    public Purpose getPurpose() {
        return this.purpose;
    }

    public void setPurpose(Purpose purpose) {
        this.purpose = purpose;
    }

    public static enum SigHash {
        ALL,
        NONE,
        SINGLE;

    }

    public static enum Purpose {
        UNKNOWN,
        USER_PAYMENT,
        KEY_ROTATION;

    }
}

