/*
 * Copyright (c) 2012 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied. See the License for the specific language governing permissions and limitations under
 * the License.
 */

package com.google.api.client.googleapis.subscriptions;

import com.google.api.client.googleapis.notifications.StoredChannel;
import com.google.api.client.util.Beta;
import com.google.api.client.util.store.DataStoreFactory;
import com.google.api.client.util.store.MemoryDataStoreFactory;

import java.util.Collection;
import java.util.Collections;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * {@link Beta} <br/>
 * {@link SubscriptionStore} which stores all subscription information in memory.
 *
 * <p>
 * Thread-safe implementation.
 * </p>
 *
 * <b>Example usage:</b>
 *
 * <pre>
    service.setSubscriptionStore(new MemorySubscriptionStore());
 * </pre>
 *
 * @author Matthias Linder (mlinder)
 * @since 1.14
 * @deprecated (scheduled to be removed in 1.17) Use
 *             {@link StoredChannel#getDefaultDataStore(DataStoreFactory)} with
 *             {@link MemoryDataStoreFactory} instead.
 */
@Deprecated
@Beta
public class MemorySubscriptionStore implements SubscriptionStore {

  /** Lock on the token response information. */
  private final Lock lock = new ReentrantLock();

  /** Map of all stored subscriptions. */
  private final SortedMap<String, StoredSubscription> storedSubscriptions =
      new TreeMap<String, StoredSubscription>();

  public void storeSubscription(StoredSubscription subscription) {
    lock.lock();
    try {
      storedSubscriptions.put(subscription.getId(), subscription);
    } finally {
      lock.unlock();
    }
  }

  public void removeSubscription(StoredSubscription subscription) {
    lock.lock();
    try {
      storedSubscriptions.remove(subscription.getId());
    } finally {
      lock.unlock();
    }
  }

  public Collection<StoredSubscription> listSubscriptions() {
    lock.lock();
    try {
      return Collections.unmodifiableCollection(storedSubscriptions.values());
    } finally {
      lock.unlock();
    }
  }

  public StoredSubscription getSubscription(String subscriptionId) {
    lock.lock();
    try {
      return storedSubscriptions.get(subscriptionId);
    } finally {
      lock.unlock();
    }
  }
}
