/*
 * Decompiled with CFR 0.152.
 */
package com.oath.halodb;

import com.oath.halodb.DBDirectory;
import com.oath.halodb.HaloDBOptions;
import com.oath.halodb.TombstoneEntry;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Iterator;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TombstoneFile {
    private static final Logger logger = LoggerFactory.getLogger(TombstoneFile.class);
    private final File backingFile;
    private FileChannel channel;
    private final DBDirectory dbDirectory;
    private final HaloDBOptions options;
    private long unFlushedData = 0L;
    private long writeOffset = 0L;
    static final String TOMBSTONE_FILE_NAME = ".tombstone";
    private static final String nullMessage = "Tombstone entry cannot be null";

    static TombstoneFile create(DBDirectory dbDirectory, int fileId, HaloDBOptions options) throws IOException {
        File file = TombstoneFile.getTombstoneFile(dbDirectory, fileId);
        while (!file.createNewFile()) {
            file = TombstoneFile.getTombstoneFile(dbDirectory, ++fileId);
        }
        TombstoneFile tombstoneFile = new TombstoneFile(file, options, dbDirectory);
        tombstoneFile.open();
        return tombstoneFile;
    }

    TombstoneFile(File backingFile, HaloDBOptions options, DBDirectory dbDirectory) {
        this.backingFile = backingFile;
        this.options = options;
        this.dbDirectory = dbDirectory;
    }

    void open() throws IOException {
        this.channel = new RandomAccessFile(this.backingFile, "rw").getChannel();
    }

    void close() throws IOException {
        if (this.channel != null) {
            this.channel.close();
        }
    }

    void delete() throws IOException {
        this.close();
        if (this.backingFile != null) {
            this.backingFile.delete();
        }
    }

    void write(TombstoneEntry entry) throws IOException {
        long written;
        Objects.requireNonNull(entry, nullMessage);
        ByteBuffer[] contents = entry.serialize();
        long toWrite = 0L;
        for (ByteBuffer buffer : contents) {
            toWrite += (long)buffer.remaining();
        }
        for (written = 0L; written < toWrite; written += this.channel.write(contents)) {
        }
        this.writeOffset += written;
        this.unFlushedData += written;
        if (this.options.isSyncWrite() || this.options.getFlushDataSizeBytes() != -1L && this.unFlushedData > this.options.getFlushDataSizeBytes()) {
            this.flushToDisk();
            this.unFlushedData = 0L;
        }
    }

    long getWriteOffset() {
        return this.writeOffset;
    }

    void flushToDisk() throws IOException {
        if (this.channel != null && this.channel.isOpen()) {
            this.channel.force(true);
        }
    }

    TombstoneFile repairFile(DBDirectory dbDirectory) throws IOException {
        TombstoneFile repairFile = this.createRepairFile();
        logger.info("Repairing tombstone file {}. Records with the correct checksum will be copied to {}", (Object)this.getName(), (Object)repairFile.getName());
        TombstoneFileIterator iterator = this.newIteratorWithCheckForDataCorruption();
        int count = 0;
        while (iterator.hasNext()) {
            TombstoneEntry entry = iterator.next();
            if (entry == null) {
                logger.info("Found a corrupted entry in tombstone file {} after copying {} entries.", (Object)this.getName(), (Object)count);
                break;
            }
            ++count;
            repairFile.write(entry);
        }
        logger.info("Recovered {} records from file {} with size {}. Size after repair {}.", new Object[]{count, this.getName(), this.getSize(), repairFile.getSize()});
        repairFile.flushToDisk();
        Files.move(repairFile.getPath(), this.getPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        dbDirectory.syncMetaData();
        repairFile.close();
        this.close();
        this.open();
        return this;
    }

    private TombstoneFile createRepairFile() throws IOException {
        File repairFile = this.dbDirectory.getPath().resolve(this.getName() + ".repair").toFile();
        while (!repairFile.createNewFile()) {
            logger.info("Repair file {} already exists, probably from a previous repair which failed. Deleting a trying again", (Object)repairFile.getName());
            repairFile.delete();
        }
        TombstoneFile tombstoneFile = new TombstoneFile(repairFile, this.options, this.dbDirectory);
        tombstoneFile.open();
        return tombstoneFile;
    }

    String getName() {
        return this.backingFile.getName();
    }

    private Path getPath() {
        return this.backingFile.toPath();
    }

    private long getSize() {
        return this.backingFile.length();
    }

    TombstoneFileIterator newIterator() throws IOException {
        return new TombstoneFileIterator(false);
    }

    TombstoneFileIterator newIteratorWithCheckForDataCorruption() throws IOException {
        return new TombstoneFileIterator(true);
    }

    private static File getTombstoneFile(DBDirectory dbDirectory, int fileId) {
        return dbDirectory.getPath().resolve(fileId + TOMBSTONE_FILE_NAME).toFile();
    }

    class TombstoneFileIterator
    implements Iterator<TombstoneEntry> {
        private final ByteBuffer buffer;
        private final boolean discardCorruptedRecords;

        TombstoneFileIterator(boolean discardCorruptedRecords) throws IOException {
            this.buffer = TombstoneFile.this.channel.map(FileChannel.MapMode.READ_ONLY, 0L, TombstoneFile.this.channel.size());
            this.discardCorruptedRecords = discardCorruptedRecords;
        }

        @Override
        public boolean hasNext() {
            return this.buffer.hasRemaining();
        }

        @Override
        public TombstoneEntry next() {
            if (this.hasNext()) {
                if (this.discardCorruptedRecords) {
                    return TombstoneEntry.deserializeIfNotCorrupted(this.buffer);
                }
                return TombstoneEntry.deserialize(this.buffer);
            }
            return null;
        }
    }
}

