/*
 * Decompiled with CFR 0.152.
 */
package com.oath.halodb;

import com.google.common.primitives.Ints;
import com.oath.halodb.HashTableUtil;
import com.oath.halodb.HashTableValueSerializer;
import com.oath.halodb.Hasher;
import com.oath.halodb.KeyBuffer;
import com.oath.halodb.OffHeapHashTable;
import com.oath.halodb.OffHeapHashTableBuilder;
import com.oath.halodb.OffHeapHashTableStats;
import com.oath.halodb.Segment;
import com.oath.halodb.SegmentNonMemoryPool;
import com.oath.halodb.SegmentStats;
import com.oath.halodb.SegmentWithMemoryPool;
import com.oath.halodb.histo.EstimatedHistogram;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class OffHeapHashTableImpl<V>
implements OffHeapHashTable<V> {
    private static final Logger logger = LoggerFactory.getLogger(OffHeapHashTableImpl.class);
    private final HashTableValueSerializer<V> valueSerializer;
    private final int fixedValueLength;
    private final List<Segment<V>> segments;
    private final long segmentMask;
    private final int segmentShift;
    private final int segmentCount;
    private volatile long putFailCount;
    private boolean closed;
    private final Hasher hasher;

    OffHeapHashTableImpl(OffHeapHashTableBuilder<V> builder) {
        this.hasher = Hasher.create(builder.getHashAlgorighm());
        this.fixedValueLength = builder.getFixedValueSize();
        if (builder.getSegmentCount() <= 0) {
            throw new IllegalArgumentException("Segment count should be > 0");
        }
        this.segmentCount = Ints.checkedCast((long)HashTableUtil.roundUpToPowerOf2(builder.getSegmentCount(), 0x40000000L));
        this.segments = new ArrayList<Segment<V>>(this.segmentCount);
        for (int i = 0; i < this.segmentCount; ++i) {
            try {
                this.segments.add(this.allocateSegment(builder));
                continue;
            }
            catch (RuntimeException e) {
                while (i >= 0) {
                    if (this.segments.get(i) != null) {
                        this.segments.get(i).release();
                    }
                    --i;
                }
                throw e;
            }
        }
        int bitNum = HashTableUtil.bitNum(this.segmentCount) - 1;
        this.segmentShift = 64 - bitNum;
        this.segmentMask = (long)this.segmentCount - 1L << this.segmentShift;
        this.valueSerializer = builder.getValueSerializer();
        if (this.valueSerializer == null) {
            throw new NullPointerException("valueSerializer == null");
        }
        logger.debug("off-heap index with {} segments created.", (Object)this.segmentCount);
    }

    private Segment<V> allocateSegment(OffHeapHashTableBuilder<V> builder) {
        if (builder.isUseMemoryPool()) {
            return new SegmentWithMemoryPool<V>(builder);
        }
        return new SegmentNonMemoryPool<V>(builder);
    }

    @Override
    public V get(byte[] key) {
        if (key == null) {
            throw new NullPointerException();
        }
        KeyBuffer keySource = this.keySource(key);
        return this.segment(keySource.hash()).getEntry(keySource);
    }

    @Override
    public boolean containsKey(byte[] key) {
        if (key == null) {
            throw new NullPointerException();
        }
        KeyBuffer keySource = this.keySource(key);
        return this.segment(keySource.hash()).containsEntry(keySource);
    }

    @Override
    public boolean put(byte[] k, V v) {
        return this.putInternal(k, v, false, null);
    }

    @Override
    public boolean addOrReplace(byte[] key, V old, V value) {
        return this.putInternal(key, value, false, old);
    }

    @Override
    public boolean putIfAbsent(byte[] k, V v) {
        return this.putInternal(k, v, true, null);
    }

    private boolean putInternal(byte[] key, V value, boolean ifAbsent, V old) {
        if (key == null || value == null) {
            throw new NullPointerException();
        }
        int valueSize = this.valueSize(value);
        if (valueSize != this.fixedValueLength) {
            throw new IllegalArgumentException("value size " + valueSize + " greater than fixed value size " + this.fixedValueLength);
        }
        if (old != null && this.valueSize(old) != this.fixedValueLength) {
            throw new IllegalArgumentException("old value size " + this.valueSize(old) + " greater than fixed value size " + this.fixedValueLength);
        }
        if (key.length > 127) {
            throw new IllegalArgumentException("key size of " + key.length + " exceeds max permitted size of " + 127);
        }
        long hash = this.hasher.hash(key);
        return this.segment(hash).putEntry(key, value, hash, ifAbsent, old);
    }

    private int valueSize(V v) {
        int sz = this.valueSerializer.serializedSize(v);
        if (sz <= 0) {
            throw new IllegalArgumentException("Illegal value length " + sz);
        }
        return sz;
    }

    @Override
    public boolean remove(byte[] k) {
        if (k == null) {
            throw new NullPointerException();
        }
        KeyBuffer keySource = this.keySource(k);
        return this.segment(keySource.hash()).removeEntry(keySource);
    }

    private Segment<V> segment(long hash) {
        int seg = (int)((hash & this.segmentMask) >>> this.segmentShift);
        return this.segments.get(seg);
    }

    private KeyBuffer keySource(byte[] key) {
        KeyBuffer keyBuffer = new KeyBuffer(key);
        return keyBuffer.finish(this.hasher);
    }

    @Override
    public void clear() {
        for (Segment<V> map : this.segments) {
            map.clear();
        }
    }

    public void setCapacity(long capacity) {
    }

    @Override
    public void close() {
        this.closed = true;
        for (Segment<V> map : this.segments) {
            map.release();
        }
        Collections.fill(this.segments, null);
        if (logger.isDebugEnabled()) {
            logger.debug("Closing OHC instance");
        }
    }

    @Override
    public void resetStatistics() {
        for (Segment<V> map : this.segments) {
            map.resetStatistics();
        }
        this.putFailCount = 0L;
    }

    @Override
    public OffHeapHashTableStats stats() {
        long hitCount = 0L;
        long missCount = 0L;
        long size = 0L;
        long freeCapacity = 0L;
        long rehashes = 0L;
        long putAddCount = 0L;
        long putReplaceCount = 0L;
        long removeCount = 0L;
        for (Segment<V> map : this.segments) {
            hitCount += map.hitCount();
            missCount += map.missCount();
            size += map.size();
            rehashes += map.rehashes();
            putAddCount += map.putAddCount();
            putReplaceCount += map.putReplaceCount();
            removeCount += map.removeCount();
        }
        return new OffHeapHashTableStats(hitCount, missCount, size, rehashes, putAddCount, putReplaceCount, this.putFailCount, removeCount, this.perSegmentStats());
    }

    @Override
    public long size() {
        long size = 0L;
        for (Segment<V> map : this.segments) {
            size += map.size();
        }
        return size;
    }

    @Override
    public int segments() {
        return this.segments.size();
    }

    @Override
    public float loadFactor() {
        return this.segments.get(0).loadFactor();
    }

    @Override
    public int[] hashTableSizes() {
        int[] r = new int[this.segments.size()];
        for (int i = 0; i < this.segments.size(); ++i) {
            r[i] = this.segments.get(i).hashTableSize();
        }
        return r;
    }

    public long[] perSegmentSizes() {
        long[] r = new long[this.segments.size()];
        for (int i = 0; i < this.segments.size(); ++i) {
            r[i] = this.segments.get(i).size();
        }
        return r;
    }

    @Override
    public SegmentStats[] perSegmentStats() {
        SegmentStats[] stats = new SegmentStats[this.segments.size()];
        for (int i = 0; i < stats.length; ++i) {
            Segment<V> map = this.segments.get(i);
            stats[i] = new SegmentStats(map.size(), map.numberOfChunks(), map.numberOfSlots(), map.freeListSize());
        }
        return stats;
    }

    @Override
    public EstimatedHistogram getBucketHistogram() {
        int i;
        EstimatedHistogram hist = new EstimatedHistogram();
        for (Segment<V> map : this.segments) {
            map.updateBucketHistogram(hist);
        }
        long[] offsets = hist.getBucketOffsets();
        long[] buckets = hist.getBuckets(false);
        for (i = buckets.length - 1; i > 0; --i) {
            if (buckets[i] == 0L) continue;
            offsets = Arrays.copyOf(offsets, i + 2);
            buckets = Arrays.copyOf(buckets, i + 3);
            System.arraycopy(offsets, 0, offsets, 1, i + 1);
            System.arraycopy(buckets, 0, buckets, 1, i + 2);
            offsets[0] = 0L;
            buckets[0] = 0L;
            break;
        }
        i = 0;
        while (i < offsets.length) {
            int n = i++;
            offsets[n] = offsets[n] - 1L;
        }
        return new EstimatedHistogram(offsets, buckets);
    }

    public String toString() {
        return this.getClass().getSimpleName() + " ,segments=" + this.segments.size();
    }
}

