/*
 * Decompiled with CFR 0.152.
 */
package com.oath.halodb;

import com.oath.halodb.DBDirectory;
import com.oath.halodb.HaloDBOptions;
import com.oath.halodb.IndexFileEntry;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.util.Iterator;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class IndexFile {
    private static final Logger logger = LoggerFactory.getLogger(IndexFile.class);
    private final int fileId;
    private final DBDirectory dbDirectory;
    private File backingFile;
    private FileChannel channel;
    private final HaloDBOptions options;
    private long unFlushedData = 0L;
    static final String INDEX_FILE_NAME = ".index";
    private static final String nullMessage = "Index file entry cannot be null";

    IndexFile(int fileId, DBDirectory dbDirectory, HaloDBOptions options) {
        this.fileId = fileId;
        this.dbDirectory = dbDirectory;
        this.options = options;
    }

    void create() throws IOException {
        this.backingFile = this.getIndexFile();
        if (!this.backingFile.createNewFile()) {
            throw new IOException("Index file with id " + this.fileId + " already exists");
        }
        this.channel = new RandomAccessFile(this.backingFile, "rw").getChannel();
    }

    void createRepairFile() throws IOException {
        this.backingFile = this.getRepairFile();
        while (!this.backingFile.createNewFile()) {
            logger.info("Repair file {} already exists, probably from a previous repair which failed. Deleting a trying again", (Object)this.backingFile.getName());
            this.backingFile.delete();
        }
        this.channel = new RandomAccessFile(this.backingFile, "rw").getChannel();
    }

    void open() throws IOException {
        this.backingFile = this.getIndexFile();
        this.channel = new RandomAccessFile(this.backingFile, "rw").getChannel();
    }

    void close() throws IOException {
        if (this.channel != null) {
            this.channel.close();
        }
    }

    void delete() throws IOException {
        if (this.channel != null && this.channel.isOpen()) {
            this.channel.close();
        }
        this.getIndexFile().delete();
    }

    void write(IndexFileEntry entry) throws IOException {
        long written;
        Objects.requireNonNull(entry, nullMessage);
        ByteBuffer[] contents = entry.serialize();
        long toWrite = 0L;
        for (ByteBuffer buffer : contents) {
            toWrite += (long)buffer.remaining();
        }
        for (written = 0L; written < toWrite; written += this.channel.write(contents)) {
        }
        this.unFlushedData += written;
        if (this.options.getFlushDataSizeBytes() != -1L && this.unFlushedData > this.options.getFlushDataSizeBytes()) {
            this.channel.force(false);
            this.unFlushedData = 0L;
        }
    }

    void flushToDisk() throws IOException {
        if (this.channel != null && this.channel.isOpen()) {
            this.channel.force(true);
        }
    }

    IndexFileIterator newIterator() throws IOException {
        return new IndexFileIterator();
    }

    Path getPath() {
        return this.backingFile.toPath();
    }

    private File getIndexFile() {
        return this.dbDirectory.getPath().resolve(this.fileId + INDEX_FILE_NAME).toFile();
    }

    private File getRepairFile() {
        return this.dbDirectory.getPath().resolve(this.fileId + INDEX_FILE_NAME + ".repair").toFile();
    }

    public class IndexFileIterator
    implements Iterator<IndexFileEntry> {
        private final ByteBuffer buffer;

        public IndexFileIterator() throws IOException {
            this.buffer = IndexFile.this.channel.map(FileChannel.MapMode.READ_ONLY, 0L, IndexFile.this.channel.size());
        }

        @Override
        public boolean hasNext() {
            return this.buffer.hasRemaining();
        }

        @Override
        public IndexFileEntry next() {
            if (this.hasNext()) {
                return IndexFileEntry.deserialize(this.buffer);
            }
            return null;
        }
    }
}

