/*
 * Decompiled with CFR 0.152.
 */
package com.oath.halodb;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.RateLimiter;
import com.oath.halodb.CompactionManager;
import com.oath.halodb.Constants;
import com.oath.halodb.DBDirectory;
import com.oath.halodb.DBMetaData;
import com.oath.halodb.FileUtils;
import com.oath.halodb.HaloDBException;
import com.oath.halodb.HaloDBFile;
import com.oath.halodb.HaloDBOptions;
import com.oath.halodb.HaloDBStats;
import com.oath.halodb.InMemoryIndex;
import com.oath.halodb.InMemoryIndexMetaData;
import com.oath.halodb.IndexFile;
import com.oath.halodb.IndexFileEntry;
import com.oath.halodb.OffHeapHashTableStats;
import com.oath.halodb.Record;
import com.oath.halodb.TombstoneEntry;
import com.oath.halodb.TombstoneFile;
import com.oath.halodb.Utils;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Matcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HaloDBInternal {
    private static final Logger logger = LoggerFactory.getLogger(HaloDBInternal.class);
    static final String SNAPSHOT_SUBDIR = "snapshot";
    private DBDirectory dbDirectory;
    private volatile HaloDBFile currentWriteFile;
    private volatile TombstoneFile currentTombstoneFile;
    private volatile Thread tombstoneMergeThread;
    private Map<Integer, HaloDBFile> readFileMap = new ConcurrentHashMap<Integer, HaloDBFile>();
    HaloDBOptions options;
    private InMemoryIndex inMemoryIndex;
    private final Map<Integer, Integer> staleDataPerFileMap = new ConcurrentHashMap<Integer, Integer>();
    private CompactionManager compactionManager;
    private AtomicInteger nextFileId;
    private volatile boolean isClosing = false;
    private volatile long statsResetTime = System.currentTimeMillis();
    private FileLock dbLock;
    private final Lock writeLock = new ReentrantLock();
    private static final int maxReadAttempts = 5;
    private AtomicLong noOfTombstonesCopiedDuringOpen;
    private AtomicLong noOfTombstonesFoundDuringOpen;
    private volatile long nextSequenceNumber;
    private volatile boolean isTombstoneFilesMerging = false;

    private HaloDBInternal() {
    }

    static HaloDBInternal open(File directory, HaloDBOptions options) throws HaloDBException, IOException {
        HaloDBInternal.checkIfOptionsAreCorrect(options);
        HaloDBInternal dbInternal = new HaloDBInternal();
        try {
            dbInternal.dbDirectory = DBDirectory.open(directory);
            dbInternal.dbLock = dbInternal.getLock();
            dbInternal.options = options;
            int maxFileId = dbInternal.buildReadFileMap();
            dbInternal.nextFileId = new AtomicInteger(maxFileId + 10);
            dbInternal.noOfTombstonesCopiedDuringOpen = new AtomicLong(0L);
            dbInternal.noOfTombstonesFoundDuringOpen = new AtomicLong(0L);
            DBMetaData dbMetaData = new DBMetaData(dbInternal.dbDirectory);
            dbMetaData.loadFromFileIfExists();
            if (dbMetaData.getMaxFileSize() != 0 && dbMetaData.getMaxFileSize() != options.getMaxFileSize()) {
                throw new IllegalArgumentException("File size cannot be changed after db was created. Current size " + dbMetaData.getMaxFileSize());
            }
            if (dbMetaData.isOpen() || dbMetaData.isIOError()) {
                logger.info("DB was not shutdown correctly last time. Files may not be consistent, repairing them.");
                dbInternal.repairFiles();
            }
            dbMetaData.setOpen(true);
            dbMetaData.setIOError(false);
            dbMetaData.setVersion(0);
            dbMetaData.setMaxFileSize(options.getMaxFileSize());
            dbMetaData.storeToFile();
            dbInternal.compactionManager = new CompactionManager(dbInternal);
            dbInternal.inMemoryIndex = new InMemoryIndex(options.getNumberOfRecords(), options.isUseMemoryPool(), options.getFixedKeySize(), options.getMemoryPoolChunkSize());
            long maxSequenceNumber = dbInternal.buildInMemoryIndex();
            if (maxSequenceNumber == -1L) {
                dbInternal.nextSequenceNumber = 1L;
                logger.info("Didn't find any existing records; initializing max sequence number to 1");
            } else {
                dbInternal.nextSequenceNumber = maxSequenceNumber + 100L;
                logger.info("Found max sequence number {}, now starting from {}", (Object)maxSequenceNumber, (Object)dbInternal.nextSequenceNumber);
            }
            if (!options.isCompactionDisabled()) {
                dbInternal.compactionManager.startCompactionThread();
            } else {
                logger.warn("Compaction is disabled in HaloDBOption. This should happen only in tests");
            }
            if (options.isCleanUpTombstonesDuringOpen()) {
                dbInternal.isTombstoneFilesMerging = true;
                dbInternal.tombstoneMergeThread = new Thread(() -> dbInternal.mergeTombstoneFiles());
                dbInternal.tombstoneMergeThread.start();
            }
            logger.info("Opened HaloDB {}", (Object)directory.getName());
            logger.info("maxFileSize - {}", (Object)options.getMaxFileSize());
            logger.info("compactionThresholdPerFile - {}", (Object)options.getCompactionThresholdPerFile());
        }
        catch (Exception e) {
            if (dbInternal.dbLock != null) {
                dbInternal.dbLock.close();
            }
            throw e;
        }
        return dbInternal;
    }

    synchronized void close() throws IOException {
        this.writeLock.lock();
        try {
            if (this.isClosing) {
                return;
            }
            this.isClosing = true;
            try {
                if (!this.compactionManager.stopCompactionThread(true)) {
                    this.setIOErrorFlag();
                }
            }
            catch (IOException e) {
                logger.error("Error while stopping compaction thread. Setting IOError flag", (Throwable)e);
                this.setIOErrorFlag();
            }
            if (this.isTombstoneFilesMerging) {
                try {
                    this.tombstoneMergeThread.join();
                }
                catch (InterruptedException e) {
                    logger.error("Interrupted when waiting the tombstone files merging");
                    this.setIOErrorFlag();
                }
            }
            if (this.options.isCleanUpInMemoryIndexOnClose()) {
                this.inMemoryIndex.close();
            }
            if (this.currentWriteFile != null) {
                this.currentWriteFile.flushToDisk();
                this.currentWriteFile.getIndexFile().flushToDisk();
                this.currentWriteFile.close();
            }
            if (this.currentTombstoneFile != null) {
                this.currentTombstoneFile.flushToDisk();
                this.currentTombstoneFile.close();
            }
            for (HaloDBFile file : this.readFileMap.values()) {
                file.close();
            }
            DBMetaData metaData = new DBMetaData(this.dbDirectory);
            metaData.loadFromFileIfExists();
            metaData.setOpen(false);
            metaData.storeToFile();
            this.dbDirectory.close();
            if (this.dbLock != null) {
                this.dbLock.close();
            }
        }
        finally {
            this.writeLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean put(byte[] key, byte[] value) throws IOException, HaloDBException {
        if (key.length > 127) {
            throw new HaloDBException("key length cannot exceed 127");
        }
        this.writeLock.lock();
        try {
            Record record = new Record(key, value);
            record.setSequenceNumber(this.getNextSequenceNumber());
            record.setVersion(0);
            InMemoryIndexMetaData entry = this.writeRecordToFile(record);
            this.markPreviousVersionAsStale(key);
            boolean bl = this.inMemoryIndex.put(key, entry);
            return bl;
        }
        finally {
            this.writeLock.unlock();
        }
    }

    byte[] get(byte[] key, int attemptNumber) throws IOException, HaloDBException {
        if (attemptNumber > 5) {
            logger.error("Tried {} attempts but read failed", (Object)(attemptNumber - 1));
            throw new HaloDBException("Tried " + (attemptNumber - 1) + " attempts but failed.");
        }
        InMemoryIndexMetaData metaData = this.inMemoryIndex.get(key);
        if (metaData == null) {
            return null;
        }
        HaloDBFile readFile = this.readFileMap.get(metaData.getFileId());
        if (readFile == null) {
            logger.debug("File {} not present. Compaction job would have deleted it. Retrying ...", (Object)metaData.getFileId());
            return this.get(key, attemptNumber + 1);
        }
        try {
            return readFile.readFromFile(metaData.getValueOffset(), metaData.getValueSize());
        }
        catch (ClosedChannelException e) {
            if (!this.isClosing) {
                logger.debug("File {} was closed. Compaction job would have deleted it. Retrying ...", (Object)metaData.getFileId());
                return this.get(key, attemptNumber + 1);
            }
            throw e;
        }
    }

    int get(byte[] key, ByteBuffer buffer) throws IOException {
        InMemoryIndexMetaData metaData = this.inMemoryIndex.get(key);
        if (metaData == null) {
            return 0;
        }
        HaloDBFile readFile = this.readFileMap.get(metaData.getFileId());
        if (readFile == null) {
            logger.debug("File {} not present. Compaction job would have deleted it. Retrying ...", (Object)metaData.getFileId());
            return this.get(key, buffer);
        }
        buffer.clear();
        buffer.limit(metaData.getValueSize());
        try {
            int read = readFile.readFromFile((long)metaData.getValueOffset(), buffer);
            buffer.flip();
            return read;
        }
        catch (ClosedChannelException e) {
            if (!this.isClosing) {
                logger.debug("File {} was closed. Compaction job would have deleted it. Retrying ...", (Object)metaData.getFileId());
                return this.get(key, buffer);
            }
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized boolean takeSnapshot() {
        File[] filesToLink;
        File snapshotDir;
        logger.info("Start generating the snapshot");
        if (this.isTombstoneFilesMerging) {
            logger.info("DB is merging the tombstone files now. Wait it finished");
            try {
                this.tombstoneMergeThread.join();
            }
            catch (InterruptedException e) {
                logger.error("Interrupted when waiting the tombstone files merging");
                return false;
            }
        }
        try {
            int currentWriteFileId;
            this.compactionManager.pauseCompactionThread();
            snapshotDir = this.getSnapshotDirectory();
            if (snapshotDir.exists()) {
                logger.warn("The snapshot dir is already existed. Delete the old one.");
                FileUtils.deleteDirectory(snapshotDir);
            }
            FileUtils.createDirectoryIfNotExists(snapshotDir);
            logger.info("Created directory for snapshot {}", (Object)snapshotDir.toString());
            this.writeLock.lock();
            try {
                this.forceRollOverCurrentWriteFile();
                this.currentTombstoneFile = this.forceRollOverTombstoneFile(this.currentTombstoneFile);
                currentWriteFileId = this.currentWriteFile.getFileId();
            }
            catch (IOException e) {
                logger.warn("IO exception when rollover current write files", (Throwable)e);
                boolean bl = false;
                this.compactionManager.resumeCompaction();
                return bl;
            }
            finally {
                this.writeLock.unlock();
            }
            filesToLink = this.dbDirectory.getPath().toFile().listFiles(file -> {
                Matcher m = Constants.STORAGE_FILE_PATTERN.matcher(file.getName());
                return m.matches() && Integer.parseInt(m.group(1)) < currentWriteFileId;
            });
        }
        catch (IOException e) {
            logger.warn("IOException when creating snapshot", (Throwable)e);
            boolean bl = false;
            return bl;
        }
        finally {
            this.compactionManager.resumeCompaction();
        }
        this.compactionManager.forceRolloverCurrentWriteFile();
        logger.info("Storage files number need to be linked: {}", (Object)filesToLink.length);
        for (File file2 : filesToLink) {
            Path dest = Paths.get(snapshotDir.getAbsolutePath(), file2.getName());
            logger.debug("Create file link from file {} to {}", (Object)file2.getName(), (Object)dest.toFile().getAbsoluteFile());
            Files.createLink(dest, file2.toPath());
        }
        return true;
    }

    File getSnapshotDirectory() {
        Path dbDirectoryPath = this.dbDirectory.getPath();
        return Paths.get(dbDirectoryPath.toFile().getAbsolutePath(), SNAPSHOT_SUBDIR).toFile();
    }

    boolean clearSnapshot() {
        File snapshotDir = this.getSnapshotDirectory();
        if (snapshotDir.exists()) {
            try {
                FileUtils.deleteDirectory(snapshotDir);
            }
            catch (IOException e) {
                logger.error("snapshot deletion error", (Throwable)e);
                return false;
            }
            return true;
        }
        logger.info("snapshot not existed");
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void delete(byte[] key) throws IOException {
        this.writeLock.lock();
        try {
            InMemoryIndexMetaData metaData = this.inMemoryIndex.get(key);
            if (metaData != null) {
                this.inMemoryIndex.remove(key);
                TombstoneEntry entry = new TombstoneEntry(key, this.getNextSequenceNumber(), -1L, 0);
                this.currentTombstoneFile = this.rollOverTombstoneFile(entry, this.currentTombstoneFile);
                this.currentTombstoneFile.write(entry);
                this.markPreviousVersionAsStale(key, metaData);
            }
        }
        finally {
            this.writeLock.unlock();
        }
    }

    long size() {
        return this.inMemoryIndex.size();
    }

    void setIOErrorFlag() throws IOException {
        DBMetaData metaData = new DBMetaData(this.dbDirectory);
        metaData.loadFromFileIfExists();
        metaData.setIOError(true);
        metaData.storeToFile();
    }

    void pauseCompaction() throws IOException {
        this.compactionManager.pauseCompactionThread();
    }

    void resumeCompaction() {
        this.compactionManager.resumeCompaction();
    }

    private InMemoryIndexMetaData writeRecordToFile(Record record) throws IOException, HaloDBException {
        this.rollOverCurrentWriteFile(record);
        return this.currentWriteFile.writeRecord(record);
    }

    private void rollOverCurrentWriteFile(Record record) throws IOException {
        int size = record.getKey().length + record.getValue().length + 18;
        if (!(this.currentWriteFile != null && this.currentWriteFile.getWriteOffset() + (long)size <= (long)this.options.getMaxFileSize() || this.isClosing)) {
            this.forceRollOverCurrentWriteFile();
        }
    }

    private void forceRollOverCurrentWriteFile() throws IOException {
        if (this.currentWriteFile != null) {
            this.currentWriteFile.flushToDisk();
            this.currentWriteFile.getIndexFile().flushToDisk();
        }
        this.currentWriteFile = this.createHaloDBFile(HaloDBFile.FileType.DATA_FILE);
        this.dbDirectory.syncMetaData();
    }

    private TombstoneFile rollOverTombstoneFile(TombstoneEntry entry, TombstoneFile tombstoneFile) throws IOException {
        int size = entry.getKey().length + 14;
        if (!(tombstoneFile != null && tombstoneFile.getWriteOffset() + (long)size <= (long)this.options.getMaxTombstoneFileSize() || this.isClosing)) {
            tombstoneFile = this.forceRollOverTombstoneFile(tombstoneFile);
        }
        return tombstoneFile;
    }

    private TombstoneFile forceRollOverTombstoneFile(TombstoneFile tombstoneFile) throws IOException {
        if (tombstoneFile != null) {
            tombstoneFile.flushToDisk();
            tombstoneFile.close();
        }
        tombstoneFile = TombstoneFile.create(this.dbDirectory, this.getNextFileId(), this.options);
        this.dbDirectory.syncMetaData();
        return tombstoneFile;
    }

    private void markPreviousVersionAsStale(byte[] key) {
        InMemoryIndexMetaData recordMetaData = this.inMemoryIndex.get(key);
        if (recordMetaData != null) {
            this.markPreviousVersionAsStale(key, recordMetaData);
        }
    }

    private void markPreviousVersionAsStale(byte[] key, InMemoryIndexMetaData recordMetaData) {
        int staleRecordSize = Utils.getRecordSize(key.length, recordMetaData.getValueSize());
        this.addFileToCompactionQueueIfThresholdCrossed(recordMetaData.getFileId(), staleRecordSize);
    }

    void addFileToCompactionQueueIfThresholdCrossed(int fileId, int staleRecordSize) {
        HaloDBFile file = this.readFileMap.get(fileId);
        if (file == null) {
            return;
        }
        int staleSizeInFile = this.updateStaleDataMap(fileId, staleRecordSize);
        if ((double)staleSizeInFile >= (double)file.getSize() * this.options.getCompactionThresholdPerFile() && this.getCurrentWriteFileId() != fileId && this.compactionManager.getCurrentWriteFileId() != fileId && this.compactionManager.submitFileForCompaction(fileId)) {
            this.staleDataPerFileMap.remove(fileId);
        }
    }

    private int updateStaleDataMap(int fileId, int staleDataSize) {
        return this.staleDataPerFileMap.merge(fileId, staleDataSize, (oldValue, newValue) -> oldValue + newValue);
    }

    void markFileAsCompacted(int fileId) {
        this.staleDataPerFileMap.remove(fileId);
    }

    InMemoryIndex getInMemoryIndex() {
        return this.inMemoryIndex;
    }

    HaloDBFile createHaloDBFile(HaloDBFile.FileType fileType) throws IOException {
        HaloDBFile file = HaloDBFile.create(this.dbDirectory, this.getNextFileId(), this.options, fileType);
        if (this.readFileMap.putIfAbsent(file.getFileId(), file) != null) {
            throw new IOException("Error while trying to create file " + file.getName() + " file with the given id already exists in the map");
        }
        return file;
    }

    private List<HaloDBFile> openDataFilesForReading() throws IOException {
        File[] files = this.dbDirectory.listDataFiles();
        ArrayList<HaloDBFile> result = new ArrayList<HaloDBFile>();
        for (File f : files) {
            HaloDBFile.FileType fileType = HaloDBFile.findFileType(f);
            result.add(HaloDBFile.openForReading(this.dbDirectory, f, fileType, this.options));
        }
        return result;
    }

    private int buildReadFileMap() throws HaloDBException, IOException {
        int maxFileId = Integer.MIN_VALUE;
        for (HaloDBFile file : this.openDataFilesForReading()) {
            if (this.readFileMap.putIfAbsent(file.getFileId(), file) != null) {
                throw new HaloDBException("Found duplicate file with id " + file.getFileId());
            }
            maxFileId = Math.max(maxFileId, file.getFileId());
        }
        if (maxFileId == Integer.MIN_VALUE) {
            maxFileId = Ints.checkedCast((long)(System.currentTimeMillis() / 1000L));
        }
        return maxFileId;
    }

    private int getNextFileId() {
        return this.nextFileId.incrementAndGet();
    }

    private Optional<HaloDBFile> getLatestDataFile(HaloDBFile.FileType fileType) {
        return this.readFileMap.values().stream().filter(f -> f.getFileType() == fileType).max(Comparator.comparingInt(HaloDBFile::getFileId));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long buildInMemoryIndex() throws IOException {
        int nThreads = this.options.getBuildIndexThreads();
        logger.info("Building index in parallel with {} threads", (Object)nThreads);
        ExecutorService executor = Executors.newFixedThreadPool(nThreads);
        try {
            long l = this.buildInMemoryIndex(executor);
            return l;
        }
        finally {
            executor.shutdown();
        }
    }

    private long buildInMemoryIndex(ExecutorService executor) throws IOException {
        List<Integer> indexFiles = this.dbDirectory.listIndexFiles();
        logger.info("About to scan {} index files to construct index ...", (Object)indexFiles.size());
        long start = System.currentTimeMillis();
        long maxSequenceNumber = -1L;
        ArrayList<ProcessIndexFileTask> indexFileTasks = new ArrayList<ProcessIndexFileTask>();
        for (int fileId : indexFiles) {
            IndexFile indexFile = new IndexFile(fileId, this.dbDirectory, this.options);
            indexFileTasks.add(new ProcessIndexFileTask(indexFile, fileId));
        }
        try {
            List results = executor.invokeAll(indexFileTasks);
            for (Future future : results) {
                maxSequenceNumber = Long.max((Long)future.get(), maxSequenceNumber);
            }
        }
        catch (InterruptedException ie) {
            throw new IOException("Building index is interrupted");
        }
        catch (ExecutionException ee) {
            throw new IOException("Error happened during building in-memory index", ee);
        }
        logger.info("Completed scanning all index files in {}s", (Object)((System.currentTimeMillis() - start) / 1000L));
        start = System.currentTimeMillis();
        File[] tombStoneFiles = this.dbDirectory.listTombstoneFiles();
        logger.info("About to scan {} tombstone files ...", (Object)tombStoneFiles.length);
        ArrayList<ProcessTombstoneFileTask> tombstoneFileTasks = new ArrayList<ProcessTombstoneFileTask>();
        for (File file : tombStoneFiles) {
            TombstoneFile tombstoneFile = new TombstoneFile(file, this.options, this.dbDirectory);
            tombstoneFileTasks.add(new ProcessTombstoneFileTask(tombstoneFile));
        }
        try {
            List list = executor.invokeAll(tombstoneFileTasks);
            for (Future result : list) {
                maxSequenceNumber = Long.max((Long)result.get(), maxSequenceNumber);
            }
        }
        catch (InterruptedException interruptedException) {
            throw new IOException("Building index is interrupted");
        }
        catch (ExecutionException executionException) {
            throw new IOException("Error happened during building in-memory index", executionException);
        }
        logger.info("Completed scanning all tombstone files in {}s", (Object)((System.currentTimeMillis() - start) / 1000L));
        return maxSequenceNumber;
    }

    HaloDBFile getHaloDBFile(int fileId) {
        return this.readFileMap.get(fileId);
    }

    void deleteHaloDBFile(int fileId) throws IOException {
        HaloDBFile file = this.readFileMap.get(fileId);
        if (file != null) {
            this.readFileMap.remove(fileId);
            file.delete();
        }
        this.staleDataPerFileMap.remove(fileId);
    }

    private void mergeTombstoneFiles() {
        File[] tombStoneFiles = this.dbDirectory.listTombstoneFiles();
        logger.info("About to merge {} tombstone files ...", (Object)tombStoneFiles.length);
        TombstoneFile mergedTombstoneFile = null;
        RateLimiter rateLimiter = RateLimiter.create((double)this.options.getCompactionJobRate());
        for (File file : tombStoneFiles) {
            TombstoneFile tombstoneFile = new TombstoneFile(file, this.options, this.dbDirectory);
            if (this.currentTombstoneFile != null && tombstoneFile.getName().equals(this.currentTombstoneFile.getName())) continue;
            try {
                tombstoneFile.open();
                TombstoneFile.TombstoneFileIterator iterator = tombstoneFile.newIterator();
                long count = 0L;
                while (iterator.hasNext()) {
                    TombstoneEntry entry = iterator.next();
                    rateLimiter.acquire(entry.size());
                    ++count;
                    mergedTombstoneFile = this.rollOverTombstoneFile(entry, mergedTombstoneFile);
                    mergedTombstoneFile.write(entry);
                }
                if (count > 0L) {
                    logger.debug("Merged {} tombstones from {} to {}", new Object[]{count, tombstoneFile.getName(), mergedTombstoneFile.getName()});
                }
                tombstoneFile.close();
                tombstoneFile.delete();
            }
            catch (IOException e) {
                logger.error("IO exception when merging tombstone file", (Throwable)e);
            }
        }
        if (mergedTombstoneFile != null) {
            try {
                mergedTombstoneFile.close();
            }
            catch (IOException e) {
                logger.error("IO exception when closing tombstone file: {}", (Object)mergedTombstoneFile.getName(), (Object)e);
            }
        }
        logger.info("Tombstone files count, before merge:{}, after merge:{}", (Object)tombStoneFiles.length, (Object)this.dbDirectory.listTombstoneFiles().length);
        this.isTombstoneFilesMerging = false;
    }

    private void repairFiles() {
        this.getLatestDataFile(HaloDBFile.FileType.DATA_FILE).ifPresent(file -> {
            try {
                logger.info("Repairing file {}.data", (Object)file.getFileId());
                HaloDBFile repairedFile = file.repairFile(this.dbDirectory);
                this.readFileMap.put(repairedFile.getFileId(), repairedFile);
            }
            catch (IOException e) {
                throw new RuntimeException("Exception while repairing data file " + file.getFileId() + " which might be corrupted", e);
            }
        });
        this.getLatestDataFile(HaloDBFile.FileType.COMPACTED_FILE).ifPresent(file -> {
            try {
                logger.info("Repairing file {}.datac", (Object)file.getFileId());
                HaloDBFile repairedFile = file.repairFile(this.dbDirectory);
                this.readFileMap.put(repairedFile.getFileId(), repairedFile);
            }
            catch (IOException e) {
                throw new RuntimeException("Exception while repairing datac file " + file.getFileId() + " which might be corrupted", e);
            }
        });
        File[] tombstoneFiles = this.dbDirectory.listTombstoneFiles();
        if (tombstoneFiles != null && tombstoneFiles.length > 0) {
            TombstoneFile lastFile = new TombstoneFile(tombstoneFiles[tombstoneFiles.length - 1], this.options, this.dbDirectory);
            try {
                logger.info("Repairing {} file", (Object)lastFile.getName());
                lastFile.open();
                TombstoneFile repairedFile = lastFile.repairFile(this.dbDirectory);
                repairedFile.close();
            }
            catch (IOException e) {
                throw new RuntimeException("Exception while repairing tombstone file " + lastFile.getName() + " which might be corrupted", e);
            }
        }
    }

    private FileLock getLock() throws HaloDBException {
        try {
            FileLock lock = FileChannel.open(this.dbDirectory.getPath().resolve("LOCK"), StandardOpenOption.CREATE, StandardOpenOption.WRITE).tryLock();
            if (lock == null) {
                logger.error("Error while opening db. Another process already holds a lock to this db.");
                throw new HaloDBException("Another process already holds a lock for this db.");
            }
            return lock;
        }
        catch (OverlappingFileLockException e) {
            logger.error("Error while opening db. Another process already holds a lock to this db.");
            throw new HaloDBException("Another process already holds a lock for this db.");
        }
        catch (IOException e) {
            logger.error("Error while trying to get a lock on the db.", (Throwable)e);
            throw new HaloDBException("Error while trying to get a lock on the db.", e);
        }
    }

    DBDirectory getDbDirectory() {
        return this.dbDirectory;
    }

    Set<Integer> listDataFileIds() {
        return new HashSet<Integer>(this.readFileMap.keySet());
    }

    boolean isRecordFresh(byte[] key, InMemoryIndexMetaData metaData) {
        InMemoryIndexMetaData currentMeta = this.inMemoryIndex.get(key);
        return currentMeta != null && metaData.getFileId() == currentMeta.getFileId() && metaData.getValueOffset() == currentMeta.getValueOffset();
    }

    private long getNextSequenceNumber() {
        return this.nextSequenceNumber++;
    }

    private int getCurrentWriteFileId() {
        return this.currentWriteFile != null ? this.currentWriteFile.getFileId() : -1;
    }

    private static void checkIfOptionsAreCorrect(HaloDBOptions options) {
        if (options.isUseMemoryPool() && (options.getFixedKeySize() < 0 || options.getFixedKeySize() > 127)) {
            throw new IllegalArgumentException("fixedKeySize must be set and should be less than 128 when using memory pool");
        }
    }

    boolean isClosing() {
        return this.isClosing;
    }

    HaloDBStats stats() {
        OffHeapHashTableStats stats = this.inMemoryIndex.stats();
        return new HaloDBStats(this.statsResetTime, stats.getSize(), this.compactionManager.isCompactionRunning(), this.compactionManager.noOfFilesPendingCompaction(), this.computeStaleDataMapForStats(), stats.getRehashCount(), this.inMemoryIndex.getNoOfSegments(), this.inMemoryIndex.getMaxSizeOfEachSegment(), stats.getSegmentStats(), this.dbDirectory.listDataFiles().length, this.dbDirectory.listTombstoneFiles().length, this.noOfTombstonesFoundDuringOpen.get(), this.options.isCleanUpTombstonesDuringOpen() ? this.noOfTombstonesFoundDuringOpen.get() - this.noOfTombstonesCopiedDuringOpen.get() : 0L, this.compactionManager.getNumberOfRecordsCopied(), this.compactionManager.getNumberOfRecordsReplaced(), this.compactionManager.getNumberOfRecordsScanned(), this.compactionManager.getSizeOfRecordsCopied(), this.compactionManager.getSizeOfFilesDeleted(), this.compactionManager.getSizeOfFilesDeleted() - this.compactionManager.getSizeOfRecordsCopied(), this.compactionManager.getCompactionJobRateSinceBeginning(), this.options.clone());
    }

    synchronized void resetStats() {
        this.inMemoryIndex.resetStats();
        this.compactionManager.resetStats();
        this.statsResetTime = System.currentTimeMillis();
    }

    private Map<Integer, Double> computeStaleDataMapForStats() {
        HashMap<Integer, Double> stats = new HashMap<Integer, Double>();
        this.staleDataPerFileMap.forEach((fileId, staleData) -> {
            HaloDBFile file = this.readFileMap.get(fileId);
            if (file != null && file.getSize() > 0L) {
                double stalePercent = 1.0 * (double)staleData.intValue() / (double)file.getSize() * 100.0;
                stats.put((Integer)fileId, stalePercent);
            }
        });
        return stats;
    }

    @VisibleForTesting
    boolean isCompactionComplete() {
        return this.compactionManager.isCompactionComplete();
    }

    @VisibleForTesting
    boolean isTombstoneFilesMerging() {
        return this.isTombstoneFilesMerging;
    }

    class ProcessTombstoneFileTask
    implements Callable<Long> {
        private final TombstoneFile tombstoneFile;

        public ProcessTombstoneFileTask(TombstoneFile tombstoneFile) {
            this.tombstoneFile = tombstoneFile;
        }

        @Override
        public Long call() throws IOException {
            long maxSequenceNumber = -1L;
            this.tombstoneFile.open();
            TombstoneFile rolloverFile = null;
            TombstoneFile.TombstoneFileIterator iterator = this.tombstoneFile.newIterator();
            long count = 0L;
            long active = 0L;
            long copied = 0L;
            while (iterator.hasNext()) {
                TombstoneEntry entry = iterator.next();
                byte[] key = entry.getKey();
                long sequenceNumber = entry.getSequenceNumber();
                maxSequenceNumber = Long.max(sequenceNumber, maxSequenceNumber);
                ++count;
                InMemoryIndexMetaData existing = HaloDBInternal.this.inMemoryIndex.get(key);
                if (existing == null || existing.getSequenceNumber() >= sequenceNumber) continue;
                HaloDBInternal.this.inMemoryIndex.remove(key);
                HaloDBInternal.this.addFileToCompactionQueueIfThresholdCrossed(existing.getFileId(), Utils.getRecordSize(key.length, existing.getValueSize()));
                ++active;
                if (!HaloDBInternal.this.options.isCleanUpTombstonesDuringOpen()) continue;
                rolloverFile = HaloDBInternal.this.rollOverTombstoneFile(entry, rolloverFile);
                rolloverFile.write(entry);
                ++copied;
            }
            logger.debug("Completed scanning tombstone file {}. Found {} tombstones, {} are still active", new Object[]{this.tombstoneFile.getName(), count, active});
            this.tombstoneFile.close();
            if (HaloDBInternal.this.options.isCleanUpTombstonesDuringOpen()) {
                logger.debug("Copied {} out of {} tombstones. Deleting {}", new Object[]{copied, count, this.tombstoneFile.getName()});
                if (rolloverFile != null) {
                    logger.debug("Closing rollover tombstone file {}", (Object)rolloverFile.getName());
                    rolloverFile.flushToDisk();
                    rolloverFile.close();
                }
                this.tombstoneFile.delete();
            }
            HaloDBInternal.this.noOfTombstonesCopiedDuringOpen.addAndGet(copied);
            HaloDBInternal.this.noOfTombstonesFoundDuringOpen.addAndGet(count);
            return maxSequenceNumber;
        }
    }

    class ProcessIndexFileTask
    implements Callable<Long> {
        private final IndexFile indexFile;
        private final int fileId;

        public ProcessIndexFileTask(IndexFile indexFile, int fileId) {
            this.indexFile = indexFile;
            this.fileId = fileId;
        }

        @Override
        public Long call() throws IOException {
            long maxSequenceNumber = -1L;
            this.indexFile.open();
            IndexFile.IndexFileIterator iterator = this.indexFile.newIterator();
            int count = 0;
            int inserted = 0;
            block0: while (iterator.hasNext()) {
                IndexFileEntry indexFileEntry = iterator.next();
                byte[] key = indexFileEntry.getKey();
                int recordOffset = indexFileEntry.getRecordOffset();
                int recordSize = indexFileEntry.getRecordSize();
                long sequenceNumber = indexFileEntry.getSequenceNumber();
                maxSequenceNumber = Long.max(sequenceNumber, maxSequenceNumber);
                int valueOffset = Utils.getValueOffset(recordOffset, key);
                int valueSize = recordSize - (18 + key.length);
                ++count;
                InMemoryIndexMetaData metaData = new InMemoryIndexMetaData(this.fileId, valueOffset, valueSize, sequenceNumber);
                if (!HaloDBInternal.this.inMemoryIndex.putIfAbsent(key, metaData)) {
                    InMemoryIndexMetaData existing;
                    do {
                        if ((existing = HaloDBInternal.this.inMemoryIndex.get(key)).getSequenceNumber() < sequenceNumber) continue;
                        HaloDBInternal.this.addFileToCompactionQueueIfThresholdCrossed(this.fileId, recordSize);
                        continue block0;
                    } while (!HaloDBInternal.this.inMemoryIndex.replace(key, existing, metaData));
                    HaloDBInternal.this.addFileToCompactionQueueIfThresholdCrossed(existing.getFileId(), Utils.getRecordSize(key.length, existing.getValueSize()));
                    ++inserted;
                    continue;
                }
                ++inserted;
            }
            logger.debug("Completed scanning index file {}. Found {} records, inserted {} records", new Object[]{this.fileId, count, inserted});
            this.indexFile.close();
            return maxSequenceNumber;
        }
    }
}

