/*
 * Decompiled with CFR 0.152.
 */
package com.oath.halodb;

import com.google.common.primitives.Ints;
import com.oath.halodb.Constants;
import com.oath.halodb.DBDirectory;
import com.oath.halodb.HaloDBException;
import com.oath.halodb.HaloDBOptions;
import com.oath.halodb.InMemoryIndexMetaData;
import com.oath.halodb.IndexFile;
import com.oath.halodb.IndexFileEntry;
import com.oath.halodb.Record;
import com.oath.halodb.Utils;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Iterator;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class HaloDBFile {
    private static final Logger logger = LoggerFactory.getLogger(HaloDBFile.class);
    private volatile int writeOffset;
    private FileChannel channel;
    private final File backingFile;
    private final DBDirectory dbDirectory;
    private final int fileId;
    private IndexFile indexFile;
    private final HaloDBOptions options;
    private long unFlushedData = 0L;
    static final String DATA_FILE_NAME = ".data";
    static final String COMPACTED_DATA_FILE_NAME = ".datac";
    private final FileType fileType;

    private HaloDBFile(int fileId, File backingFile, DBDirectory dbDirectory, IndexFile indexFile, FileType fileType, FileChannel channel, HaloDBOptions options) throws IOException {
        this.fileId = fileId;
        this.backingFile = backingFile;
        this.dbDirectory = dbDirectory;
        this.indexFile = indexFile;
        this.fileType = fileType;
        this.channel = channel;
        this.writeOffset = Ints.checkedCast((long)channel.size());
        this.options = options;
    }

    byte[] readFromFile(int offset, int length) throws IOException {
        byte[] value = new byte[length];
        ByteBuffer valueBuf = ByteBuffer.wrap(value);
        int read = this.readFromFile((long)offset, valueBuf);
        assert (read == length);
        return value;
    }

    int readFromFile(long position, ByteBuffer destinationBuffer) throws IOException {
        int bytesRead;
        long currentPosition = position;
        do {
            bytesRead = this.channel.read(destinationBuffer, currentPosition);
            currentPosition += (long)bytesRead;
        } while (bytesRead != -1 && destinationBuffer.hasRemaining());
        return (int)(currentPosition - position);
    }

    private Record readRecord(int offset) throws HaloDBException, IOException {
        long tempOffset = offset;
        ByteBuffer headerBuf = ByteBuffer.allocate(18);
        int readSize = this.readFromFile((long)offset, headerBuf);
        if (readSize != 18) {
            throw new HaloDBException("Corrupted header at " + offset + " in file " + this.fileId);
        }
        tempOffset += (long)readSize;
        Record.Header header = Record.Header.deserialize(headerBuf);
        if (!Record.Header.verifyHeader(header)) {
            throw new HaloDBException("Corrupted header at " + offset + " in file " + this.fileId);
        }
        ByteBuffer recordBuf = ByteBuffer.allocate(header.getKeySize() + header.getValueSize());
        readSize = this.readFromFile(tempOffset, recordBuf);
        if (readSize != recordBuf.capacity()) {
            throw new HaloDBException("Corrupted record at " + offset + " in file " + this.fileId);
        }
        Record record = Record.deserialize(recordBuf, header.getKeySize(), header.getValueSize());
        record.setHeader(header);
        int valueOffset = offset + 18 + header.getKeySize();
        record.setRecordMetaData(new InMemoryIndexMetaData(this.fileId, valueOffset, header.getValueSize(), header.getSequenceNumber()));
        return record;
    }

    InMemoryIndexMetaData writeRecord(Record record) throws IOException {
        this.writeToChannel(record.serialize());
        int recordSize = record.getRecordSize();
        int recordOffset = this.writeOffset;
        this.writeOffset += recordSize;
        IndexFileEntry indexFileEntry = new IndexFileEntry(record.getKey(), recordSize, recordOffset, record.getSequenceNumber(), 0, -1L);
        this.indexFile.write(indexFileEntry);
        int valueOffset = Utils.getValueOffset(recordOffset, record.getKey());
        return new InMemoryIndexMetaData(this.fileId, valueOffset, record.getValue().length, record.getSequenceNumber());
    }

    void rebuildIndexFile() throws IOException {
        this.indexFile.delete();
        this.indexFile = new IndexFile(this.fileId, this.dbDirectory, this.options);
        this.indexFile.create();
        HaloDBFileIterator iterator = new HaloDBFileIterator();
        int offset = 0;
        while (iterator.hasNext()) {
            Record record = iterator.next();
            IndexFileEntry indexFileEntry = new IndexFileEntry(record.getKey(), record.getRecordSize(), offset, record.getSequenceNumber(), 0, -1L);
            this.indexFile.write(indexFileEntry);
            offset += record.getRecordSize();
        }
    }

    HaloDBFile repairFile(DBDirectory dbDirectory) throws IOException {
        HaloDBFile repairFile = this.createRepairFile();
        logger.info("Repairing file {}.", (Object)this.getName());
        HaloDBFileIterator iterator = new HaloDBFileIterator();
        int count = 0;
        while (iterator.hasNext()) {
            Record record = iterator.next();
            if (record != null && record.verifyChecksum()) {
                repairFile.writeRecord(record);
                ++count;
                continue;
            }
            logger.info("Found a corrupted record after copying {} records", (Object)count);
            break;
        }
        logger.info("Recovered {} records from file {} with size {}. Size after repair {}.", new Object[]{count, this.getName(), this.getSize(), repairFile.getSize()});
        repairFile.flushToDisk();
        repairFile.indexFile.flushToDisk();
        Files.move(repairFile.indexFile.getPath(), this.indexFile.getPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        Files.move(repairFile.getPath(), this.getPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        dbDirectory.syncMetaData();
        repairFile.close();
        this.close();
        return HaloDBFile.openForReading(dbDirectory, this.getPath().toFile(), this.fileType, this.options);
    }

    private HaloDBFile createRepairFile() throws IOException {
        File repairFile = this.dbDirectory.getPath().resolve(this.getName() + ".repair").toFile();
        while (!repairFile.createNewFile()) {
            logger.info("Repair file {} already exists, probably from a previous repair which failed. Deleting and trying again", (Object)repairFile.getName());
            repairFile.delete();
        }
        FileChannel channel = new RandomAccessFile(repairFile, "rw").getChannel();
        IndexFile indexFile = new IndexFile(this.fileId, this.dbDirectory, this.options);
        indexFile.createRepairFile();
        return new HaloDBFile(this.fileId, repairFile, this.dbDirectory, indexFile, this.fileType, channel, this.options);
    }

    private long writeToChannel(ByteBuffer[] buffers) throws IOException {
        long written;
        long toWrite = 0L;
        for (ByteBuffer buffer : buffers) {
            toWrite += (long)buffer.remaining();
        }
        for (written = 0L; written < toWrite; written += this.channel.write(buffers)) {
        }
        this.unFlushedData += written;
        if (this.options.isSyncWrite() || this.options.getFlushDataSizeBytes() != -1L && this.unFlushedData > this.options.getFlushDataSizeBytes()) {
            this.flushToDisk();
            this.unFlushedData = 0L;
        }
        return written;
    }

    void flushToDisk() throws IOException {
        if (this.channel != null && this.channel.isOpen()) {
            this.channel.force(true);
        }
    }

    long getWriteOffset() {
        return this.writeOffset;
    }

    void setWriteOffset(int writeOffset) {
        this.writeOffset = writeOffset;
    }

    long getSize() {
        return this.writeOffset;
    }

    IndexFile getIndexFile() {
        return this.indexFile;
    }

    FileChannel getChannel() {
        return this.channel;
    }

    FileType getFileType() {
        return this.fileType;
    }

    int getFileId() {
        return this.fileId;
    }

    static HaloDBFile openForReading(DBDirectory dbDirectory, File filename, FileType fileType, HaloDBOptions options) throws IOException {
        int fileId = HaloDBFile.getFileTimeStamp(filename);
        FileChannel channel = new RandomAccessFile(filename, "r").getChannel();
        IndexFile indexFile = new IndexFile(fileId, dbDirectory, options);
        indexFile.open();
        return new HaloDBFile(fileId, filename, dbDirectory, indexFile, fileType, channel, options);
    }

    static HaloDBFile create(DBDirectory dbDirectory, int fileId, HaloDBOptions options, FileType fileType) throws IOException {
        BiFunction<DBDirectory, Integer, File> toFile = fileType == FileType.DATA_FILE ? HaloDBFile::getDataFile : HaloDBFile::getCompactedDataFile;
        File file = toFile.apply(dbDirectory, fileId);
        while (!file.createNewFile()) {
            file = toFile.apply(dbDirectory, ++fileId);
        }
        FileChannel channel = new RandomAccessFile(file, "rw").getChannel();
        IndexFile indexFile = new IndexFile(fileId, dbDirectory, options);
        indexFile.create();
        return new HaloDBFile(fileId, file, dbDirectory, indexFile, fileType, channel, options);
    }

    HaloDBFileIterator newIterator() throws IOException {
        return new HaloDBFileIterator();
    }

    void close() throws IOException {
        if (this.channel != null) {
            this.channel.close();
        }
        if (this.indexFile != null) {
            this.indexFile.close();
        }
    }

    void delete() throws IOException {
        this.close();
        if (this.backingFile != null) {
            this.backingFile.delete();
        }
        if (this.indexFile != null) {
            this.indexFile.delete();
        }
    }

    String getName() {
        return this.backingFile.getName();
    }

    Path getPath() {
        return this.backingFile.toPath();
    }

    private static File getDataFile(DBDirectory dbDirectory, int fileId) {
        return dbDirectory.getPath().resolve(fileId + DATA_FILE_NAME).toFile();
    }

    private static File getCompactedDataFile(DBDirectory dbDirectory, int fileId) {
        return dbDirectory.getPath().resolve(fileId + COMPACTED_DATA_FILE_NAME).toFile();
    }

    static FileType findFileType(File file) {
        String name = file.getName();
        return name.endsWith(COMPACTED_DATA_FILE_NAME) ? FileType.COMPACTED_FILE : FileType.DATA_FILE;
    }

    static int getFileTimeStamp(File file) {
        Matcher matcher = Constants.DATA_FILE_PATTERN.matcher(file.getName());
        matcher.find();
        String s = matcher.group(1);
        return Integer.parseInt(s);
    }

    static enum FileType {
        DATA_FILE,
        COMPACTED_FILE;

    }

    class HaloDBFileIterator
    implements Iterator<Record> {
        private final int endOffset;
        private int currentOffset = 0;

        HaloDBFileIterator() throws IOException {
            this.endOffset = Ints.checkedCast((long)HaloDBFile.this.channel.size());
        }

        @Override
        public boolean hasNext() {
            return this.currentOffset < this.endOffset;
        }

        @Override
        public Record next() {
            Record record;
            try {
                record = HaloDBFile.this.readRecord(this.currentOffset);
            }
            catch (HaloDBException | IOException e) {
                logger.error("Error in iterator", (Throwable)e);
                this.currentOffset = this.endOffset;
                return null;
            }
            this.currentOffset += record.getRecordSize();
            return record;
        }
    }
}

