/*
 * Decompiled with CFR 0.152.
 */
package com.oath.halodb;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.RateLimiter;
import com.oath.halodb.HaloDBFile;
import com.oath.halodb.HaloDBInternal;
import com.oath.halodb.InMemoryIndexMetaData;
import com.oath.halodb.IndexFile;
import com.oath.halodb.IndexFileEntry;
import com.oath.halodb.Utils;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class CompactionManager {
    private static final Logger logger = LoggerFactory.getLogger(CompactionManager.class);
    private final HaloDBInternal dbInternal;
    private volatile boolean isRunning = false;
    private final RateLimiter compactionRateLimiter;
    private volatile HaloDBFile currentWriteFile = null;
    private int currentWriteFileOffset = 0;
    private final BlockingQueue<Integer> compactionQueue;
    private volatile CompactionThread compactionThread;
    private volatile long numberOfRecordsCopied = 0L;
    private volatile long numberOfRecordsReplaced = 0L;
    private volatile long numberOfRecordsScanned = 0L;
    private volatile long sizeOfRecordsCopied = 0L;
    private volatile long sizeOfFilesDeleted = 0L;
    private volatile long totalSizeOfRecordsCopied = 0L;
    private volatile long compactionStartTime = System.currentTimeMillis();
    private static final int STOP_SIGNAL = -10101;
    private final ReentrantLock startStopLock = new ReentrantLock();
    private volatile boolean stopInProgress = false;

    CompactionManager(HaloDBInternal dbInternal) {
        this.dbInternal = dbInternal;
        this.compactionRateLimiter = RateLimiter.create((double)dbInternal.options.getCompactionJobRate());
        this.compactionQueue = new LinkedBlockingQueue<Integer>();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean stopCompactionThread(boolean closeCurrentWriteFile) throws IOException {
        this.stopInProgress = true;
        this.startStopLock.lock();
        try {
            this.isRunning = false;
            if (this.isCompactionRunning()) {
                this.compactionQueue.put(-10101);
                this.compactionThread.join();
                if (closeCurrentWriteFile && this.currentWriteFile != null) {
                    this.currentWriteFile.flushToDisk();
                    this.currentWriteFile.getIndexFile().flushToDisk();
                    this.currentWriteFile.close();
                }
            }
        }
        catch (InterruptedException e) {
            logger.error("Error while waiting for compaction thread to stop", (Throwable)e);
            boolean bl = false;
            return bl;
        }
        finally {
            this.stopInProgress = false;
            this.startStopLock.unlock();
        }
        return true;
    }

    void startCompactionThread() {
        this.startStopLock.lock();
        try {
            if (!this.isCompactionRunning()) {
                this.isRunning = true;
                this.compactionThread = new CompactionThread();
                this.compactionThread.start();
            }
        }
        finally {
            this.startStopLock.unlock();
        }
    }

    void pauseCompactionThread() throws IOException {
        logger.info("Pausing compaction thread ...");
        this.stopCompactionThread(false);
    }

    void resumeCompaction() {
        logger.info("Resuming compaction thread");
        this.startCompactionThread();
    }

    int getCurrentWriteFileId() {
        return this.currentWriteFile != null ? this.currentWriteFile.getFileId() : -1;
    }

    boolean submitFileForCompaction(int fileId) {
        return this.compactionQueue.offer(fileId);
    }

    int noOfFilesPendingCompaction() {
        return this.compactionQueue.size();
    }

    long getNumberOfRecordsCopied() {
        return this.numberOfRecordsCopied;
    }

    long getNumberOfRecordsReplaced() {
        return this.numberOfRecordsReplaced;
    }

    long getNumberOfRecordsScanned() {
        return this.numberOfRecordsScanned;
    }

    long getSizeOfRecordsCopied() {
        return this.sizeOfRecordsCopied;
    }

    long getSizeOfFilesDeleted() {
        return this.sizeOfFilesDeleted;
    }

    long getCompactionJobRateSinceBeginning() {
        long timeInSeconds = (System.currentTimeMillis() - this.compactionStartTime) / 1000L;
        long rate = 0L;
        if (timeInSeconds > 0L) {
            rate = this.totalSizeOfRecordsCopied / timeInSeconds;
        }
        return rate;
    }

    void resetStats() {
        this.sizeOfFilesDeleted = 0L;
        this.sizeOfRecordsCopied = 0L;
        this.numberOfRecordsScanned = 0L;
        this.numberOfRecordsReplaced = 0L;
        this.numberOfRecordsCopied = 0L;
    }

    boolean isCompactionRunning() {
        return this.compactionThread != null && this.compactionThread.isAlive();
    }

    void forceRolloverCurrentWriteFile() throws IOException {
        if (this.currentWriteFile != null) {
            this.currentWriteFile.flushToDisk();
            this.currentWriteFile.getIndexFile().flushToDisk();
        }
        this.currentWriteFile = this.dbInternal.createHaloDBFile(HaloDBFile.FileType.COMPACTED_FILE);
        this.dbInternal.getDbDirectory().syncMetaData();
        this.currentWriteFileOffset = 0;
    }

    @VisibleForTesting
    synchronized boolean isCompactionComplete() {
        if (!this.isCompactionRunning()) {
            return true;
        }
        if (this.compactionQueue.isEmpty()) {
            try {
                this.isRunning = false;
                this.submitFileForCompaction(-10101);
                this.compactionThread.join();
            }
            catch (InterruptedException e) {
                logger.error("Error in isCompactionComplete", (Throwable)e);
            }
            return true;
        }
        return false;
    }

    private class CompactionThread
    extends Thread {
        private long unFlushedData;

        CompactionThread() {
            super("CompactionThread");
            this.unFlushedData = 0L;
            this.setUncaughtExceptionHandler((t, e) -> {
                logger.error("Compaction thread crashed", e);
                if (CompactionManager.this.currentWriteFile != null) {
                    try {
                        CompactionManager.this.currentWriteFile.flushToDisk();
                    }
                    catch (IOException ex) {
                        logger.error("Error while flushing " + CompactionManager.this.currentWriteFile.getFileId() + " to disk", (Throwable)ex);
                    }
                    CompactionManager.this.currentWriteFile = null;
                }
                CompactionManager.this.currentWriteFileOffset = 0;
                if (!CompactionManager.this.stopInProgress) {
                    CompactionManager.this.startStopLock.lock();
                    try {
                        CompactionManager.this.compactionThread = null;
                        CompactionManager.this.startCompactionThread();
                    }
                    finally {
                        CompactionManager.this.startStopLock.unlock();
                    }
                } else {
                    logger.info("Not restarting thread as the lock is held by stop compaction method.");
                }
            });
        }

        @Override
        public void run() {
            logger.info("Starting compaction thread ...");
            int fileToCompact = -1;
            while (CompactionManager.this.isRunning) {
                try {
                    fileToCompact = (Integer)CompactionManager.this.compactionQueue.take();
                    if (fileToCompact == -10101) {
                        logger.debug("Received a stop signal.");
                        continue;
                    }
                    logger.debug("Compacting {} ...", (Object)fileToCompact);
                    this.copyFreshRecordsToNewFile(fileToCompact);
                    logger.debug("Completed compacting {} to {}", (Object)fileToCompact, (Object)CompactionManager.this.getCurrentWriteFileId());
                    CompactionManager.this.dbInternal.markFileAsCompacted(fileToCompact);
                    CompactionManager.this.dbInternal.deleteHaloDBFile(fileToCompact);
                }
                catch (Exception e) {
                    logger.error(String.format("Error while compacting file %d to %d", fileToCompact, CompactionManager.this.getCurrentWriteFileId()), (Throwable)e);
                }
            }
            logger.info("Compaction thread stopped.");
        }

        private void copyFreshRecordsToNewFile(int idOfFileToCompact) throws IOException {
            HaloDBFile fileToCompact = CompactionManager.this.dbInternal.getHaloDBFile(idOfFileToCompact);
            if (fileToCompact == null) {
                logger.debug("File doesn't exist, was probably compacted already.");
                return;
            }
            FileChannel readFrom = fileToCompact.getChannel();
            IndexFile.IndexFileIterator iterator = fileToCompact.getIndexFile().newIterator();
            long recordsCopied = 0L;
            long recordsScanned = 0L;
            while (iterator.hasNext()) {
                IndexFileEntry indexFileEntry = iterator.next();
                byte[] key = indexFileEntry.getKey();
                long recordOffset = indexFileEntry.getRecordOffset();
                int recordSize = indexFileEntry.getRecordSize();
                ++recordsScanned;
                InMemoryIndexMetaData currentRecordMetaData = CompactionManager.this.dbInternal.getInMemoryIndex().get(key);
                if (!this.isRecordFresh(indexFileEntry, currentRecordMetaData, idOfFileToCompact)) continue;
                ++recordsCopied;
                CompactionManager.this.compactionRateLimiter.acquire(recordSize);
                this.rollOverCurrentWriteFile(recordSize);
                CompactionManager.this.sizeOfRecordsCopied = CompactionManager.this.sizeOfRecordsCopied + (long)recordSize;
                CompactionManager.this.totalSizeOfRecordsCopied = CompactionManager.this.totalSizeOfRecordsCopied + (long)recordSize;
                long transferred = readFrom.transferTo(recordOffset, recordSize, CompactionManager.this.currentWriteFile.getChannel());
                if (transferred != (long)recordSize) {
                    logger.error("Had to transfer {} but only did {}", (Object)recordSize, (Object)transferred);
                }
                this.unFlushedData += transferred;
                if (((CompactionManager)CompactionManager.this).dbInternal.options.getFlushDataSizeBytes() != -1L && this.unFlushedData > ((CompactionManager)CompactionManager.this).dbInternal.options.getFlushDataSizeBytes()) {
                    CompactionManager.this.currentWriteFile.getChannel().force(false);
                    this.unFlushedData = 0L;
                }
                IndexFileEntry newEntry = new IndexFileEntry(key, recordSize, CompactionManager.this.currentWriteFileOffset, indexFileEntry.getSequenceNumber(), indexFileEntry.getVersion(), -1L);
                CompactionManager.this.currentWriteFile.getIndexFile().write(newEntry);
                int valueOffset = Utils.getValueOffset(CompactionManager.this.currentWriteFileOffset, key);
                InMemoryIndexMetaData newMetaData = new InMemoryIndexMetaData(CompactionManager.this.currentWriteFile.getFileId(), valueOffset, currentRecordMetaData.getValueSize(), indexFileEntry.getSequenceNumber());
                boolean updated = CompactionManager.this.dbInternal.getInMemoryIndex().replace(key, currentRecordMetaData, newMetaData);
                if (updated) {
                    CompactionManager.this.numberOfRecordsReplaced++;
                } else {
                    CompactionManager.this.dbInternal.addFileToCompactionQueueIfThresholdCrossed(CompactionManager.this.currentWriteFile.getFileId(), recordSize);
                }
                CompactionManager.this.currentWriteFileOffset = CompactionManager.this.currentWriteFileOffset + recordSize;
                CompactionManager.this.currentWriteFile.setWriteOffset(CompactionManager.this.currentWriteFileOffset);
            }
            if (recordsCopied > 0L) {
                CompactionManager.this.currentWriteFile.flushToDisk();
            }
            CompactionManager.this.numberOfRecordsCopied = CompactionManager.this.numberOfRecordsCopied + recordsCopied;
            CompactionManager.this.numberOfRecordsScanned = CompactionManager.this.numberOfRecordsScanned + recordsScanned;
            CompactionManager.this.sizeOfFilesDeleted = CompactionManager.this.sizeOfFilesDeleted + fileToCompact.getSize();
            logger.debug("Scanned {} records in file {} and copied {} records to {}.datac", new Object[]{recordsScanned, idOfFileToCompact, recordsCopied, CompactionManager.this.getCurrentWriteFileId()});
        }

        private boolean isRecordFresh(IndexFileEntry entry, InMemoryIndexMetaData metaData, int idOfFileToMerge) {
            return metaData != null && metaData.getFileId() == idOfFileToMerge && metaData.getValueOffset() == Utils.getValueOffset(entry.getRecordOffset(), entry.getKey());
        }

        private void rollOverCurrentWriteFile(int recordSize) throws IOException {
            if (CompactionManager.this.currentWriteFile == null || CompactionManager.this.currentWriteFileOffset + recordSize > ((CompactionManager)CompactionManager.this).dbInternal.options.getMaxFileSize()) {
                CompactionManager.this.forceRolloverCurrentWriteFile();
            }
        }
    }
}

