package com.vhall.rtc.vrtc;

import android.content.Context;
import android.media.projection.MediaProjection;
import android.text.TextUtils;
import com.vhall.framework.common.ICallback;
import com.vhall.ilss.VHRoomListener;
import com.vhall.logmanager.VLog;
import com.vhall.rtc.Utils;
import com.vhall.rtc.VRTCCode;
import com.vhall.rtc.VRTCParams;
import com.vhall.rtc.VhallRTC;
import com.vhall.rtc.absrtc.ISwitchFocusModeListener;
import com.vhall.rtc.absrtc.IVHRTC;
import com.vhall.rtc.absrtc.IVHRTCRenderView;
import com.vhall.vhallrtc.client.FinishCallback;
import com.vhall.vhallrtc.client.IFocusModeChangeListener;
import com.vhall.vhallrtc.client.Room;
import com.vhall.vhallrtc.client.Stream;
import com.vhall.vhallrtc.client.VHRenderView;
import com.vhall.vhallrtc.common.AppRTCAudioManager;
import org.json.JSONException;
import org.json.JSONObject;
import org.vhwebrtc.SurfaceViewRenderer;
import java.util.List;

/**
 * @author：vhall Email：jooperge@163.com
 * 描述：
 * 修改历史:
 * <p>
 * 创建于： 2022/10/18
 */
public class VhallVRTCImpl implements IVHRTC {

    private final String TAG = "VhallVRTC";
    private Context mContext;
    private VhallRTC mVhallRTC;
    private Stream mLocalStream;
    private VHRenderView mLocalRenderView;
    /**
     * 进房成功后是否自动上麦/订阅
     */
    private boolean mAutoPublish, mAutoSubscribe;
    private VHRoomListener mRoomListener = null;
    private Room mRoom = null;
    /**
     * 摄像头是否为前置
     */
    private boolean mCameraFacingFront = true;

    /**
     * 混流配置
     */
    private JSONObject mMixConfig = null;
    private JSONObject _bgColorJSON = null;
    private JSONObject _boardJSON = null;

    public VhallVRTCImpl(Context context, VRTCParams.VRTCRoomParamsInner roomParams) {
        mContext = context;
        mVhallRTC = new VhallRTC(context, roomParams.softEchoCanceller, true);
    }

    @Override
    public void enterRoom(VRTCParams.VRTCRoomParamsInner roomParams, boolean autoPublish, boolean autoSubscribe, String attribute, VHRoomListener roomListener) {
        if (null != mVhallRTC) {
            mAutoPublish = autoPublish;
            mAutoSubscribe = autoSubscribe;
            mRoomListener = roomListener;
            mVhallRTC.setListener(new VRTCRoomDelegate());
            mVhallRTC.setCameraListener(new VhallRTC.IOnCameraListener() {
                @Override
                public void onCameraError(String err) {
                    if (null != mRoomListener) {
                        mRoomListener.onCameraError(0, err);
                    }
                }

                @Override
                public void onFirstFrame() {
                    if (null != mRoomListener) {
                        mRoomListener.onFirstFrameAvailable(true, null, 0, 0);
                    }
                }
            });
            mVhallRTC.enterRoom(roomParams.inavToken, attribute);
        } else {
            VLog.e(TAG, "please init first");
        }
    }

    @Override
    public void startLocalPreview(IVHRTCRenderView renderView, VRTCParams.VRTCVideoEncodeParam videoParam, VRTCParams.VRTCStreamParam streamParam, boolean isFrontCamera) {
        mLocalStream = initLocalStream(videoParam, streamParam);
        mLocalRenderView = initRenderView(renderView);

        if (null != mLocalRenderView) {//需要预览才处理
            mLocalStream.removeAllRenderView();
            mLocalStream.addRenderView(mLocalRenderView);
        }
    }

    @Override
    public void startRemoteView(IVHRTCRenderView renderView, String streamId) {
        VHRenderView remoteRenderView = initRenderView(renderView);
        if (null != remoteRenderView && null != mRoom) {
            Stream remoteStream = mRoom.getRemoteStreamById(streamId);
            if (null != remoteStream) {
                remoteStream.removeAllRenderView();
                remoteStream.addRenderView(remoteRenderView);
            }
        } else {
            VLog.e(TAG, "Room#startRemoteView Room is null");
        }
    }

    @Override
    public void createScreenShareStream(MediaProjection mediaProjection, VRTCParams.VRTCVideoEncodeParam videoParam, VRTCParams.VRTCStreamParam streamParam) {
        if (null == mediaProjection) {
            VLog.e(TAG, "createScreenShareStream erroor, mediaProjection is null!");
            return;
        }

        VRTCParams.VRTCVideoEncodeParam encodeParam;
        if (null == videoParam) {//使用流媒体推荐参数配置
            encodeParam = new VRTCParams.VRTCVideoEncodeParam();
            encodeParam.videoResolution = VRTCCode.RTCVideoProfile.PROFILE_1280_720_16x9;
            encodeParam.videoFps = 5;
            encodeParam.minVideoBitrate = 800;
            encodeParam.videoBitrate = 900;
            encodeParam.maxVideoBitrate = 1000;

            encodeParam.resolutionStrategy = VRTCCode.RTCDegradationMode.BALANCED;
        } else {
            encodeParam = videoParam;
        }

        mLocalStream = initScreenShareStream(mediaProjection, encodeParam, streamParam);
    }

    @Override
    public void stopScreenShare() {
        leaveRoom();
    }

    @Override
    public String getUserIdByStreamId(String streamId) {
        Stream stream = mRoom.getRemoteStreamById(streamId);
        return null != stream ? stream.userId : null;
    }

    @Override
    public void publish() {
        if (null != mVhallRTC) {
            mVhallRTC.publish();
        }
    }

    @Override
    public void unPublish() {
        if (null != mVhallRTC) {
            mVhallRTC.unpublish();
        }
    }

    @Override
    public void subscribe(String streamId) {
        if (null != mRoom) {
            Stream targetStream = mRoom.getRemoteStreamById(streamId);
            if (null != targetStream) {
                mRoom.subscribe(targetStream);
            }
        }
    }

    @Override
    public void unSubscribe(String streamId) {
        if (null != mRoom) {
            Stream targetStream = mRoom.getRemoteStreamById(streamId);
            if (null != targetStream) {
                mRoom.unsubscribe(targetStream);
            }
        }
    }

    @Override
    public void leaveRoom() {
        if (null != mVhallRTC) {
            mVhallRTC.leaveRoom();
        }
    }

    @Override
    public void muteVideo(String streamId, boolean mute) {
        Stream targetStream = null;
        if (TextUtils.isEmpty(streamId)) {
            targetStream = mLocalStream;
        } else {
            if (null != mRoom) {
                targetStream = mRoom.getRemoteStreamById(streamId);
            }
        }
        if (null != targetStream) {
            if (mute) {
                targetStream.muteVideo(new FinishCallback() {
                    @Override
                    public void onFinish(int i, String s) {
                        VLog.d(TAG, "muteVideo, code:" + i + ", message:" + s);
                    }
                });
            } else {
                targetStream.unmuteVideo(new FinishCallback() {
                    @Override
                    public void onFinish(int i, String s) {
                        VLog.d(TAG, "unmuteVideo, code:" + i + ", message:" + s);
                    }
                });
            }
        } else {
            VLog.e(TAG, "target stream does not exists");
        }
    }

    @Override
    public void muteAudio(String streamId, boolean mute) {
        Stream targetStream = null;
        if (TextUtils.isEmpty(streamId)) {
            targetStream = mLocalStream;
        } else {
            if (null != mRoom) {
                targetStream = mRoom.getRemoteStreamById(streamId);
            }
        }
        if (null != targetStream) {
            if (mute) {
                targetStream.muteAudio(new FinishCallback() {
                    @Override
                    public void onFinish(int i, String s) {
                        VLog.d(TAG, "muteAudio(, code:" + i + ", message:" + s);
                    }
                });
            } else {
                targetStream.unmuteAudio(new FinishCallback() {
                    @Override
                    public void onFinish(int i, String s) {
                        VLog.d(TAG, "unmuteAudio(, code:" + i + ", message:" + s);
                    }
                });
            }
        } else {
            VLog.e(TAG, "target stream does not exists");
        }
    }

    @Override
    public void switchCamera() {
        mCameraFacingFront = !mCameraFacingFront;
        if (null != mLocalStream) {
            mLocalStream.switchCamera();
        }
    }

    @Override
    public boolean isFrontCamera() {
        return mCameraFacingFront;
    }

    @Override
    public void setMirror(boolean isMirror) {
        if (null != mLocalRenderView) {
            mLocalRenderView.setMirror(isMirror);
        }
    }

    @Override
    public void switchAutoFocusMode(boolean enableAuto, ISwitchFocusModeListener switchFocusModeListener) {
        if (null != mLocalStream) {
            mLocalStream.switchAutoFocusMode(enableAuto, new IFocusModeChangeListener() {
                @Override
                public void onComplete(boolean result, String errMsg) {
                    if (null != switchFocusModeListener) {
                        switchFocusModeListener.onComplete(result, errMsg);
                    }
                }
            });
        }
    }

    @Override
    public void release() {
        mVhallRTC.setListener(null);
        if (null != mVhallRTC) {
            mVhallRTC.release();
            mVhallRTC = null;
        }
        mLocalStream = null;
        if (null != mLocalRenderView) {
            mLocalRenderView.release();
            mLocalRenderView = null;
        }
    }

    @Override
    public void setDataReport(JSONObject data) {
        if (null != mRoom) {
            mRoom.setReportLogData(data);
        }
    }

    @Override
    public void setMixBackgroundImage(VRTCParams.VRTCBroadcastParamInner broadcastParam, String url, int cropType, ICallback callback) {
        if (null != mVhallRTC) {
            FinishCallback finishCallback = new FinishCallback() {
                @Override
                public void onFinish(int code, String msg) {
                    if (null != callback) {
                        if (200 == code) {
                            callback.onSuccess();
                        } else {
                            callback.onFailure(code, msg);
                        }
                    }
                }
            };
            if (!TextUtils.isEmpty(url)) {
                mVhallRTC.setRoomBroadCastBackgroundImage(url, cropType, finishCallback);
            } else {
                mVhallRTC.resetRoomBroadCastBackgroundImage(finishCallback);
            }
        }
    }

    @Override
    public void setMixPlaceholderImage(String url, ICallback callback) {
        if (null != mVhallRTC) {
            FinishCallback finishCallback = new FinishCallback() {
                @Override
                public void onFinish(int code, String msg) {
                    if (null != callback) {
                        if (200 == code) {
                            callback.onSuccess();
                        } else {
                            callback.onFailure(code, msg);
                        }
                    }
                }
            };
            if (!TextUtils.isEmpty(url)) {
                mVhallRTC.setRoomBroadCastPlaceholderImage(url, finishCallback);
            } else {
                mVhallRTC.resetRoomBroadCastPlaceholderImage(finishCallback);
            }
        }
    }

    @Override
    public void docCloudRender(String appId, String channelId, boolean start, ICallback callback) {
        if (null != mRoom) {
            JSONObject config = new JSONObject();
            try {
                config.put("appId", appId);
                config.put("channelId", channelId);

                FinishCallback finishCallback = new FinishCallback() {
                    @Override
                    public void onFinish(int code, String msg) {
                        if (null != callback) {
                            if (200 == code) {
                                callback.onSuccess();
                            } else {
                                callback.onFailure(code, msg);
                            }
                        }
                    }
                };

                if (start) {
                    mRoom.startDocCloudRender(config, finishCallback);
                } else {
                    mRoom.stopDocCloudRender(config, finishCallback);
                }
            } catch (JSONException exception) {
                exception.printStackTrace();
            }
        }
    }

    @Override
    public void setScaleType(int scaleType) {
        SurfaceViewRenderer.VHRenderViewScalingMode scalingMode;
        switch (scaleType) {
            default:
            case 0:
                scalingMode = SurfaceViewRenderer.VHRenderViewScalingMode.kVHRenderViewScalingModeNone;
                break;
            case 1:
                scalingMode = SurfaceViewRenderer.VHRenderViewScalingMode.kVHRenderViewScalingModeAspectFit;
                break;
            case 2:
                scalingMode = SurfaceViewRenderer.VHRenderViewScalingMode.kVHRenderViewScalingModeAspectFill;
                break;
        }
        if (null != mLocalRenderView) {
            mLocalRenderView.setScalingMode(scalingMode);
        }
    }

    @Override
    public void setVideoResolution(int videoProfile) {
        if (null != mLocalStream) {
            VRTCParams.VRTCVideoEncodeParam encodeParam = new VRTCParams.VRTCVideoEncodeParam();
            encodeParam.videoResolution = videoProfile;
            VRTCVideoEncodeCompat.compat(encodeParam);
            mLocalStream.ChangeCarameFormat(encodeParam.videoWidth, encodeParam.videoHeight, encodeParam.videoFps);
        }
    }

    @Override
    public void setDefaultAudioDevice(Object object) {
        if (null != mVhallRTC && object instanceof AppRTCAudioManager.AudioDevice) {
            mVhallRTC.setDefaultAudioDevice((AppRTCAudioManager.AudioDevice) object);
        }
    }

    @Override
    public void broadcastRoom(VRTCParams.VRTCBroadcastParamInner broadcastParam, String accessToken, ICallback callback) {
        if (null == broadcastParam || null == broadcastParam.broadcastParam) {
            stopBroadcastRoom(callback);
        } else {
            startBroadcastRoom(broadcastParam, callback);
        }
    }

    @Override
    public void setMixLayoutMainScreen(String streamId, ICallback callback) {
        if (null != mRoom && null != mLocalStream) {
            mRoom.setMixLayoutMainScreen(TextUtils.isEmpty(streamId) ? mLocalStream.streamId : streamId, new FinishCallback() {
                @Override
                public void onFinish(int code, String message) {
                    if (null != callback) {
                        if (code != 200) {
                            callback.onFailure(code, message);
                        } else {
                            callback.onSuccess();
                        }
                    }
                }
            });
        }
    }

    @Override
    public void setMixLayoutMode(VRTCParams.VRTCBroadcastParamInner broadcastParam, int layoutMode, ICallback callback) {
        if (null != mVhallRTC) {
            mVhallRTC.setMixLayoutMode(layoutMode, null, new FinishCallback() {
                @Override
                public void onFinish(int code, String message) {
                    if (null != callback) {
                        if (code != 200) {
                            callback.onFailure(code, message);
                        } else {
                            callback.onSuccess();
                        }
                    }
                }
            });

            //refresh local config cache
            if (null != mMixConfig) {
                if (layoutMode >= 1000) {
                    try {
                        mMixConfig.remove("layoutMode");
                        mMixConfig.put("adaptiveLayoutMode", layoutMode - 1000);
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                } else {
                    try {
                        mMixConfig.put("layoutMode", layoutMode);
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    @Override
    public void setMixBackgroundColor(VRTCParams.VRTCBroadcastParamInner broadcastParam, String hexColor, ICallback callback) {
        if (null != this.mRoom) {
            if (TextUtils.isEmpty(hexColor)) {
                hexColor = "0x333338";
            }
            try {
                if (null != mMixConfig) {
                    mMixConfig.put("backgroundColor", Utils.compatColor2HexStyle(hexColor));
                    this.mRoom.configRoomBroadCast(mMixConfig, null, new FinishCallback() {
                        @Override
                        public void onFinish(int i, String s) {
                            if (null != callback) {
                                if (200 == i) {
                                    callback.onSuccess();
                                } else {
                                    callback.onFailure(i, s);
                                }
                            }
                        }
                    });
                } else {
                    _bgColorJSON = new JSONObject();
                    _bgColorJSON.put("backgroundColor", Utils.compatColor2HexStyle(hexColor));
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        } else {
            VLog.d(TAG, "Room is null");
        }
    }

    @Override
    public void setMixBorderColor(VRTCParams.VRTCBroadcastParam.Border border, ICallback callback) {
        if (null != border) {
            setRoomBroadCastBorderColor(border, callback);
        } else {
            setRoomBroadCastBorderColor(null, callback);
        }
    }

    /**
     * 设置旁路背景色
     *
     * @param callback
     */
    private void setRoomBroadCastBorderColor(VRTCParams.VRTCBroadcastParam.Border borderConfig, ICallback callback) {
        boolean enable = null != borderConfig && borderConfig.exist;
        int width = null == borderConfig ? 0 : borderConfig.width;
        String hexColor = null == borderConfig ? null : borderConfig.color;
        boolean transparent = null != borderConfig && borderConfig.transparent;

        if (null != this.mRoom) {
            if (!enable) {
                try {
                    if (null != mMixConfig) {
                        JSONObject border = new JSONObject();
                        border.put("exist", false);
                        mMixConfig.put("border", border);
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            } else {
                if (width > 8) {
                    width = 8;
                } else if (width < 1) {
                    width = 0;
                }
                try {
                    JSONObject border = new JSONObject();
                    border.put("exist", true);
                    border.put("width", width);
                    border.put("color", Utils.compatColor2HexStyle(hexColor));
                    border.put("transparency", transparent ? 100 : 0);

                    if (null != mMixConfig) {
                        mMixConfig.put("border", border);
                    } else {
                        _boardJSON = new JSONObject();
                        _boardJSON.put("border", border);
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            if (null != mMixConfig) {
                this.mRoom.configRoomBroadCast(mMixConfig, null, new FinishCallback() {
                    @Override
                    public void onFinish(int i, String s) {
                        if (null != callback) {
                            if (200 == i) {
                                callback.onSuccess();
                            } else {
                                callback.onFailure(i, s);
                            }
                        }
                    }
                });
            }
        } else {
            VLog.d(TAG, "Room is null");
        }
    }

    /**
     * 开始旁路直播
     *
     * @param callback
     */
    private void startBroadcastRoom(VRTCParams.VRTCBroadcastParamInner broadcastParam, ICallback callback) {
        if (null != mVhallRTC) {
            final JSONObject tempConfig = new JSONObject();
            try {
                tempConfig.put("profile", broadcastParam.compat_profile_json);

                if (broadcastParam.isAdaptiveLayoutMode) {
                    tempConfig.remove("layoutMode");
                    tempConfig.put("adaptiveLayoutMode", String.valueOf(broadcastParam.compat_layout_mode));
                } else {
                    tempConfig.remove("adaptiveLayoutMode");
                    tempConfig.put("layoutMode", String.valueOf(broadcastParam.compat_layout_mode));
                }

                tempConfig.put("layoutMode", broadcastParam.compat_layout_mode);
                tempConfig.put("precast_pic_exist", broadcastParam.broadcastParam.precast_pic_exist);

                String bgColor = Utils.compatColor2HexStyle(broadcastParam.broadcastParam.backgroundColor);
                if (!TextUtils.isEmpty(bgColor)) {
                    tempConfig.put("backgroundColor", broadcastParam.broadcastParam.backgroundColor);
                }

                if (null != broadcastParam.broadcastParam.border) {
                    VRTCParams.VRTCBroadcastParam.Border tempBrder = broadcastParam.broadcastParam.border.copy();
                    tempBrder.color = tempBrder.convertColor2VRTCStyle();
                    tempConfig.put("border", tempBrder.convertToJson());
                }
                tempConfig.put("publishUrl", broadcastParam.pushUrl);

                if (broadcastParam.broadcastParam.nickName) {
                    JSONObject nickNameConfig = new JSONObject();
                    nickNameConfig.put("display", true);
                    tempConfig.put("nickName", nickNameConfig);
                }
                tempConfig.put("crop_type", broadcastParam.broadcastParam.cropType);

                boundPreConfigColor(tempConfig);
                mVhallRTC.configRoomBroadCast(tempConfig, null, new FinishCallback() {
                    @Override
                    public void onFinish(int code, String message) {
                        mMixConfig = tempConfig;
                        if (code != 200) {
                            if (callback != null)
                                callback.onFailure(code, message);
                        } else {
                            if (callback != null) {
                                callback.onSuccess();
                            }
                        }
                    }
                });
            } catch (JSONException e) {
                e.printStackTrace();
                if (callback != null)
                    callback.onFailure(VRTCCode.ERR_BROADCAST_CONFIG_JSONEX, e.getMessage());
            }
        } else {
            VLog.d(TAG, "please init VHInteractiveV2 first");
        }
    }

    /**
     * 兼容处理开始旁路直播前的预置颜色设置
     *
     * @param baseConfig
     */
    private void boundPreConfigColor(JSONObject baseConfig) {
        if (null != _bgColorJSON) {
            try {
                baseConfig.put("backgroundColor", _bgColorJSON.opt("backgroundColor"));
                _bgColorJSON = null;
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        if (null != _boardJSON) {
            try {
                baseConfig.put("border", _boardJSON.opt("border"));
                _boardJSON = null;
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 停止旁路直播
     *
     * @param callback
     */
    private void stopBroadcastRoom(ICallback callback) {
        if (null != mVhallRTC) {
            mVhallRTC.stopRoomBroadCast(null, new FinishCallback() {
                @Override
                public void onFinish(int code, String message) {
                    if (code != 200) {
                        if (callback != null)
                            callback.onFailure(code, message);
                    } else {
                        if (callback != null) {
                            callback.onSuccess();
                        }
                    }
                }
            });
        }
    }

    private VHRenderView initRenderView(IVHRTCRenderView renderView) {
        if (null == renderView) return null;
        VHRenderView tempRenderView = new VHRenderView(mContext);
        tempRenderView.init(null, null);
        renderView.bindSurfaceView(tempRenderView);
        return tempRenderView;
    }

    private Stream initLocalStream(VRTCParams.VRTCVideoEncodeParam videoParam, VRTCParams.VRTCStreamParam streamParam) {
        VRTCVideoEncodeCompat.compat(videoParam);
        JSONObject option = new JSONObject();
        try {
            if (videoParam.videoResolutionMode == VRTCCode.RTC_VIDEO_PROFILE_MODE_LANDSCAPE) {
                option.put(Stream.kVideoWidthKey, videoParam.videoWidth);
                option.put(Stream.kVideoHeightKey, videoParam.videoHeight);
            } else {
                option.put(Stream.kVideoWidthKey, videoParam.videoHeight);
                option.put(Stream.kVideoHeightKey, videoParam.videoWidth);
            }

            option.put(Stream.kVideoFpsKey, videoParam.videoFps);

            option.put(Stream.kMinBitrateKbpsKey, videoParam.minVideoBitrate);
            option.put(Stream.kCurrentBitrateKey, videoParam.videoBitrate);
            option.put(Stream.kMaxBitrateKey, videoParam.maxVideoBitrate);

            option.put(Stream.kStreamOptionStreamType, streamParam.streamType);

            if (!TextUtils.isEmpty(streamParam.nickName)) {
                option.put("nickName", streamParam.nickName);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        return mVhallRTC.createLocalStream(option, streamParam.attributes);
    }

    private Stream initScreenShareStream(MediaProjection mediaProjection, VRTCParams.VRTCVideoEncodeParam videoParam, VRTCParams.VRTCStreamParam streamParam) {

        VRTCVideoEncodeCompat.compat(videoParam);
        JSONObject option = new JSONObject();
        try {
            option.put(Stream.kVideoWidthKey, videoParam.videoWidth);
            option.put(Stream.kVideoHeightKey, videoParam.videoHeight);

            option.put(Stream.kVideoFpsKey, 5);

            option.put(Stream.kMinBitrateKbpsKey, videoParam.minVideoBitrate);
            option.put(Stream.kCurrentBitrateKey, videoParam.videoBitrate);
            option.put(Stream.kMaxBitrateKey, videoParam.maxVideoBitrate);

            option.put(Stream.kStreamOptionStreamType, Stream.VhallStreamType.VhallStreamTypeScreen.getValue());

            if (!TextUtils.isEmpty(streamParam.nickName)) {
                option.put("nickName", streamParam.nickName);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return mVhallRTC.createScreenLocalStream(mediaProjection, option, streamParam.attributes);
    }

    private class VRTCRoomDelegate implements Room.RoomDelegate {

        @Override
        public void onDidConnect(Room room, JSONObject jsonObject) {//进入房间
            mRoom = room;
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidConnect");

            if (null != mRoomListener) {
                mRoomListener.onDidConnect(jsonObject);
            }
            if (mAutoPublish && null != mVhallRTC) {
                mVhallRTC.publish();
            }
            if (mAutoSubscribe) {
                subscribeStreams(room, room.getRemoteStreams());
            }
            List<Stream> internalStreams = room.getRemoteInternalStreams();
            if (internalStreams != null && !internalStreams.isEmpty()) {
                for (Stream stream : internalStreams) {
                    mRoomListener.onDidInternalStreamAdded(stream.userId, stream.streamId);
                }
            }
        }

        /**
         * 受设备硬件及网速原因影响，部分机型一次性加载16路流存在大概率崩溃风险
         *
         * @param streams
         */
        private void subscribeStreams(Room room, List<Stream> streams) {
            if (null != streams && !streams.isEmpty()) {
                for (int i = 0; i < streams.size(); i++) {
                    room.subscribe(streams.get(i));
                }
            }
        }

        @Override
        public void onDidError(Room room, Room.VHRoomErrorStatus vhRoomErrorStatus, String errMsg) {
            VLog.e(TAG, "---> VRTCRoomDelegate#onDidError, Room.VHRoomErrorStatus=" + vhRoomErrorStatus.getValue());
            if (null != mRoomListener) {
                mRoomListener.onDidError(0, errMsg, null);
            }
        }

        @Override
        public void onDidPublishStream(Room room, Stream stream) {
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidPublishStream");
            if (null != mRoomListener) {
                mRoomListener.onDidPublishStream(stream.userId, stream.streamId);
            }
        }

        @Override
        public void onDidInternalStreamAdded(Room room, Stream stream) {
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidInternalStreamAdded");
            if (null != mRoomListener) {
                mRoomListener.onDidInternalStreamAdded(stream.userId, stream.streamId);
            }
        }

        @Override
        public void onDidInternalStreamRemoved(Room room, Stream stream) {
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidInternalStreamRemoved");
            if (null != mRoomListener) {
                mRoomListener.onDidInternalStreamRemoved(stream.userId, stream.streamId);
            }
        }

        @Override
        public void onDidInternalStreamFailed(Room room, Stream stream, JSONObject jsonObject) {
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidInternalStreamFailed");
            if (null != mRoomListener) {
                mRoomListener.onDidInternalStreamFailed(stream.userId, stream.streamId, jsonObject);
            }
        }

        @Override
        public void onDidUnPublishStream(Room room, Stream stream) {//下麦
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidUnPublishStream");
            if (null != mRoomListener) {
                mRoomListener.onDidUnPublishStream(stream.userId, stream.streamId);
            }
        }

        @Override
        public void onDidSubscribeStream(Room room, Stream stream) {//订阅其他流
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidSubscribeStream");
            if (null != mRoomListener) {
                mRoomListener.onDidSubscribeStream(stream.userId, stream.streamId);
            }
        }

        @Override
        public void onDidUnSubscribeStream(Room room, Stream stream) {//取消订阅
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidUnSubscribeStream");
            if (null != mRoomListener) {
                mRoomListener.onDidUnSubscribeStream(stream.userId, stream.streamId);
            }
        }

        @Override
        public void onDidChangeStatus(Room room, Room.VHRoomStatus vhRoomStatus) {
            /**已废弃，请使用{@link VHRoomListener#onDidConnect(JSONObject)} {@link VHRoomListener#onDidExitRoom(int)}**/
            if (vhRoomStatus == Room.VHRoomStatus.VHRoomStatusDisconnected) {
                if (null != mRoomListener) {
                    mRoomListener.onDidExitRoom();
                }
            }
        }

        @Override
        public void onDidAddStream(Room room, Stream stream) {
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidAddStream");
            if (mAutoSubscribe) {
                room.subscribe(stream);
            }
            if (null != mRoomListener) {
                mRoomListener.onDidAddStream(stream.userId, stream.streamId);
            }
        }

        @Override
        public void onDidRemoveStream(Room room, Stream stream) {//有流退出
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidRemoveStream");
            if (null != mRoomListener) {
                mRoomListener.onDidRemoveStream(stream.userId, stream.streamId);
            }
        }

        @Override
        public void onDidUpdateOfStream(Stream stream, JSONObject jsonObject) {//流状态更新
            VLog.d(TAG, "---> VRTCRoomDelegate#onDidUpdateOfStream");
            //订阅端如需更新标识可自行处理业务逻辑
            if (null != mRoomListener) {
                mRoomListener.onDidUpdateOfStream(stream.streamId, jsonObject);

                if (null != jsonObject && jsonObject.has("muteStream")) {
                    JSONObject obj = jsonObject.optJSONObject("muteStream");
                    JSONObject msgV2 = new JSONObject();
                    try {
                        if (obj.has("video")) {
                            msgV2.put("video", obj.optBoolean("video", true));
                        }
                        if (obj.has("audio")) {
                            msgV2.put("audio", obj.optBoolean("audio", true));
                        }
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                    mRoomListener.onDidUpdateOfStreamV2(stream.streamId, msgV2);
                }
            }
        }

        /**
         * 互动房间重连
         *
         * @param total   总重连次数
         * @param current 当前重连次数
         */
        @Override
        public void onReconnect(int total, int current) {
            VLog.d(TAG, "---> VRTCRoomDelegate#onReconnect: total = " + total + " current " + current);
            if (null != mRoomListener) {
                mRoomListener.onReconnect(total, current);
            }
        }

        @Override
        public void onStreamMixed(JSONObject jsonObject) {
            VLog.d(TAG, "---> VRTCRoomDelegate#onStreamMixed");
            if (null != mRoomListener) {
                mRoomListener.onStreamMixed(jsonObject);
            }
        }
    }
}