package com.vhall.rtc.trtc;

import static com.tencent.liteav.TXLiteAVCode.ERR_CAMERA_NOT_AUTHORIZED;
import static com.tencent.liteav.TXLiteAVCode.ERR_CAMERA_OCCUPY;
import static com.tencent.liteav.TXLiteAVCode.ERR_CAMERA_SET_PARAM_FAIL;
import static com.tencent.liteav.TXLiteAVCode.ERR_CAMERA_START_FAIL;
import static com.tencent.trtc.TRTCCloudDef.TRTC_APP_SCENE_LIVE;
import static com.vhall.rtc.VRTCCode.ERR_BROADCAST_NONSUPPORT_DOCCLOUD;
import static com.vhall.rtc.VRTCCode.ERR_BROADCAST_DOCCLOUD_REQUEST_FAIL;
import static com.vhall.rtc.VRTCCode.ERR_BROADCAST_NONSUPPORT_GRIDIMAGE;
import static com.vhall.rtc.VRTCCode.ERR_BROADCAST_RESPONSE_UNKNOW;

import android.content.Context;
import android.media.projection.MediaProjection;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.MotionEvent;
import android.view.View;

import com.tencent.liteav.device.TXDeviceManager;
import com.tencent.rtmp.ui.TXCloudVideoView;
import com.tencent.trtc.TRTCCloud;
import com.tencent.trtc.TRTCCloudDef;
import com.tencent.trtc.TRTCCloudListener;
import com.tencent.trtc.TRTCStatistics;
import com.vhall.framework.common.ICallback;
import com.vhall.ilss.VHInteractiveApi;
import com.vhall.ilss.VHRoomListener;
import com.vhall.logmanager.VLog;
import com.vhall.rtc.Utils;
import com.vhall.rtc.VRTCCode;
import com.vhall.rtc.VRTCParams;
import com.vhall.rtc.absrtc.ISwitchFocusModeListener;
import com.vhall.rtc.absrtc.IVHRTC;
import com.vhall.rtc.absrtc.IVHRTCRenderView;
import com.vhall.rtc.trtc.log.LogReport;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.HashMap;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;

/**
 * @author：vhall Email：jooperge@163.com
 * 描述：
 * 修改历史:
 * <p>
 * 创建于： 2022/10/18
 */
public class VhallTRTCImpl implements IVHRTC {

    private final String TAG = "VhallTRTC";
    private TRTCCloud mTRTCCloud;
    private TXDeviceManager mTXDeviceManager;
    private VRTCParams.VRTCRoomParamsInner mRoomParams;
    /**
     * 进房成功后是否自动上麦/订阅
     */
    private boolean mAutoPublish, mAutoSubscribe;
    private VHRoomListener mRoomListener = null;
    /**
     * vhall业务流类型
     */
    private int mLocalStreamType;

    /**
     * 主路流
     */
    private final String TRTC_STREAM_TYPE_MAIN = "main";
    /**
     * 辅路流
     */
    private final String TRTC_STREAM_TYPE_AUXILIARY = "auxiliary";

    /**
     * TRTC主/辅流标识
     */
    private String mLocalTRTCStreamType = TRTC_STREAM_TYPE_MAIN;

    /**
     * 房间内流缓存, streamId: Stream
     */
    private HashMap<String, Stream> mStreamList;
    private Stream mLocalStream;
    private VRTCParams.VRTCVideoEncodeParam mInitVideoEncoderParam;

    private String mStreamAttributes;

    // streamId -> userId
    private HashMap<String, String> userIdList;
    /**
     * TRTC角色
     */
    private int mCurTRTCRole = TRTCCloudDef.TRTCRoleAudience;
    /**
     * 本地流创建时，若早于enterRoom，则暂时没有streamId，需要用该值作为占位key，待enterRoom后替换该占位key为真实streamId(userId)
     */
    private final String PLACEHOLDER_LOCAL_STREAMID = "placeholder_local_streamid";

    /**
     * 业务层定义
     */
    private int mDevicePlatform = 5;

    public VhallTRTCImpl(Context context) {
        if (null != context) {
            mTRTCCloud = TRTCCloud.sharedInstance(context.getApplicationContext());
            mTXDeviceManager = mTRTCCloud.getDeviceManager();
            mStreamList = new HashMap<>();
            userIdList = new HashMap<>();
        } else {
            throw new IllegalArgumentException("context should not be null");
        }
    }

    /**
     * 仅用作临时缓存
     */
    private String mTempUserId = null;

    @Override
    public void enterRoom(VRTCParams.VRTCRoomParamsInner roomParams, boolean autoPublish, boolean autoSubscribe, String attribute, VHRoomListener roomListener) {
        mRoomParams = roomParams;
        if (null != mLocalStream) {
            mLocalStream.streamId = mRoomParams.userId;
            mLocalStream.userId = mRoomParams.userId;

            mStreamList.remove(PLACEHOLDER_LOCAL_STREAMID);
            addStream2Cache(mLocalStream);
        } else {
            mTempUserId = mRoomParams.userId;
        }
        if (null != mTRTCCloud) {
            mAutoPublish = autoPublish;
            mAutoSubscribe = autoSubscribe;
            if (!mAutoSubscribe) {
                mTRTCCloud.setDefaultStreamRecvMode(false, false);
            }
            mRoomListener = roomListener;
            mTRTCCloud.setListener(new TRTCCloudImplListener());
            TRTCCloudDef.TRTCParams trtcParams = makeTRTCParam(roomParams);
            mTRTCCloud.enterRoom(trtcParams, TRTC_APP_SCENE_LIVE);
        } else {
            VLog.e(TAG, "please init first");
        }
    }

    @Override
    public void startLocalPreview(IVHRTCRenderView renderView, VRTCParams.VRTCVideoEncodeParam videoParam, VRTCParams.VRTCStreamParam streamParam, boolean isFrontCamera) {
        mLocalStreamType = streamParam.streamType;
        mLocalTRTCStreamType = TRTC_STREAM_TYPE_MAIN;
        mInitVideoEncoderParam = videoParam;
        mStreamAttributes = streamParam.attributes;

        mLocalStream = new Stream();
        mLocalStream.isLocal = true;
        if (null != mRoomParams) {
            mLocalStream.streamId = mRoomParams.userId;
            mLocalStream.userId = mRoomParams.userId;
        } else {
            //占位key，待enterRoom时替换为真实值
            mLocalStream.streamId = PLACEHOLDER_LOCAL_STREAMID;
        }
        mLocalStream.streamTypeVrtc = mLocalStreamType;
        mLocalStream.streamTypeTrtc = TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG;
        addStream2Cache(mLocalStream);

        if (null != mTRTCCloud) {
            //开启本地预览前，默认设置填充模式为'裁剪填充'；镜像模式为自动模式(前置开启镜像，后置不开启)
            TRTCCloudDef.TRTCRenderParams localRenderParam = new TRTCCloudDef.TRTCRenderParams();
            localRenderParam.fillMode = TRTCCloudDef.TRTC_VIDEO_RENDER_MODE_FILL;
            localRenderParam.mirrorType = TRTCCloudDef.TRTC_VIDEO_MIRROR_TYPE_AUTO;
            mTRTCCloud.setLocalRenderParams(localRenderParam);

            if (hasAudio(mLocalStreamType)) {
                mTRTCCloud.startLocalAudio(TRTCCloudDef.TRTC_AUDIO_QUALITY_SPEECH);
            }

            updateVideoResolution(videoParam, false);

            if (null != renderView) {
                mTRTCCloud.startLocalPreview(isFrontCamera, generateRenderView(renderView));
            }
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }

        if (null != renderView) {
            renderView.setOnTouchListener(new View.OnTouchListener() {
                @Override
                public boolean onTouch(View v, MotionEvent event) {
                    //TODO mTXDeviceManager.setCameraFocusPosition(int, int)
                    //TODO 验证是否会影响 IVHRTCRenderView UI层的点击事件响应
                    return v.onTouchEvent(event);
                }
            });
        }
    }

    @Override
    public void createScreenShareStream(MediaProjection mediaProjection, VRTCParams.VRTCVideoEncodeParam videoParam, VRTCParams.VRTCStreamParam streamParam) {
        //TRTC不处理mediaProjection，传Null即可
        mStreamAttributes = streamParam.attributes;

        mLocalStreamType = VRTCParams.VRTCStreamType.VhallStreamTypeScreen;
        mLocalTRTCStreamType = TRTC_STREAM_TYPE_AUXILIARY;

        VRTCParams.VRTCVideoEncodeParam encodeParam;
        if (null == videoParam) {//使用TRTC推荐参数配置, https://cloud.tencent.com/document/product/647/79628#37f453c413986eb62c0ffacb15c392fe
            encodeParam = new VRTCParams.VRTCVideoEncodeParam();
            encodeParam.videoResolution = VRTCCode.RTCVideoProfile.PROFILE_1280_720_16x9;
            encodeParam.videoFps = 10;
            encodeParam.videoBitrate = 1200;
            encodeParam.resolutionStrategy = VRTCCode.RTCDegradationMode.MAINTAIN_RESOLUTION;
        } else {
            encodeParam = videoParam;
        }

        mInitVideoEncoderParam = encodeParam;

        mLocalStream = new Stream();
        mLocalStream.isLocal = true;
        if (null != mRoomParams) {
            mLocalStream.streamId = mRoomParams.userId;
            mLocalStream.userId = mRoomParams.userId;
        }
        mLocalStream.streamTypeVrtc = VRTCParams.VRTCStreamType.VhallStreamTypeScreen;
        mLocalStream.streamTypeTrtc = TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_SUB;
        addStream2Cache(mLocalStream);

        TRTCCloudDef.TRTCVideoEncParam trtcVideoEncParam = updateVideoResolution(encodeParam, true);

        mTRTCCloud.startScreenCapture(mLocalStream.streamTypeTrtc, trtcVideoEncParam, new TRTCCloudDef.TRTCScreenShareParams());
    }

    @Override
    public void stopScreenShare() {
        mTRTCCloud.stopScreenCapture();
    }

    @Override
    public String getUserIdByStreamId(String streamId) {
        Stream stream = getStreamById(streamId);
        return null != stream ? stream.userId : null;
    }

    private void addStream2Cache(Stream stream) {
        if (null != mStreamList) {
            mStreamList.put(stream.streamId, stream);
        }
    }

    private void removeStreamFromCache(String streamId) {
        if (TextUtils.isEmpty(streamId)) {
            return;
        }
        if (null != mStreamList) {
            mStreamList.remove(streamId);
        }
        if (null != userIdList) {
            userIdList.remove(streamId);
        }
    }

    private Stream getStreamById(String streamId) {
        if (TextUtils.isEmpty(streamId)) {
            return null;
        }
        if (null != mStreamList) {
            if (mStreamList.containsKey(streamId)) {
                return mStreamList.get(streamId);
            }
        }
        return null;
    }

    private boolean containsStreamId(String streamId) {
        return null != getStreamById(streamId);
    }

    private TXCloudVideoView generateRenderView(IVHRTCRenderView renderView) {
        if (null == renderView) return null;
        TXCloudVideoView tempRenderView = new TXCloudVideoView(renderView.getContext());
        renderView.bindFramelayout(tempRenderView);
        return tempRenderView;
    }

    @Override
    public void startRemoteView(IVHRTCRenderView renderView, String streamId) {
        if (null != mTRTCCloud) {
            mTRTCCloud.muteRemoteAudio(userIdList.get(streamId), false);
            Stream stream = getStreamById(streamId);
            if (null != stream) {

                mTRTCCloud.startRemoteView(userIdList.get(streamId), stream.streamTypeTrtc, generateRenderView(renderView));

                TRTCCloudDef.TRTCRenderParams param = new TRTCCloudDef.TRTCRenderParams();
                param.fillMode = TRTCCloudDef.TRTC_VIDEO_RENDER_MODE_FIT;
                param.mirrorType = TRTCCloudDef.TRTC_VIDEO_MIRROR_TYPE_DISABLE;
                mTRTCCloud.setRemoteRenderParams(userIdList.get(streamId), stream.streamTypeTrtc, param);
            } else {
                VLog.e(TAG, "streamId do not exists");
            }
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    @Override
    public void publish() {
        //调用后房间内其他用户会收到 onRemoteUserEnterRoom
        if (null != mTRTCCloud) {
            if (!mStreamAttributes.isEmpty()) {
                VHInteractiveApi.stream_update_attributes(mRoomParams.roomParams.inavRoomId, mRoomParams.roomParams.accessToken, getCombinedStreamId(mRoomParams.userId), mStreamAttributes, new Callback() {
                    @Override
                    public void onFailure(Call call, IOException e) {
                        VLog.e(TAG, "stream_update_attributes failed");
                    }

                    @Override
                    public void onResponse(Call call, Response response) throws IOException {
                        VLog.i(TAG, "stream_update_attributes succeeded");
                    }
                });

                VHInteractiveApi.stream_get_attributes(mRoomParams.roomParams.inavRoomId, mRoomParams.roomParams.accessToken, getCombinedStreamId(mRoomParams.userId), new Callback() {
                    @Override
                    public void onFailure(Call call, IOException e) {
                        VLog.e(TAG, "stream_update_attributes failed");
                    }

                    @Override
                    public void onResponse(Call call, Response response) throws IOException {
                        String result = response.body().string();
                        VLog.i(TAG, "stream_update_attributes succeeded");
                    }
                });
            }
            mCurTRTCRole = TRTCCloudDef.TRTCRoleAnchor;
            mTRTCCloud.switchRole(TRTCCloudDef.TRTCRoleAnchor);
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    @Override
    public void unPublish() {
        //调用后房间内其他用户会收到 onRemoteUserLeaveRoom
        if (null != mTRTCCloud) {
            mCurTRTCRole = TRTCCloudDef.TRTCRoleAudience;
            mTRTCCloud.switchRole(TRTCCloudDef.TRTCRoleAudience);
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    @Override
    public void subscribe(String streamId) {
        if (null != mRoomListener) {
            Stream stream = getStreamById(streamId);
            if (null == stream) {
                return;
            }
            String bzUserId = stream.userId;
            mRoomListener.onDidSubscribeStream(bzUserId, streamId);
            LogReport.instance().streamStart(stream);
        }
    }

    @Override
    public void unSubscribe(String streamId) {
        if (null != mTRTCCloud) {
            mTRTCCloud.muteRemoteAudio(userIdList.get(streamId), true);
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }

        muteVideo(streamId, true);

        if (null != mRoomListener) {
            HashMap<String, String> user = tryParseWebStyleUserId(streamId);
            mRoomListener.onDidUnSubscribeStream(user.get("userId"), streamId);
        }
    }

    @Override
    public void leaveRoom() {
        //回调成功后会触发 onExitRoom
        if (null != mTRTCCloud) {
            mTRTCCloud.exitRoom();
            LogReport.instance().streamAllStop();
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    @Override
    public void muteVideo(String streamId, boolean mute) {
        if (null != mTRTCCloud) {
            if (TextUtils.isEmpty(streamId)) {
                // 本地流不需要提取userId
                mTRTCCloud.muteLocalVideo(isMainTRTCStream() ? TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG : TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_SUB, mute);
                notifyEvent(getUserId(), mute ? VRTCCode.TRTCStreamEvent.MUTE_VIDEO : VRTCCode.TRTCStreamEvent.UN_MUTE_VIDEO);
            } else {
                HashMap<String, String> user = tryParseWebStyleUserId(streamId);
                int type = (user.get("vrtcStreamType").equals(String.valueOf(VRTCParams.VRTCStreamType.VhallStreamTypeAudioAndVideo))) ? TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG : TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_SUB;
                mTRTCCloud.muteRemoteVideoStream(userIdList.get(streamId), type, mute);
            }
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    @Override
    public void muteAudio(String streamId, boolean mute) {
        if (null != mTRTCCloud) {
            if (TextUtils.isEmpty(streamId)) {
                mTRTCCloud.muteLocalAudio(mute);
                notifyEvent(getUserId(), mute ? VRTCCode.TRTCStreamEvent.MUTE_AUDIO : VRTCCode.TRTCStreamEvent.UN_MUTE_AUDIO);
            } else {
                mTRTCCloud.muteRemoteAudio(userIdList.get(streamId), mute);
            }
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    @Override
    public void switchCamera() {
        if (null != mTXDeviceManager) {
            mTXDeviceManager.switchCamera(!mTXDeviceManager.isFrontCamera());
        }
    }

    @Override
    public boolean isFrontCamera() {
        return null == mTXDeviceManager || mTXDeviceManager.isFrontCamera();
    }

    /**
     * 设置镜像
     * **启动本地预览时，默认镜像模式为自动{@link TRTCCloudDef#TRTC_VIDEO_MIRROR_TYPE_AUTO}，自动模式：如果正使用前置摄像头则开启镜像，如果是后置摄像头则不开启镜像（仅适用于移动设备）**
     *
     * @param isMirror
     */
    @Override
    public void setMirror(boolean isMirror) {
        if (null != mTRTCCloud) {
            TRTCCloudDef.TRTCRenderParams localRenderParam = new TRTCCloudDef.TRTCRenderParams();
            localRenderParam.mirrorType = isMirror ? TRTCCloudDef.TRTC_VIDEO_MIRROR_TYPE_ENABLE : TRTCCloudDef.TRTC_VIDEO_MIRROR_TYPE_DISABLE;
            mTRTCCloud.setLocalRenderParams(localRenderParam);
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    @Override
    public void switchAutoFocusMode(boolean enableAuto, ISwitchFocusModeListener switchFocusModeListener) {
        mTXDeviceManager.enableCameraAutoFocus(enableAuto);
        if (null != switchFocusModeListener) {
            // TRTC模式暂不支持对焦模式切换状态的回调，故暂定立即返回true
            switchFocusModeListener.onComplete(true, null);
        }
    }

    @Override
    public void release() {
        if (null != mStreamList) {
            mStreamList.clear();
            mStreamList = null;
        }
        mLocalStream = null;
        mInitVideoEncoderParam = null;
        if (mTRTCCloud != null) {
            mTRTCCloud.stopAllRemoteView();
            mTRTCCloud.stopLocalAudio();
            mTRTCCloud.stopLocalPreview();
            leaveRoom();
            mTRTCCloud.setListener(null);
        }
        mTRTCCloud = null;
        TRTCCloud.destroySharedInstance();
    }

    @Override
    public void setScaleType(int scaleType) {
        if (null != mTRTCCloud) {
            TRTCCloudDef.TRTCRenderParams localRenderParam = new TRTCCloudDef.TRTCRenderParams();
            localRenderParam.fillMode = convertScaleType2TRTCMode(scaleType);
            mTRTCCloud.setLocalRenderParams(localRenderParam);
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    @Override
    public void setVideoResolution(int videoProfile) {
        if (null != mTRTCCloud) {
            mInitVideoEncoderParam.videoResolution = videoProfile;
            updateVideoResolution(mInitVideoEncoderParam, false);
        } else {
            VLog.e(TAG, "mTRTCCloud should not be null");
        }
    }

    /**
     * 根据推荐配置重新处理业务层传入的视频编码参数
     *
     * @param videoParam
     */
    private TRTCCloudDef.TRTCVideoEncParam updateVideoResolution(VRTCParams.VRTCVideoEncodeParam videoParam, boolean isScreenShare) {
        TRTCVideoEncodeCompat.compat(videoParam);
        TRTCCloudDef.TRTCVideoEncParam encParam = new TRTCCloudDef.TRTCVideoEncParam();
        encParam.videoResolution = videoParam.videoResolution2TRTC;
        encParam.videoFps = videoParam.videoFps;
        encParam.videoResolutionMode = TRTCCloudDef.TRTC_VIDEO_RESOLUTION_MODE_PORTRAIT;
        encParam.videoBitrate = videoParam.videoBitrate;
        //取值参考官方文档：https://cloud.tencent.com/document/product/647/79634 ，小标题"TRTCVideoEncParam"
        if (videoParam.resolutionStrategy == VRTCCode.RTCDegradationMode.BALANCED) {
            //平衡模式
            encParam.minVideoBitrate = 0;
            encParam.enableAdjustRes = true;
        } else if (videoParam.resolutionStrategy == VRTCCode.RTCDegradationMode.MAINTAIN_NONE) {
            //不允许降分辨率和码率
            encParam.minVideoBitrate = videoParam.videoBitrate;
            encParam.enableAdjustRes = false;
        } else if (videoParam.resolutionStrategy == VRTCCode.RTCDegradationMode.MAINTAIN_RESOLUTION) {
            //不允许降分辨率，允许降码率
            encParam.minVideoBitrate = (int) (videoParam.videoBitrate * 0.6);
            encParam.enableAdjustRes = false;
        } else if (videoParam.resolutionStrategy == VRTCCode.RTCDegradationMode.MAINTAIN_FLUENCY) {
            //保持流畅度，允许降分辨率
            encParam.minVideoBitrate = 100;
            encParam.enableAdjustRes = true;
        }

        if (!isScreenShare) {
            mTRTCCloud.setVideoEncoderParam(encParam);
        }
        return encParam;
    }

    @Override
    public void setDefaultAudioDevice(Object object) {
        //not yet
    }

    @Override
    public void broadcastRoom(VRTCParams.VRTCBroadcastParamInner broadcastParam, String accessToken, ICallback callback) {
        if (null == broadcastParam || null == broadcastParam.broadcastParam) {
            VHInteractiveApi.stopBroadcastTRTC(mRoomParams.roomParams.inavRoomId, accessToken, new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    if (null != callback) {
                        callback.onFailure(ERR_BROADCAST_RESPONSE_UNKNOW, null != e ? e.getMessage() : null);
                    }
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    if (null != callback) {
                        Utils.parseOkHttpCallback(callback, response);
                    }
                    if (null != mRoomListener) {
                        mRoomListener.onStreamMixed(new JSONObject());
                    }
                }
            });
        } else {
            VHInteractiveApi.startBroadcastTRTC(broadcastParam, accessToken, new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {
                    if (null != callback) {
                        callback.onFailure(ERR_BROADCAST_RESPONSE_UNKNOW, null != e ? e.getMessage() : null);
                    }
                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    if (null != callback) {
                        Utils.parseOkHttpCallback(callback, response);
                    }
                    if (null != mRoomListener) {
                        mRoomListener.onStreamMixed(new JSONObject());
                    }
                }
            });
        }
    }

    @Override
    public void setMixLayoutMainScreen(String streamId, ICallback callback) {
        if (TextUtils.isEmpty(streamId)) {
            // streamId 为空，设置本地流为主屏
            streamId = mLocalStream.streamId;
        }
        VHInteractiveApi.setMixMainScreenTRTC(
                mRoomParams.roomParams.inavRoomId,
                (userIdList.get(streamId) == null || userIdList.get(streamId).isEmpty()) ? streamId : userIdList.get(streamId),
                mRoomParams.roomParams.accessToken,
                new Callback() {
                    @Override
                    public void onFailure(Call call, IOException e) {
                        callback.onFailure(ERR_BROADCAST_RESPONSE_UNKNOW, null != e ? e.getMessage() : null);
                    }

                    @Override
                    public void onResponse(Call call, Response response) throws IOException {
                        if (null != callback) {
                            Utils.parseOkHttpCallback(callback, response);
                        }
                    }
                });
    }

    @Override
    public void setMixLayoutMode(VRTCParams.VRTCBroadcastParamInner broadcastParam, int layoutMode, ICallback callback) {
        broadcastParam.isAdaptiveLayoutMode = layoutMode >= 1000;
        broadcastParam.broadcastParam.layoutMode = layoutMode;
        broadcastParam.compat_layout_mode = Utils.compatAdaptiveLayoutMode(layoutMode);
        broadcastRoom(broadcastParam, mRoomParams.roomParams.accessToken, callback);
    }

    @Override
    public void setMixBackgroundColor(VRTCParams.VRTCBroadcastParamInner broadcastParam, String hexColor, ICallback callback) {
        broadcastParam.broadcastParam.backgroundColor = Utils.compatColor2HexStyle(hexColor);
        broadcastRoom(broadcastParam, mRoomParams.roomParams.accessToken, callback);
    }

    @Override
    public void setMixBackgroundImage(VRTCParams.VRTCBroadcastParamInner broadcastParam, String url, int cropType, ICallback callback) {
        broadcastParam.broadcastParam.backgroundImageUrl = url;
        broadcastRoom(broadcastParam, mRoomParams.roomParams.accessToken, callback);
    }

    @Override
    public void setMixBorderColor(VRTCParams.VRTCBroadcastParam.Border border, ICallback callback) {
        VLog.e(TAG, "TRTC暂不支持设置旁路边框");
    }

    @Override
    public void setDataReport(JSONObject dataReport) {
        if (null != dataReport) {
            LogReport logReport = LogReport.instance();
            logReport.setLogDataInfo(dataReport);
            if (dataReport.has("pf")) {
                this.mDevicePlatform = dataReport.optInt("pf");
            }
        }
    }

    @Override
    public void setMixPlaceholderImage(String url, ICallback callback) {
        VLog.e(TAG, "TRTC暂不支持设置旁路窗格占位图");
        if (null != callback) {
            callback.onFailure(ERR_BROADCAST_NONSUPPORT_GRIDIMAGE, "TRTC暂不支持设置旁路窗格占位图");
        }
    }

    @Override
    public void docCloudRender(String appId, String channelId, boolean start, ICallback callback) {
        VHInteractiveApi.docCloudRender(mRoomParams.roomParams.inavRoomId, mRoomParams.roomParams.accessToken, channelId, start, new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                callback.onFailure(ERR_BROADCAST_DOCCLOUD_REQUEST_FAIL, null != e ? e.getMessage() : null);
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                if (null != callback) {
                    Utils.parseOkHttpCallback(callback, response);
                }
            }
        });
        VLog.e(TAG, "docCloudRender " + appId + " " + channelId + " " + start);
    }

    private TRTCCloudDef.TRTCParams makeTRTCParam(VRTCParams.VRTCRoomParamsInner roomParams) {
        TRTCCloudDef.TRTCParams trtcParams = new TRTCCloudDef.TRTCParams();
        trtcParams.sdkAppId = roomParams.trtcSdkAppId;
        trtcParams.userId = roomParams.userId;
        trtcParams.strRoomId = roomParams.roomParams.inavRoomId;
        trtcParams.userSig = roomParams.inavToken;
//        trtcParams.role = isHostAndAutoPublish(mAutoPublish, roomParams) ? TRTCCloudDef.TRTCRoleAnchor : TRTCCloudDef.TRTCRoleAudience;
        trtcParams.role = TRTCCloudDef.TRTCRoleAudience;
        mCurTRTCRole = TRTCCloudDef.TRTCRoleAudience;
        return trtcParams;
    }

    /**
     * 官方释义见：https://cloud.tencent.com/document/product/647/79629#d6e99ea808109fd61ea58de9e493f42a
     */
    private class TRTCCloudImplListener extends TRTCCloudListener {
        @Override
        public void onEnterRoom(long result) {
            VLog.d(TAG, "onEnterRoom, result=" + result);
            if (null != mRoomListener) {
                try {
                    mRoomListener.onDidConnect(new JSONObject().put("result", result));
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            if (mAutoPublish) {
                publish();
            }

            String clientId = String.valueOf(System.currentTimeMillis()); //对应VRTC中的clientId含义：信令服务器为每个连接生成的uniqueId
            LogReport.instance().setLogRoominfo(mRoomParams.userId, mRoomParams.roomParams.inavRoomId, clientId, mDevicePlatform);
        }

        @Override
        public void onExitRoom(int reason) {
            VLog.d(TAG, "onExitRoom, reason=" + reason);
            if (null != mRoomListener) {
                mRoomListener.onDidExitRoom();
            }
        }

        @Override
        public void onError(int errCode, String errMsg, Bundle extraInfo) {
            VLog.e(TAG, "onError, errCode=" + errMsg + ", errMsg=" + errMsg + ", extraInfo=" + (null != extraInfo ? extraInfo.toString() : null));
            if (null != mRoomListener) {
                //相机异常单独处理
                if (errCode == ERR_CAMERA_START_FAIL || errCode == ERR_CAMERA_NOT_AUTHORIZED ||
                        errCode == ERR_CAMERA_SET_PARAM_FAIL || errCode == ERR_CAMERA_OCCUPY) {
                    mRoomListener.onCameraError(errCode, errMsg);
                } else {
                    mRoomListener.onDidError(errCode, errMsg, null != extraInfo ? extraInfo.toString() : null);
                }
            }
        }

        @Override
        public void onWarning(int warningCode, String warningMsg, Bundle extraInfo) {
            VLog.d(TAG, "onWarning, warningCode=" + warningCode + ", warningMsg=" + warningMsg + ", extraInfo=" + (null != extraInfo ? extraInfo.toString() : null));
            if (null != mRoomListener) {
                mRoomListener.onWarning(warningCode, warningMsg, null != extraInfo ? extraInfo.toString() : null);
            }
        }

//        /**
//         * @param streamType TRTC视频流类型：主路（Main）一般用于承载摄像头画面，辅路（Sub）一般用于承载屏幕分享画面。
//         */
//        @Override
//        public void onSendFirstLocalVideoFrame(int streamType) {
//            if (!isOnlyAudio() && null != mRoomListener) {
//                mRoomListener.onDidPublishStream(getUserId(), getUserId());
//            }
//        }
//
//        @Override
//        public void onSendFirstLocalAudioFrame() {
//            //只有音频的情况下也需要触发vrtc通用接口
//            if (isOnlyAudio() && null != mRoomListener) {
//                mRoomListener.onDidPublishStream(getUserId(), getUserId());
//            }
//        }

        @Override
        public void onSwitchRole(int errCode, String errMsg) {
            VLog.e(TAG, "onSwitchRole, errCode=" + errCode + ", errMsg=" + errMsg);
            if (0 == errCode) {
                if (null != mRoomListener) {
                    if (TRTCCloudDef.TRTCRoleAnchor == mCurTRTCRole) {
                        mRoomListener.onDidPublishStream(getUserId(), getUserId());
                        notifyEvent(getUserId(), VRTCCode.TRTCStreamEvent.PUBLISH);

                        LogReport.instance().streamStart(mLocalStream);
                    } else if (TRTCCloudDef.TRTCRoleAudience == mCurTRTCRole) {
                        mRoomListener.onDidUnPublishStream(getUserId(), getUserId());
                        notifyEvent(getUserId(), VRTCCode.TRTCStreamEvent.UN_PUBLISH);

                        LogReport.instance().streamStop(mLocalStream.streamId);
                    }
                }
            } else {
                //角色切换失败，本地记录还原
                if (TRTCCloudDef.TRTCRoleAnchor == mCurTRTCRole) {
                    mCurTRTCRole = TRTCCloudDef.TRTCRoleAudience;
                } else if (TRTCCloudDef.TRTCRoleAudience == mCurTRTCRole) {
                    mCurTRTCRole = TRTCCloudDef.TRTCRoleAnchor;
                }
                if (null != mRoomListener) {
                    JSONObject extraInfo = new JSONObject();
                    try {
                        extraInfo.put("errcode_trtc", errCode);
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                    mRoomListener.onDidError(VRTCCode.ERR_ROOM_PUBLISH, errMsg, extraInfo.toString());
                }
            }
        }

        @Override
        public void onFirstVideoFrame(String userId, int streamType, int width, int height) {
            VLog.d(TAG, "onFirstVideoFrame, userId=" + userId + ", streamType=" + streamType + ", width=" + width + ", height=" + height);
            if (null != mRoomListener) {
                mRoomListener.onFirstFrameAvailable(TextUtils.isEmpty(userId), userId, width, height);
            }
        }

        @Override
        public void onRemoteUserEnterRoom(String userId) {
            VLog.d(TAG, "onRemoteUserEnterRoom, userId=" + userId);

            //仅用来处理web端事件，因为web端的enterRoom事件和推流事件是绑定的，即每次推流会同时触发(video or audio or screen)和enterRoom事件
            //且web端流id会携带vrtc业务中的流类型
            if (isStreamFromWeb(userId)) {
                HashMap<String, String> user = tryParseWebStyleUserId(userId);
                // 过滤文档云渲染流
                if (100 == Integer.parseInt(user.get("vrtcStreamType"))) {
                    return;
                }
                doWhenRemoteStreamAvailable(userId, -1, Integer.parseInt(user.get("vrtcStreamType")));
            }
        }

        private void removeRemoteStream(String streamId) {
            HashMap<String, String> user = tryParseWebStyleUserId(streamId);
            if (null == getStreamById(streamId)) {
                return;
            }
            removeStreamFromCache(streamId);
            if (null != mRoomListener) {
                mRoomListener.onDidRemoveStream(user.get("userId"), streamId);
            }
            LogReport.instance().streamStop(streamId);
        }

        @Override
        public void onRemoteUserLeaveRoom(String userId, int reason) {
            VLog.d(TAG, "onRemoteUserLeaveRoom, userId=" + userId + ", reason=" + reason);
            HashMap<String, String> user = tryParseWebStyleUserId(userId);
            // 过滤文档云渲染流
            if (100 == Integer.parseInt(user.get("vrtcStreamType"))) {
                return;
            }
            if (!isStreamFromWeb(userId)) {
                String streamId = userIdToStreamId(userId, TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_SUB);
                removeRemoteStream(streamId); // 移除辅流
                streamId = userIdToStreamId(userId, TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG);
                removeRemoteStream(streamId); // 移除主流
            } else {
                // web端只移除流，所有流退出再上报用户用户离开, 用户是否退出信令不重要，不上报
                removeRemoteStream(userId); // 移除web端推流
            }
        }

        @Override
        public void onTryToReconnect() {
            VLog.e(TAG, "onTryToReconnect");
            if (null != mRoomListener) {
                mRoomListener.onReconnect(0, 0);
            }
        }

        /**
         * TRTC主路(camera)画面可用[远端用户]
         *
         * @param userId
         * @param available
         */
        @Override
        public void onUserVideoAvailable(String userId, boolean available) {
            VLog.d(TAG, "onUserVideoAvailable, userId=" + userId + ", available" + available);
            HashMap<String, String> user = tryParseWebStyleUserId(userId);
            if (100 == Integer.parseInt(user.get("vrtcStreamType"))) {
                return;
            }
            if (available && !isStreamFromWeb(userId)) {
                //只处理native端，[web端由onRemoteUserEnter处理]
                doWhenRemoteStreamAvailable(userId, TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG, -1);
            }

            // 视频不可用是不能判断mute或停止，当做mute处理
            if (null != mRoomListener) {
                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject.put("video", available);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
                mRoomListener.onDidUpdateOfStreamV2(userIdToStreamId(userId, TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG), jsonObject);
            }
        }

        /**
         * TRTC辅路(桌面共享)画面可用
         *
         * @param userId
         * @param available
         */
        @Override
        public void onUserSubStreamAvailable(String userId, boolean available) {
            VLog.d(TAG, "onUserSubStreamAvailable, userId=" + userId + ", available" + available);
            HashMap<String, String> user = tryParseWebStyleUserId(userId);
            if (100 == Integer.parseInt(user.get("vrtcStreamType"))) {
                return;
            }
            if (available && !isStreamFromWeb(userId)) {
                //只处理native端，[web端由onRemoteUserEnter处理]
                doWhenRemoteStreamAvailable(userId, TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_SUB, -1);
            }
            if (!available) {
                // 辅流（桌面共享）mute或停止推流，直接移除
                String streamId = userId;
                if (!isStreamFromWeb(userId)) {
                    streamId = userIdToStreamId(userId, TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_SUB);
                }
                removeStreamFromCache(streamId);
                if (null != mRoomListener) {
                    mRoomListener.onDidRemoveStream(user.get("userId"), streamId);
                }
            }
        }

        @Override
        public void onUserAudioAvailable(String userId, boolean available) {
            VLog.d(TAG, "onUserAudioAvailable, userId=" + userId + ", available" + available);
            if (available && !isStreamFromWeb(userId)) {
                //只处理native端，[web端由onRemoteUserEnter处理]
                doWhenRemoteStreamAvailable(userId, TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG, -1);
            }

            // 音频available只当做mute属性处理
            if (null != mRoomListener) {
                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject.put("audio", available);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
                mRoomListener.onDidUpdateOfStreamV2(userIdToStreamId(userId, TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG), jsonObject);
            }
        }

        @Override
        public void onUserVideoSizeChanged(String userId, int streamType, int newWidth, int newHeight) {
            VLog.d(TAG, "onUserVideoSizeChanged, userId=" + userId + ", streamType=" + streamType + ", newWidth=" + newWidth + ", newHeight=" + newHeight);
            if (null != mRoomListener) {
                JSONObject jsonObject = new JSONObject();
                try {
                    jsonObject.put("newWidth", newWidth);
                    jsonObject.put("newHeight", newHeight);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
                mRoomListener.onDidUpdateOfStreamV2(userIdToStreamId(userId, streamType), jsonObject);
            }
        }

        @Override
        public void onStatistics(TRTCStatistics statistics) {
            if (null != statistics) {
                LogReport.instance().updateLatestStatistics(statistics);
            }
        }
    }

    /**
     * UserId 算出streamId
     **/
    private String userIdToStreamId(String userId, int trtcStreamType) {
        if (isStreamFromWeb(userId)) {
            return userId;
        } else {
            StringBuilder stringBuilder = new StringBuilder(userId);
            stringBuilder
                    .append(":")
                    .append(TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG == trtcStreamType ? TRTC_STREAM_TYPE_MAIN : TRTC_STREAM_TYPE_AUXILIARY)
                    .append(":")
                    .append(TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG == trtcStreamType ? VRTCParams.VRTCStreamType.VhallStreamTypeAudioAndVideo : VRTCParams.VRTCStreamType.VhallStreamTypeScreen);
            return stringBuilder.toString();
        }
    }

    /**
     * 当远端视频/桌面共享可用事件到来时，判断该流Id(userId)是否存在，若不存在，则执行add操作并给业务层抛出onDidAddStream回调
     * <p>
     *
     * @param userId
     * @param trtcStreamType 小于0时，需要根据 vrtcStreamType 推到得出
     * @param vrtcStreamType 小于0时，需要根据 trtcStreamType 推到得出
     */
    private void doWhenRemoteStreamAvailable(String userId, int trtcStreamType, int vrtcStreamType) {
        HashMap<String, String> user = tryParseWebStyleUserId(userId);
        String streamId = userId;
        if (!isStreamFromWeb(userId)) {
            StringBuilder stringBuilder = new StringBuilder(userId);
            stringBuilder
                    .append(":")
                    .append(TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG == trtcStreamType ? TRTC_STREAM_TYPE_MAIN : TRTC_STREAM_TYPE_AUXILIARY)
                    .append(":")
                    .append(vrtcStreamType >= 0 ? vrtcStreamType : (TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG == trtcStreamType ? VRTCParams.VRTCStreamType.VhallStreamTypeAudioAndVideo : VRTCParams.VRTCStreamType.VhallStreamTypeScreen));
            streamId = stringBuilder.toString();
        }
        userIdList.put(streamId, userId); // 缓存trtc实际用到的userId

        if (!containsStreamId(streamId)) {

            addStream2Cache(mockRemoteStream(user.get("userId"), streamId, trtcStreamType, vrtcStreamType));
            if (null != mRoomListener) {
                // 回调业务定义的userId
                mRoomListener.onDidAddStream(user.get("userId"), streamId);
            }
            if (mAutoSubscribe) {
                subscribe(streamId);
            }
        }
    }

    /**
     * 流是否来自web端[web格式： main/auxiliary:uuid:vrtcstreamtype]
     *
     * @param userStreamId
     * @return
     */
    private boolean isStreamFromWeb(String userStreamId) {
        return !TextUtils.isEmpty(userStreamId) && userStreamId.contains(":");
    }

    /**
     * 解析原始userId，若为web侧格式，则解析出真实userId和vrtcStreamType
     *
     * @param originUserId
     * @return userId / vrtcStreamType
     */
    private HashMap<String, String> tryParseWebStyleUserId(String originUserId) {
        HashMap<String, String> resultMap = new HashMap<>();
        resultMap.put("vrtcStreamType", "-1");
        if (!TextUtils.isEmpty(originUserId) && originUserId.contains(":")) {
            String realUserId = null;
            String vrtcStreamType = null;
            try {
                String[] splitId = originUserId.split(":");
                realUserId = splitId[0];
                vrtcStreamType = splitId[2];
            } catch (Exception e) {
                e.printStackTrace();
            }

            if (!TextUtils.isEmpty(realUserId)) {
                resultMap.put("userId", realUserId);
                resultMap.put("vrtcStreamType", vrtcStreamType);
            } else {
                resultMap.put("userId", originUserId);
            }
        } else {
            resultMap.put("userId", originUserId);
        }
        return resultMap;
    }

    /**
     * @param userId 本地流userId
     * @param event  事件
     */
    private void notifyEvent(String userId, String event) {
        VHInteractiveApi.eventNotifyTRTC(mRoomParams.roomParams.inavRoomId, getCombinedStreamId(userId), event, mRoomParams.roomParams.accessToken, null);
    }

    /**
     * TRTC旁路/事件上报相关业务场景下对组合流ID，格式见: http://yapi.vhallops.com/project/48/interface/api/54002
     */
    private String getCombinedStreamId(String streamId) {
        Stream stream = getStreamById(streamId);
        if (null != stream) {
            if (!TextUtils.isEmpty(streamId) && (!streamId.contains(":"))) {
                StringBuilder stringBuilder = new StringBuilder(stream.userId);
                stringBuilder
                        .append(":")
                        .append(TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG == stream.streamTypeTrtc ? TRTC_STREAM_TYPE_MAIN : TRTC_STREAM_TYPE_AUXILIARY)
                        .append(":")
                        .append(stream.streamTypeVrtc);
                return stringBuilder.toString();
            }
        } else {
            VLog.e(TAG, "target streamId does not exists, streamId:" + streamId);
        }
        return streamId;
    }

    /**
     * native端 userId == streamId；web端需要经过 {@link this#tryParseWebStyleUserId(String)} 解析出真实的userId
     *
     * @param userId
     * @param streamId
     * @param trtcStreamType 小于0时，需要根据 vrtcStreamType 推到得出
     * @param vrtcStreamType 小于0时，需要根据 trtcStreamType 推到得出
     * @return
     */
    private Stream mockRemoteStream(String userId, String streamId, int trtcStreamType, int vrtcStreamType) {
        Stream remoteStream = new Stream();
        remoteStream.streamId = streamId;
        remoteStream.userId = userId;
        remoteStream.isLocal = false;
        remoteStream.streamTypeTrtc = trtcStreamType >= 0 ? trtcStreamType : (!isStreamFromWeb(streamId) && VRTCParams.VRTCStreamType.VhallStreamTypeScreen == vrtcStreamType ? TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_SUB : TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG);
        remoteStream.streamTypeVrtc = vrtcStreamType >= 0 ? vrtcStreamType : (TRTCCloudDef.TRTC_VIDEO_STREAM_TYPE_BIG == trtcStreamType ? VRTCParams.VRTCStreamType.VhallStreamTypeAudioAndVideo : VRTCParams.VRTCStreamType.VhallStreamTypeScreen);

        try (getStreamAttributes requestAttributes = new getStreamAttributes()) {
            remoteStream.attributes = requestAttributes.getStreamAttributes(mRoomParams.roomParams.accessToken, mRoomParams.roomParams.inavRoomId, streamId);
            VLog.i(TAG, "attributes:" + remoteStream.attributes);
        } catch (IOException e) {
            throw new RuntimeException(e);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }

        return remoteStream;

    }

    private String getUserId() {
        return null != mRoomParams ? mRoomParams.userId : null;
    }

    private boolean hasAudio(int vrtcStreamType) {
        return vrtcStreamType != VRTCParams.VRTCStreamType.VhallStreamTypeOnlyVideo;
    }

    private boolean isMainTRTCStream() {
        return TRTC_STREAM_TYPE_MAIN.equals(mLocalTRTCStreamType);
    }

    /**
     * 将通用抽象视频填充类型转换为TRTC模式
     *
     * @param scaleType
     * @return
     */
    private int convertScaleType2TRTCMode(int scaleType) {
        int trtcScaleType;
        switch (scaleType) {
            default:
            case VRTCParams.VRTCVideoFillMode.RENDER_MODE_NONE:
            case VRTCParams.VRTCVideoFillMode.RENDER_MODE_ASPECT_FILL:
                trtcScaleType = TRTCCloudDef.TRTC_VIDEO_RENDER_MODE_FILL;
                break;
            case VRTCParams.VRTCVideoFillMode.RENDER_MODE_FIT:
                trtcScaleType = TRTCCloudDef.TRTC_VIDEO_RENDER_MODE_FIT;
                break;
        }
        return trtcScaleType;
    }
}