package com.vhall.ilss;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import com.vhall.framework.VhallSDK;
import com.vhall.framework.connect.IVHService;
import com.vhall.framework.connect.VhallConnectService;
import com.vhall.logmanager.L;
import com.vhall.logmanager.LogInfo;
import com.vhall.logmanager.LogReporter;
import com.vhall.message.ConnectServer;
import com.vhall.rtc.VhallRTC;
import com.vhall.vhallrtc.client.Room;
import com.vhall.vhallrtc.client.Stream;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.concurrent.CopyOnWriteArraySet;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;

import static com.vhall.ilss.VHInteractiveApi.TYPE_INAV;

/**
 * Created by zwp on 2019-12-05
 */
public class VHOTOInteractive extends VhallRTC implements IVHService {
    private static final String TAG = "VHOTOInteractive";
    public static final String publish_inav_stream = "publish_inav_stream";//上麦
    public static final String askfor_inav_publish = "askfor_inav_publish";//邀请推流
    public static final String user_publish_callback = "user_publish_callback";//上下麦，拒绝上麦
    public static final String inav_close = "inav_close";//房间关闭
    public static final int STATUS_AGREE = 1;//接听
    public static final int STATUS_LEAVE = 2;//挂断
    public static final int STATUS_REJECT = 3;//拒绝

    private String targetId = "";//主叫时为被呼叫者id，被叫时为呼叫者id，即永远为1v1目标id
    private String mRoomId = "";
    private String mAccessToken = "";
    private String TOKEN = "";
    private Handler mHandler;
    private CopyOnWriteArraySet<String> mPermissions = new CopyOnWriteArraySet<>();
    private OTOInteractiveListener listener;

    public interface OTOInteractiveListener {
        void onRoomDidConnect();//房间连接成功

        void onRoomConnectError();//房间连接失败

        void onRoomReconnect();//断线重连成功

        void onServerConnecting();//服务连接中

        void onServerConnected();//服务连接成功

        void onServerDisConnected();//服务断开连接

        void onOTOCall(String callerId);//被邀请上麦即被呼叫

        void onOTOPositiveAnswer();//呼叫被接听

        void onOTONegativeAnswer();//呼叫被拒绝

        void onOTOCallEnd();//通话挂断

        void onDidAddStream(Stream stream);
    }

    public VHOTOInteractive(Context context) {
        super(context, true);
        setListener(roomDelegate);
        mHandler = new Handler(Looper.getMainLooper());
    }

    public void setOTOListener(OTOInteractiveListener listener) {
        this.listener = listener;
    }

    @Override
    public String getChannelId() {
        return mRoomId;
    }

    @Override
    public String getAccessToken() {
        return mAccessToken;
    }

    @Override
    public void onConnectStateChanged(ConnectServer.State state, int serverType) {
        if (listener != null && serverType == VhallConnectService.SERVER_MSG) {
            switch (state) {
                case STATE_CONNECTED:
                    listener.onServerConnected();
                    break;
                case STATE_DISCONNECT:
                    listener.onServerDisConnected();
                    break;
                case STATE_CONNECTIONG:
                    listener.onServerConnecting();
                    break;
            }

        }
    }

    @Override
    public void onMessage(String msg) {
        if (TextUtils.isEmpty(msg)) {
            return;
        }
        if (listener != null) {
            try {
                JSONObject result = new JSONObject(msg);
                String event = result.optString("event");
                if (event.equals(TYPE_INAV)) {
                    JSONObject data = result.optJSONObject("data");
                    String senderId = result.optString("third_party_user_id");//发送者id
                    String dataEvent = data.getString("inav_event");
                    String userId = data.optString("third_party_user_id");//目标id
                    if (!VhallSDK.getInstance().getmUserId().equals(senderId)) {
                        //自己发送的消息不处理
                        switch (dataEvent) {
                            case askfor_inav_publish://被呼叫
                                if (VhallSDK.getInstance().getmUserId().equals(userId)) {
                                    listener.onOTOCall(senderId);
                                    targetId = senderId;
                                }
                                break;
                            case user_publish_callback://呼叫响应回调
                                if (senderId.equals(targetId)) {//当且仅当响应用户为主叫或者被叫用户时才处理相关消息，保证房间内1v1逻辑不被干扰
                                    int status = data.optInt("status");
                                    if (status == STATUS_AGREE) {//接听后上麦成功
                                        listener.onOTOPositiveAnswer();//接听
                                    } else if (status == STATUS_LEAVE) {//接通后挂断
                                        listener.onOTOCallEnd();//挂断
                                    } else if (status == STATUS_REJECT) {
                                        listener.onOTONegativeAnswer();//拒接
                                    }
                                }
                                break;
                        }
                    }

                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * @param roomid
     * @param accessToken
     * @param callback
     */
    public void init(final String roomid, final String accessToken, final VHInteractive.InitCallback callback) {
        VHInteractiveApi.getRoomInfo(roomid, accessToken,null,VHInteractive.MODE_RTC,VHInteractive.ROLE_HOST, new Callback() {
            @Override
            public void onFailure(final Call call, final IOException e) {
                L.e(TAG, e.getMessage());
                trackInitEvent(LogReporter.LOG_ERROR_NET);
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (callback != null)
                            callback.onFailure(-1, "error network,please try later！");
                    }
                });
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                final String content = response.body().string();
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            JSONObject result = new JSONObject(content);
                            String msg = result.optString("msg");
                            int code = result.optInt("code");
                            if (code == 200) {
                                mRoomId = roomid;
                                mAccessToken = accessToken;
                                JSONObject data = result.optJSONObject("data");
                                TOKEN = data.getString("inav_token");
                                JSONArray permissions = data.optJSONArray("permission");
                                if (permissions != null && permissions.length() > 0) {
                                    for (int i = 0; i < permissions.length(); i++) {
                                        mPermissions.add((String) permissions.get(i));
                                    }
                                }
                                LogInfo.getInstance().roomId = mRoomId;
                                setDataReport(new JSONObject(LogInfo.getInstance().toString()));
//                                Room.setLogReportData(new JSONObject(LogInfo.getInstance().toString()));
                                if (callback != null)
                                    callback.onSuccess();
                                trackInitEvent();
                            } else {
                                if (callback != null)
                                    callback.onFailure(code, msg);
                                trackInitEvent(code + ":" + msg);
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                            if (callback != null)
                                callback.onFailure(-1, "Request data exception");
                            trackInitEvent(LogReporter.LOG_ERROR_EXCEPTION);
                        }
                    }
                });

            }
        });
    }

    /**
     * @param attribute 自定义信息，其他用户订阅时，可以从Stream中获取
     */
    public void enterRoom(String attribute) {
        if (!TextUtils.isEmpty(TOKEN))
            super.enterRoom(TOKEN, attribute);
        VhallSDK.getInstance().join(this);
    }

    /**
     * 1V1 呼叫
     */
    public int OTOCall(String userId) {
        final int[] result = {-1};
        targetId = userId;
        if (isInviteAvailable())
            VHInteractiveApi.invitePush(mRoomId, mAccessToken, userId, new Callback() {
                @Override
                public void onFailure(Call call, IOException e) {

                }

                @Override
                public void onResponse(Call call, Response response) throws IOException {
                    if (response.code() == 200) {
                        String res = response.body().string();
                        try {
                            JSONObject obj = new JSONObject(res);
                            int code = obj.optInt("code");
                            if (code == 200) {
                                result[0] = 1;
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
        return result[0];
    }

    public int OTOPositiveAnswer(){
        return OTOAnswer(STATUS_AGREE);
    }

    public int OTONegativeAnswer(){
        if(mLocalStream!=null && mLocalStream.isPublish){
            return OTOAnswer(STATUS_LEAVE);
        }else{
            return OTOAnswer(STATUS_REJECT);
        }
    }


    /**
     * 1V1 应答
     * type 1 上麦(默认) 2 下麦 3 拒绝上麦
     */
    private int OTOAnswer(int type) {
        final int[] result = {-1};
        if (type == STATUS_AGREE) {//接听后自动上麦
            super.publish();
        } else {
            super.unpublish();
        }
        VHInteractiveApi.onStateChanged(mRoomId, mAccessToken, String.valueOf(mLocalStream.streamId), type, new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                if (response.code() == 200) {
                    try {
                        JSONObject obj = new JSONObject(response.body().string());
                        int code = obj.optInt("code");
                        if (code == 200) {
                            result[0] = 1;
                        }
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }
            }
        });
        return result[0];
    }

    public Stream createLocalStream(String extraInfo) {
        JSONObject option = new JSONObject();
        try {
            option.put(Stream.kFrameResolutionTypeKey, 3);
            option.put(Stream.kMinBitrateKbpsKey, 1200);
            option.put(Stream.kCurrentBitrateKey, 1500);
            option.put(Stream.kMaxBitrateKey, 1900);
            option.put(Stream.kStreamOptionStreamType, Stream.VhallStreamType.VhallStreamTypeAudioAndVideo.getValue());
            option.put(Stream.kNumSpatialLayersKey, 2);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return super.createLocalStream(option, extraInfo);
    }


    @Override
    public void publish() {
        if (isPushAvailable()) {
            super.publish();
            VHInteractiveApi.onStateChanged(mRoomId, mAccessToken, String.valueOf(mLocalStream.streamId), STATUS_AGREE, null);
        } else {
            Log.e(TAG, "Do not have publish permission.");
        }
    }

    @Override
    public void unpublish() {
        super.unpublish();
        VHInteractiveApi.onStateChanged(mRoomId, mAccessToken, String.valueOf(mLocalStream.streamId), STATUS_LEAVE, null);
    }

    @Override
    public void leaveRoom() {
        super.leaveRoom();
        VhallSDK.getInstance().leave(this);
        VHInteractiveApi.onLeave(mRoomId, mAccessToken);
    }

    public void getMembers(Callback callback) {
        if (TextUtils.isEmpty(mRoomId))
            return;
        VHInteractiveApi.getRoomMember(mRoomId, mAccessToken, callback);
    }

    public boolean isPushAvailable() {
        if (mPermissions != null && mPermissions.contains(publish_inav_stream))
            return true;
        else
            return false;
    }

    public boolean isInviteAvailable() {
        if (mPermissions != null && mPermissions.contains(askfor_inav_publish))
            return true;
        else
            return false;
    }

    private Room.RoomDelegate roomDelegate = new Room.RoomDelegate() {
        @Override
        public void onDidConnect(Room room, JSONObject jsonObject) {
            //enterRoom 成功后回调；连接成功
            Log.i(TAG, "onDidConnect");
            //OTO呼叫，进入房间后默认等待呼叫操作
            if (listener != null) {
                listener.onRoomDidConnect();
            }
        }

        @Override
        public void onDidError(Room room, Room.VHRoomErrorStatus vhRoomErrorStatus, String s) {
            Log.i(TAG, "onDidError");
            //进入房间失败回调
            if (listener != null) {
                listener.onRoomConnectError();
            }
        }

        @Override
        public void onDidPublishStream(Room room, Stream stream) {
            //发布本地流成功回调
        }

        @Override
        public void onDidInternalStreamAdded(Room room, Stream stream) {

        }

        @Override
        public void onDidInternalStreamRemoved(Room room, Stream stream) {

        }

        @Override
        public void onDidInternalStreamFailed(Room room, Stream stream, JSONObject jsonObject) {

        }

        @Override
        public void onDidUnPublishStream(Room room, Stream stream) {
            //取消本地流成功回调
        }

        @Override
        public void onDidSubscribeStream(Room room, Stream stream) {
            //订阅远端流成功回调
            if (listener != null) {
                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        listener.onDidAddStream(stream);
                    }
                });
            }
        }

        @Override
        public void onDidUnSubscribeStream(Room room, Stream stream) {
            //取消订阅远端流成功回调
            Log.e(TAG, "onDidUnSubscribeStream: ");
            unpublish();
        }

        @Override
        public void onDidChangeStatus(Room room, Room.VHRoomStatus vhRoomStatus) {
            //连接状态改变回调
        }

        @Override
        public void onDidAddStream(Room room, Stream stream) {
            //有流加入房间的回调
            if(!targetId.equals(stream.userId)){//不是目标用户上麦，不订阅，不处理
                return;
            }
            try {
                stream.streamOption.put(Stream.kDualKey, 1);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            room.subscribe(stream);
            if (mLocalStream != null && !mLocalStream.isPublish) {
                publish();
            }
        }

        @Override
        public void onDidRemoveStream(Room room, Stream stream) {
            //有流退出房间的回调，代表另一方挂断，当前用户自动下麦
            Log.e(TAG, "onDidRemoveStream: ");
            if(targetId.equals(stream.userId)){//目标用户下麦后，自动下麦，否则不处理
                unpublish();
            }
        }

        @Override
        public void onDidUpdateOfStream(Stream stream, JSONObject jsonObject) {
            //更新音视频流属性的回调
        }

        @Override
        public void onReconnect(int i, int i1) {
            //房间重连的回调
            if (listener != null) {
                listener.onRoomReconnect();
            }
        }

        @Override
        public void onStreamMixed(JSONObject jsonObject) {
            //通知设置旁路的回调--混流数据准备完成可以开始旁路直播时的回调（目前有流加入均会回调）
        }
    };


    private void trackInitEvent(String error){
        JSONObject params = new JSONObject();
        try {
            params.put("inavId",mRoomId);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        LogReporter.getInstance().setErr(error);
        LogReporter.getInstance().onCollection(LogReporter.LOG_EVENT_INITILSS, false, params);
    }

    private void trackInitEvent(){
        JSONObject params = new JSONObject();
        try {
            params.put("inavId",mRoomId);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        LogReporter.getInstance().onCollection(LogReporter.LOG_EVENT_INITILSS, params);
    }

}
