package com.vhall.ilss;

import static com.vhall.ilss.VHInteractiveApi.TYPE_INAV;
import static com.vhall.rtc.VRTCCode.ERR_BROADCAST_URL_NULL;
import static com.vhall.rtc.VRTCCode.ERR_SDK_DEPNONE;

import android.content.Context;
import android.media.projection.MediaProjection;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;

import com.vhall.framework.VhallBaseSDK;
import com.vhall.framework.common.ICallback;
import com.vhall.framework.connect.IVHService;
import com.vhall.framework.connect.VhallConnectService;
import com.vhall.logmanager.LogInfo;
import com.vhall.logmanager.LogReporter;
import com.vhall.logmanager.VLog;
import com.vhall.message.ConnectServer;
import com.vhall.rtc.Utils;
import com.vhall.rtc.VRTCCode;
import com.vhall.rtc.VRTCParams;
import com.vhall.rtc.absrtc.ISwitchFocusModeListener;
import com.vhall.rtc.absrtc.IVHRTC;
import com.vhall.rtc.absrtc.IVHRTCRenderView;
import com.vhall.rtc.absrtc.VHInteractiveFactory;
import com.vhall.vhallrtc.client.Room;
import com.vhall.vhallrtc.client.Stream;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URLDecoder;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;

/**
 * @author：Jooper Email：jooperge@163.com
 * 描述：RTC能力对外开放入口，与原《VHInteractive》相比，新增了对抽象RTC能力对支持，可对接如VRTC、TRTC
 * 修改历史:
 * <p>
 * 创建于： 2022/10/21
 */
public class VHInteractiveV2 implements IVHService {
    private static final String TAG = "VHInteractive";

    private IVHRTC mRtcInstance = null;

    private VRTCParams.VRTCRoomParamsInner mRoomParams;
    private VRTCParams.VRTCBroadcastParamInner mBroadcastParams;
    private String mRoomId = "";//互动房间ID
    private String mAccessToken = "";
    private String mBroadcastId = "";//旁路房间id
    private String mPushUrl = "";//旁路推流地址
    private String mMode = VRTCCode.MODE_RTC;//
    private String mRole = VRTCCode.ROLE_HOST;//
    private CopyOnWriteArraySet<String> mPermissions = new CopyOnWriteArraySet<>();
    private OnMessageListener mMessageListener;
    private VhallConnectService.OnConnectStateChangedListener mOnConnectChangedListener;
    private JSONObject _bgColorJSON = null;
    private JSONObject _boardColorJSON = null;
    private Handler mHandler;
    private Context mContext;
    private VHRoomListener mRoomListener;

    private VHInteractiveV2() {
    }

    /**
     * @param context
     * @param roomParams       房间基本信息
     * @param roomInfoCallback 房间信息获取回调
     */
    public VHInteractiveV2(Context context, VRTCParams.VRTCRoomParams roomParams, ICallback roomInfoCallback) {
        this(context, roomParams, roomInfoCallback, null);
    }

    /**
     * @param context
     * @param roomParams
     * @param roomInfoCallback
     * @param rtcSDKType       取值如：{@link VRTCParams#RTCTYPE_VRTC}只有在进入房间之前确定使用sdk type的前提下，才可设置该参数，否则需置为NULL或者调用不带该参数的构造器
     *                         <p>
     *                         该参数存在的意义：
     *                         1。在getRoomInfo成功之前，无法自动感知当前房间使用的rtc sdk类型，所以会造成getRoomInfo成功之前无法进行本地预览的问题
     *                         2。基于该问题，新增了预置 sdk type的参数，设置后可立即创建本地预览
     *                         3。若该参数赋值与getRoomInfo中回调的sdkType不一致，则会被强制覆盖
     *                         4。由于VRTC/TRTC可组合或独立引入app打包依赖，如：若rtcSDKType设置为VRTC，而业务应用没有引入VRTC sdk，则依然无法在getRoomInfo成功之前进行本地预览
     */
    public VHInteractiveV2(Context context, VRTCParams.VRTCRoomParams roomParams, ICallback roomInfoCallback, String rtcSDKType) {

        mContext = context;
        mHandler = new Handler(Looper.getMainLooper());
        mRoomParams = new VRTCParams.VRTCRoomParamsInner(roomParams);
        mRoomParams.userId = VhallBaseSDK.getInstance().getmUserId();

        if (!TextUtils.isEmpty(rtcSDKType)) {
            doRtcInit(rtcSDKType, null);
        }

        init(roomParams.inavRoomId, roomParams.accessToken, roomParams.broadcastRoomId, mRoomParams.mode, roomParams.role, new ICallback() {
            @Override
            public void onSuccess() {
                if (null != roomInfoCallback) {
                    roomInfoCallback.onSuccess();
                }
            }

            @Override
            public void onFailure(int code, String msg) {
                if (null != roomInfoCallback) {
                    roomInfoCallback.onFailure(code, msg);
                }
            }
        });
    }

    public void setListener(VHRoomListener roomListener) {
        mRoomListener = roomListener;
    }

    /**
     * @param roomid
     * @param broadcastid 旁路直播id
     * @param accessToken
     * @param mode        MODE_RTC 实时通信场景，例如互动连麦 MODE_LIVE 无延迟直播房间 直播场景，例如无延迟直播
     *                    应用场景模式，选填，可选值参考 VHInteractiveRoomMode。支持版本：2.4.0及以上
     * @param role        ROLE_HOST   可以发布和订阅互动流  ROLE_AUDIENCE 只能订阅互动流，无法发布
     *                    用户角色，选填，可选值参考下文VHInteractiveRoomRole。当mode为rtc模式时，不需要配置role。支持版本：2.4.0及以上
     * @param callback
     */
    private void init(final String roomid, final String accessToken, final String broadcastid, final String mode, final String role, final ICallback callback) {
        mRoomId = roomid;
        mMode = mode;//
        mRole = role;//
        if (TextUtils.isEmpty(mMode)) {
            mMode = VRTCCode.MODE_RTC;
        }
        if (!mMode.equals(VRTCCode.MODE_LIVE)) {
            mRole = VRTCCode.ROLE_HOST;
        }
        mAccessToken = accessToken;
        getRoomInfo(roomid, accessToken, broadcastid, mode, role, callback);
    }

    public void setOnMessageListener(OnMessageListener listener) {
        this.mMessageListener = listener;
    }

    @Deprecated
    public void setOnConnectChangedListener(VhallConnectService.OnConnectStateChangedListener listener) {
        mOnConnectChangedListener = listener;
    }

    private void getRoomInfo(final String roomid, final String accessToken, final String broadcastid, final String mode, final String role, final ICallback callback) {
        VHInteractiveApi.getRoomInfo(roomid, accessToken, broadcastid, mode, role, new Callback() {
            @Override
            public void onFailure(final Call call, final IOException e) {
                VLog.d(TAG, e.getMessage());
                trackInitEvent(LogReporter.LOG_ERROR_NET);
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (callback != null)
                            callback.onFailure(-1, "error network,please try later！");
                    }
                });
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                final String content = response.body().string();
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            JSONObject result = new JSONObject(content);
                            String msg = result.optString("msg");
                            int code = result.optInt("code");
                            if (code == 200) {
                                mBroadcastId = broadcastid;
                                JSONObject data = result.optJSONObject("data");

                                mRoomParams.inavToken = data.getString("inav_token");

                                mPushUrl = data.getString("pushUrl");
                                JSONObject logObj = data.optJSONObject("log_info");
                                JSONArray permissions = data.optJSONArray("permission");

                                boolean initResult = doRtcInit(data.optString("type"), data.optString("sdk_app_id"));
                                if (initResult) {
                                    if (permissions != null && permissions.length() > 0) {
                                        for (int i = 0; i < permissions.length(); i++) {
                                            mPermissions.add((String) permissions.get(i));
                                        }
                                    }
                                    //设置日志地址
                                    LogInfo.getInstance().roomId = mRoomId;
                                    if (logObj != null) {
                                        LogInfo.getInstance().initBaseData(logObj);
                                    }
                                    mRtcInstance.setDataReport(new JSONObject(LogInfo.getInstance().toString()));
                                    if (mRoomParams.type.equals(VRTCParams.RTCTYPE_VRTC)) {
                                        /* verify token */
                                        if (!Utils.verifyToken(mRoomParams.inavToken) && null != callback) {
                                            callback.onFailure(-1, "token error: " + mRoomParams.inavToken);
                                            trackInitEvent();
                                            return;
                                        }
                                    }
                                    if (callback != null) {
                                        callback.onSuccess();
                                    }
                                    trackInitEvent();
                                } else {
                                    if (callback != null) {
                                        callback.onFailure(ERR_SDK_DEPNONE, "SDK不存在，请参考官方集成手册");
                                        return;
                                    }
                                }
                            } else {
                                if (callback != null) {
                                    callback.onFailure(code, msg);
                                }
                                trackInitEvent(code + ":" + msg);
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                            if (callback != null) {
                                callback.onFailure(-1, "Request data exception");
                            }
                            trackInitEvent(LogReporter.LOG_ERROR_EXCEPTION);
                        }
                    }
                });
            }
        });
    }

    private boolean doRtcInit(String type, String trtcSdkAppId) {
        if (!TextUtils.isEmpty(trtcSdkAppId)) {
            try {
                mRoomParams.trtcSdkAppId = Integer.parseInt(trtcSdkAppId);
            } catch (NumberFormatException e) {
                e.printStackTrace();
            }
        }
        if (null != mRtcInstance && mRoomParams.type.equals(type)) {
            //已初始化，且sdk类型与getRoomInfo获取的sdk类型一致
            return true;
        }
        mRoomParams.type = type;

        mRtcInstance = VHInteractiveFactory.newRTC(mContext, mRoomParams);
        return null != mRtcInstance;
    }

    public void addExtraLogParam(String s) {
        LogInfo.getInstance().setExtraData(s);
    }

    /**
     * 业务自定义数据，目前仅VRTC支持，收到远端流时可从 Stream#getUserAttributes()获取
     *
     * @param attribute
     */
    public void enterRoom(String attribute) {
        enterRoom(true, true, attribute);
    }

    /**
     * @param attribute     业务自定义数据，目前仅VRTC支持，收到远端流时可从 Stream#getUserAttributes()获取
     * @param autoPublish   是否自动上麦，需在创建本地流后使用才有效【角色为观众时不生效】
     * @param autoSubscribe 是否自动订阅
     */
    public void enterRoom(boolean autoPublish, boolean autoSubscribe, String attribute) {
        if (null != mRtcInstance) {
            if (!isPushAvailable()) {
                autoPublish = false;
            }
            VhallBaseSDK.getInstance().join(this);//TODO 待join事件返回时再执行真正待enterRoom，确保vhall相关服务前置且正常
            mRtcInstance.enterRoom(mRoomParams, autoPublish, autoSubscribe, attribute, mRoomListener);
        } else {
            VLog.e(TAG, "enterRoom#please call VHInteractiveV2#init first");
        }
    }

    /**
     * 本地预览
     * <p>
     * 注意：
     * 1。若创建该(VHInteractiveV2)对象时，无法预先判断rtc sdk type，则需要在创建VHInteractiveV2中的ICallback成功后再调用**该方法**，否则会因为无法创建mRtcInstance而异常
     * 2。若创建该(VHInteractiveV2)对象时，**可以**预先判断rtc sdk type，则可以在VHInteractiveV2对象创建成功后同步调用**该方法**，可立即进行本地预览
     *
     * @param renderView  不需要则传空
     * @param videoParam  video采集配置参数
     * @param streamParam 流类型/属性等配置
     */
    public void createLocalStream(IVHRTCRenderView renderView, VRTCParams.VRTCVideoEncodeParam videoParam, VRTCParams.VRTCStreamParam streamParam) {
        if (null == streamParam) {
            throw new IllegalArgumentException("streamParam should not be null");
        } else {
            if (null != mRtcInstance) {
                mRtcInstance.startLocalPreview(renderView, videoParam, streamParam, true);
            } else {
                VLog.e(TAG, "createLocalStream#please call VHInteractiveV2#init first");
            }
        }
    }

    /**
     * 创建远端流的预览
     *
     * @param renderView 不需要则传空
     * @param streamId
     */
    public void createRemoteView(IVHRTCRenderView renderView, String streamId) {
        if (null != mRtcInstance) {
            mRtcInstance.startRemoteView(renderView, streamId);
        } else {
            VLog.e(TAG, "createLocalStream#please call VHInteractiveV2#init first");
        }
    }

    /**
     * 创建屏幕分享(录屏直播)本地流
     *
     * @param mediaProjection VRTC需要由业务层创建；TRTC传Null即可
     * @param videoParam      推荐传Null，此时sdk内部会使用流媒体侧推荐的适合屏幕分享场景的分辨率/帧率/码率配置
     * @param streamParam     流类型/属性等配置
     */
    public void createScreenShareStream(MediaProjection mediaProjection, VRTCParams.VRTCVideoEncodeParam videoParam, VRTCParams.VRTCStreamParam streamParam) {
        if (null == streamParam) {
            throw new IllegalArgumentException("streamParam should not be null");
        } else {
            if (null != mRtcInstance) {
                mRtcInstance.createScreenShareStream(mediaProjection, videoParam, streamParam);
            } else {
                VLog.e(TAG, "createLocalStream#please call VHInteractiveV2#init first");
            }
        }
    }

    /**
     * 停止屏幕共享
     * VRTC: leaveRoom
     * TRTC: stopScreenShare
     */
    public void stopScreenShare() {
        if (null != mRtcInstance) {
            mRtcInstance.stopScreenShare();
        } else {
            VLog.e(TAG, "stopScreenShare#please call VHInteractiveV2#init first");
        }
    }

    public String getUserIdByStreamId(String streamId) {
        if (null != mRtcInstance) {
            return mRtcInstance.getUserIdByStreamId(streamId);
        }
        return "";
    }

    /**
     * 上麦(推流)
     *
     * @return
     */
    public boolean publish() {
        if (isPushAvailable()) {
            if (null == mRtcInstance) {
                VLog.e(TAG, "publish#please call VHInteractiveV2#init first");
                return false;
            }
            mRtcInstance.publish();
            return true;
        } else {
            VLog.e(TAG, "Do not have publish permission.");
            return false;
        }
    }

    /**
     * 下麦(取消推流)
     */
    public void unPublish() {
        if (null == mRtcInstance) {
            VLog.e(TAG, "unpublish#please call VHInteractiveV2#init first");
            return;
        }
        mRtcInstance.unPublish();
    }

    /**
     * 订阅
     */
    public void subscribe(String streamId) {
        if (null == mRtcInstance) {
            VLog.e(TAG, "subscribe#please call VHInteractiveV2#init first");
            return;
        }
        mRtcInstance.subscribe(streamId);
    }

    /**
     * 取消订阅
     */
    public void unSubscribe(String streamId) {
        if (null == mRtcInstance) {
            VLog.e(TAG, "subscribe#please call VHInteractiveV2#init first");
            return;
        }
        mRtcInstance.unSubscribe(streamId);
    }

    public void leaveRoom() {
        if (null == mRtcInstance) {
            VLog.e(TAG, "leaveRoom#please call VHInteractiveV2#init first");
            return;
        }
        mRtcInstance.leaveRoom();
        VhallBaseSDK.getInstance().leave(this);
        VHInteractiveApi.onLeave(mRoomId, mAccessToken);
    }

    /**
     * 强制离开房间，一般用于自己异常退出互动房间后，再进入提示已在房间中使用，不会拉入黑名单
     *
     * @param userid
     */
    public void forceLeaveRoom(String userid, Callback callback) {
        if (userid == null) userid = VhallBaseSDK.getInstance().getmUserId();
        VHInteractiveApi.forceLeaveRoom(mRoomId, mAccessToken, userid, callback);
    }

    /**
     * 旁路直播
     *
     * @param broadcastParam 旁路参数，需要关闭时传Null或调用 {@link this#stopBroadcastRoom(ICallback)}
     * @param callback       configRoomBroadCast 配置旁路混流参数 流程，1、通知微吼云旁路状态改变、2、调用sdk接口修改旁路、 3、失败后通知微吼云恢复旁路状态
     *                       http://wiki.vhallops.com/pages/viewpage.action?title=Room&spaceKey=media#Room-%E9%85%8D%E7%BD%AE%E6%97%81%E8%B7%AF%E5%8F%82%E6%95%B0
     */
    public void broadcastRoom(VRTCParams.VRTCBroadcastParam broadcastParam, ICallback callback) {
        if (null != mRtcInstance) {
            if (!isBrocastRoomAvailable()) {
                if (callback != null)
                    callback.onFailure(VRTCCode.ERR_BROADCAST_PERMISSION_FAILED, "请开通设置旁路权限");
                return;
            }

            if (null != broadcastParam) {
                if (TextUtils.isEmpty(mBroadcastId)) {
                    if (callback != null)
                        callback.onFailure(VRTCCode.ERR_BROADCAST_ID_NULL, "未设置旁路ID");
                    return;
                }

                if (TextUtils.isEmpty(mPushUrl)) {
                    if (callback != null)
                        callback.onFailure(ERR_BROADCAST_URL_NULL, "旁路推流地址为空");
                    return;
                }

                mBroadcastParams = new VRTCParams.VRTCBroadcastParamInner(broadcastParam);
                mBroadcastParams.inavRoomId = mRoomId;
                mBroadcastParams.broadcastRoomId = mBroadcastId;
                mBroadcastParams.pushUrl = mPushUrl;
                mBroadcastParams.isAdaptiveLayoutMode = broadcastParam.layoutMode >= 1000;
                mBroadcastParams.compat_layout_mode = Utils.compatAdaptiveLayoutMode(broadcastParam.layoutMode);

                try {
                    mBroadcastParams.compat_profile_json = Utils.compatBroadcastProfile(broadcastParam.profile);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            } else {
                mBroadcastParams = null;
            }
            mRtcInstance.broadcastRoom(mBroadcastParams, mAccessToken, callback);
        }
    }

    /**
     * 关闭旁路直播
     *
     * @param callback
     */
    public void stopBroadcastRoom(ICallback callback) {
        broadcastRoom(null, callback);
    }

    /**
     * 设置混流主屏
     *
     * @param streamId 为Null代表本地流
     * @param callback
     */
    public void setMixLayoutMainScreen(String streamId, ICallback callback) {
        if (null != mRtcInstance) {
            mRtcInstance.setMixLayoutMainScreen(streamId, callback);
        }
    }

    /**
     * **仅适用与VRTC**
     * 收到的InternalStream 是否是云渲染流
     *
     * @param config 来源：{@link Room.RoomDelegate#onDidInternalStreamAdded(Room, Stream)}
     * @return
     */
    public boolean isInternalStreamDocCloudRender(JSONObject config) {
        return null != config && config.has("type") && config.optString("type").equals(VRTCCode.MIX_INTERNALSTREAM_TYPE_DOC_CLOUD_RENDER);
    }

    /**
     * 不推荐，请使用{@link this#setMixLayoutMode(int, ICallback)}
     */
    @Deprecated
    public void broadcastLayout(int layout, ICallback callback) {
        setMixLayoutMode(layout, callback);
    }

    /**
     * 修改旁路布局
     *
     * @param layoutMode 见{@link VRTCCode.BroadcastLayout}
     * @param callback
     */
    public void setMixLayoutMode(int layoutMode, ICallback callback) {
        if (null == mBroadcastParams) {
            callback.onFailure(ERR_BROADCAST_URL_NULL, "请先开启旁路");
            return;
        }
        if (!isBrocastRoomAvailable()) {
            if (callback != null)
                callback.onFailure(VRTCCode.ERR_BROADCAST_PERMISSION_FAILED, "请开通设置旁路权限");
            return;
        }

        if (null != mRtcInstance) {
            mRtcInstance.setMixLayoutMode(mBroadcastParams, layoutMode, callback);
        }
    }

    /**
     * 设置旁路背景色
     *
     * @param hexColor 支持格式： 0x000000、#000000、000000；传Null代表取消背景色(重置为流媒体侧预置的背景色0x333338)
     * @param callback
     */
    public void setMixBackgroundColor(String hexColor, ICallback callback) {
        if (null == mBroadcastParams) {
            callback.onFailure(ERR_BROADCAST_URL_NULL, "请先开启旁路");
            return;
        }
        if (null != mRtcInstance) {
            mRtcInstance.setMixBackgroundColor(mBroadcastParams, hexColor, callback);
        }
    }

    /**
     * 设置旁路窗格边框背景色
     *
     * @param border   传Null代表关闭，另可以创建border后置exist字段为false也可。
     * @param callback
     */
    public void setMixBorderColor(VRTCParams.VRTCBroadcastParam.Border border, ICallback callback) {
        if (null != mRtcInstance) {
            mRtcInstance.setMixBorderColor(border, callback);
        }
    }

    /**
     * 设置旁路画布背景图
     *
     * @param url      VRTC只支持https; 传Null代表关闭
     * @param cropType TRTC暂不支持；详情见：{@link VRTCCode#MCU_BG_MODE_SCALE} {@link VRTCCode#MCU_BG_MODE_ASPECTFIT} {@link VRTCCode#MCU_BG_MODE_FILL}
     * @param callback
     */
    public void setMixBackgroundImage(String url, int cropType, ICallback callback) {
        if (null == mBroadcastParams) {
            callback.onFailure(ERR_BROADCAST_URL_NULL, "请先开启旁路");
            return;
        }
        if (null != mRtcInstance) {
            mRtcInstance.setMixBackgroundImage(mBroadcastParams, url, cropType, callback);
        }
    }

    /**
     * 设置窗格占位图
     *
     * @param url      VRTC只支持https; 传Null代表关闭
     * @param callback
     */
    public void setMixPlaceholderImage(String url, ICallback callback) {
        if (null != mRtcInstance) {
            mRtcInstance.setMixPlaceholderImage(url, callback);
        }
    }

    /**
     * 启动文档容屏
     *
     * @param channelId
     * @param callback
     */
    public void startDocCloudRender(String channelId, final ICallback callback) {
        if (null != mRtcInstance) {
            mRtcInstance.docCloudRender(VhallBaseSDK.getInstance().getAPP_ID(), channelId, true, callback);
        }
    }

    /**
     * 关闭文档容屏
     *
     * @param channelId
     * @param callback
     */
    public void stopDocCloudRender(String channelId, final ICallback callback) {
        if (null != mRtcInstance) {
            mRtcInstance.docCloudRender(VhallBaseSDK.getInstance().getAPP_ID(), channelId, false, callback);
        }
    }

    /**
     * 设置互动预览画面裁剪模式
     *
     * @param scaleType {@link VRTCParams.VRTCVideoFillMode}
     */
    public void setScaleType(int scaleType) {
        if (null != mRtcInstance) {
            mRtcInstance.setScaleType(scaleType);
        }
    }

    /**
     * 直播过程中修改视频分辨率
     *
     * @param videoProfile 详情见{@link com.vhall.rtc.VRTCCode.RTCVideoProfile}
     */
    public void updateVideoResolution(int videoProfile) {
        if (null != mRtcInstance) {
            mRtcInstance.setVideoResolution(videoProfile);
        }
    }

    /**
     * 根据流ID进行视频mute/unmute
     *
     * @param streamId 需要mute本地流时传Null即可
     * @param mute
     */
    public void muteVideo(String streamId, boolean mute) {
        if (null != mRtcInstance) {
            mRtcInstance.muteVideo(streamId, mute);
        }
    }

    /**
     * 根据流ID进行音频mute/unmute
     *
     * @param streamId
     * @param mute
     */
    public void muteAudio(String streamId, boolean mute) {
        if (null != mRtcInstance) {
            mRtcInstance.muteAudio(streamId, mute);
        }
    }

    /**
     * 切换前后摄像头
     */
    public void switchCamera() {
        if (null != mRtcInstance) {
            mRtcInstance.switchCamera();
        }
    }

    /**
     * 获取当前摄像头是否是前置摄像头
     *
     * @return
     */
    public boolean isFrontCamera() {
        if (null != mRtcInstance) {
            return mRtcInstance.isFrontCamera();
        }
        return false;
    }

    /**
     * 设置本地流摄像头镜像
     *
     * @param isMirror
     */
    public void setMirror(boolean isMirror) {
        if (null != mRtcInstance) {
            mRtcInstance.setMirror(isMirror);
        }
    }

    /**
     * 摄像头对焦模式切换
     *
     * @param enableAuto
     * @param switchFocusModeListener TRTC模式下暂无法感知切换状态的回调，故会立即返回true
     */
    public void switchAutoFocusMode(boolean enableAuto, ISwitchFocusModeListener switchFocusModeListener) {
        if (null != mRtcInstance) {
            mRtcInstance.switchAutoFocusMode(enableAuto, switchFocusModeListener);
        }
    }

    /**
     * 获取权限列表
     *
     * @return
     */
    public Set<String> getPermissions() {
        return mPermissions;
    }

    @Override
    public String getChannelId() {
        return mRoomId;
    }

    @Override
    public String getAccessToken() {
        return mAccessToken;
    }

    @Override
    public void onConnectStateChanged(ConnectServer.State state, int serverType) {
        if (mOnConnectChangedListener != null) {
            mOnConnectChangedListener.onStateChanged(state, serverType);
        }
    }

    @Override
    public void onMessage(String msg) {
        if (TextUtils.isEmpty(msg)) return;
        VLog.d(TAG, "onMsg:" + msg);
        try {
            JSONObject result = new JSONObject(msg);
            String event = result.optString("event");
            if (mMessageListener != null) {
                /**
                 * 收到消息通知用户刷新用户状态
                 * 收到消息立刻通知更新用户状态
                 * 1.房间内用户状态为：
                 *  1 推流中 2 观看中 3 受邀中 4 申请中 5 被踢出
                 */
                mMessageListener.onRefreshMemberState();
            }
            //temp 临时处理首次打开房间，获取房间详情结果在join消息之前导致无法识别正确的host. 下一步：优化framework机制，将init/start的状态与各服务绑定并作为前置接口，仅成功后允许继续执行功能模块api
            if (event.equals("Join")) {
                mRtcInstance.setDataReport(new JSONObject(LogInfo.getInstance().toString()));
            }
            if (!event.equals(TYPE_INAV)) {
                /**
                 * 非互动消息（上下线消息）
                 * 因为最外层加了消息解析，故更新逻辑
                 */
                JSONObject text;
                if (!TextUtils.isEmpty(result.optString("text"))) {
                    text = new JSONObject(URLDecoder.decode(result.optString("text")));
                } else {
                    text = result;
                }
                if (mMessageListener != null && text != null)
                    mMessageListener.onRefreshMembers(text);
                return;
            }
            JSONObject data = result.getJSONObject("data");
            String dataEvent = data.getString("inav_event");
            String userId = data.optString("third_party_user_id");
            switch (dataEvent) {
                case VRTCCode.apply_inav_publish://申请上麦消息
                    if (userId.equals(VhallBaseSDK.getInstance().mUserId))//自己不审核自己上麦申请
                        break;
                    //判断当前用户是否有审核权限
                    if (isCheckReqAvailable() && mMessageListener != null)
                        mMessageListener.onMessage(data);
                    break;
                case VRTCCode.audit_inav_publish://上麦审核结果
                    if (userId.equals(VhallBaseSDK.getInstance().mUserId)) {
                        int status = data.getInt("status");
                        if (status == 1)//同意
                            mPermissions.add(VRTCCode.publish_inav_stream);
                        else mPermissions.remove(VRTCCode.publish_inav_stream);
                        if (mMessageListener != null) mMessageListener.onMessage(data);
                    }
                    break;
                case VRTCCode.askfor_inav_publish://邀请上麦消息
                    if (userId.equals(VhallBaseSDK.getInstance().mUserId)) {
                        mPermissions.add(VRTCCode.publish_inav_stream);
                        if (mMessageListener != null) mMessageListener.onMessage(data);
                    }
                    break;
                case VRTCCode.kick_inav_stream://请下麦消息
                    if (userId.equals(VhallBaseSDK.getInstance().mUserId)) {
                        unPublish();
                        mPermissions.remove(VRTCCode.publish_inav_stream);
                        if (mMessageListener != null) mMessageListener.onMessage(data);
                    }
                    break;
                case VRTCCode.kick_inav://请出房间消息
                    if (userId.equals(VhallBaseSDK.getInstance().mUserId)) {
                        mPermissions.remove(VRTCCode.publish_inav_stream);
                        leaveRoom();
                        if (mMessageListener != null) mMessageListener.onMessage(data);
                    }
                    break;
                case VRTCCode.force_leave_inav://强制下线消息
                    if (userId.equals(VhallBaseSDK.getInstance().mUserId)) {
                        leaveRoom();
                        if (mMessageListener != null) mMessageListener.onMessage(data);
                    }
                    break;
                case VRTCCode.user_publish_callback:
                    if (mMessageListener != null) mMessageListener.onMessage(data);
                    break;
                case VRTCCode.inav_close:
                    leaveRoom();
                    if (mMessageListener != null) mMessageListener.onMessage(data);
                    break;

            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

    }

    /**
     * 获取当前互动直播间内人员
     *
     * @param callback
     */
    public void getMembers(Callback callback) {
        if (TextUtils.isEmpty(mRoomId)) return;
        VHInteractiveApi.getRoomMember(mRoomId, mAccessToken, callback);
    }


    public boolean isKickoutStreamAvailable() {
        if (mPermissions != null && mPermissions.contains(VRTCCode.kick_inav_stream)) return true;
        else return false;
    }

    public boolean isKickoutRoomAvailable() {
        if (mPermissions != null && mPermissions.contains(VRTCCode.kick_inav)) return true;
        else return false;
    }

    public boolean isBrocastRoomAvailable() {
        if (mPermissions != null && mPermissions.contains(VRTCCode.publish_inav_another))
            return true;
        else return false;
    }

    public String getToken() {
        return null == mRoomParams ? null : mRoomParams.inavToken;
    }

    @Deprecated
    public boolean isReqPushAvailable() {
        if (mPermissions != null && mPermissions.contains(VRTCCode.apply_inav_publish)) return true;
        else return false;
    }

    public boolean isPushAvailable() {
        if (mMode.equals(VRTCCode.MODE_LIVE) && mRole.equals(VRTCCode.ROLE_AUDIENCE)) {
            return false;
        }

        return mPermissions != null && mPermissions.contains(VRTCCode.publish_inav_stream);
    }

    public boolean isInviteAvailable() {
        if (mPermissions != null && mPermissions.contains(VRTCCode.askfor_inav_publish))
            return true;
        else return false;
    }

    public boolean isCheckReqAvailable() {
        if (mPermissions != null && mPermissions.contains(VRTCCode.audit_inav_publish)) return true;
        else return false;
    }

    public void release() {
        VhallBaseSDK.getInstance().leave(this);
        VHInteractiveApi.onLeave(mRoomId, mAccessToken);

        if (null != mRtcInstance) {
            mRtcInstance.release();
            mRtcInstance = null;
        } else {
            VLog.e(TAG, "please call VHInteractiveV2#init first");
        }
        mRoomParams = null;
        mBroadcastParams = null;
    }

    /**
     * 互动直播过程中事件监听
     */
    public interface OnMessageListener {
        void onMessage(JSONObject msg);//事件处理

        void onRefreshMemberState();//更新房间内用户状态

        void onRefreshMembers(JSONObject obj);//更新房间内人数
    }

    private void trackInitEvent(String error) {
        JSONObject params = new JSONObject();
        try {
            params.put("inavId", mRoomId);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        LogReporter.getInstance().setErr(error);
        LogReporter.getInstance().onCollection(LogReporter.LOG_EVENT_INITILSS, false, params);
    }

    private void trackInitEvent() {
        JSONObject params = new JSONObject();
        try {
            params.put("inavId", mRoomId);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        LogReporter.getInstance().onCollection(LogReporter.LOG_EVENT_INITILSS, params);
    }
}
