package com.vhall.ilss;

import static com.vhall.ilss.VHInteractiveApi.TYPE_INAV;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import com.vhall.framework.VhallSDK;
import com.vhall.framework.connect.IVHService;
import com.vhall.framework.connect.VhallConnectService;
import com.vhall.logmanager.L;
import com.vhall.logmanager.LogInfo;
import com.vhall.logmanager.LogReporter;
import com.vhall.logmanager.VLog;
import com.vhall.message.ConnectServer;
import com.vhall.rtc.VRTCCode;
import com.vhall.rtc.Utils;
import com.vhall.rtc.VRTCParams;
import com.vhall.rtc.VhallRTC;
import com.vhall.vhallrtc.client.FinishCallback;
import com.vhall.vhallrtc.client.Room;
import com.vhall.vhallrtc.client.Stream;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.net.URLDecoder;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.Response;

public class VHInteractive extends VhallRTC implements IVHService {
    private static final String TAG = "VHInteractive";
    public static final String kick_inav = "kick_inav";//踢房间
    public static final String force_leave_inav = "force_leave_inav";//踢房间
    public static final String kick_inav_stream = "kick_inav_stream";//请下麦
    public static final String publish_inav_another = "publish_inav_another";//推旁路
    public static final String apply_inav_publish = "apply_inav_publish";//申请上麦
    public static final String publish_inav_stream = "publish_inav_stream";//上麦
    public static final String askfor_inav_publish = "askfor_inav_publish";//邀请推流
    public static final String audit_inav_publish = "audit_inav_publish";//审核申请
    public static final String user_publish_callback = "user_publish_callback";//上下麦，拒绝上麦
    public static final String inav_close = "inav_close";//房间关闭

    public static final String MODE_RTC = "rtc";//默认为互动房间 实时通信场景，例如互动连麦
    public static final String MODE_LIVE = "live";//无延迟直播房间 直播场景，例如无延迟直播

    public static final String ROLE_HOST = "administrator";//    可以发布和订阅互动流
    public static final String ROLE_AUDIENCE = "viewer";//    只能订阅互动流，无法发布

    //旁路分辨率 dpi
    public static final int BROADCAST_VIDEO_PROFILE_480P_0 = 0;//"{\"resolution\":[640,480],\"framerate\":2rtc5,\"bitrate\":700},"
    public static final int BROADCAST_VIDEO_PROFILE_480P_1 = 1;//"{\"resolution\":[848,480],\"framerate\":25,\"bitrate\":750},"
    public static final int BROADCAST_VIDEO_PROFILE_540P_0 = 2;//"{\"resolution\":[720,540],\"framerate\":25,\"bitrate\":950},"
    public static final int BROADCAST_VIDEO_PROFILE_540P_1 = 3;// "{\"resolution\":[960,540],\"framerate\":25,\"bitrate\":1150},"
    public static final int BROADCAST_VIDEO_PROFILE_720P_0 = 4;//"{\"resolution\":[960,720],\"framerate\":25,\"bitrate\":1400},"
    public static final int BROADCAST_VIDEO_PROFILE_720P_1 = 5;//"{\"resolution\":[1280,720],\"framerate\":25,\"bitrate\":1600},"
    public static final int BROADCAST_VIDEO_PROFILE_960P_0 = 6;//"{\"resolution\":[1280,960],\"framerate\":25,\"bitrate\":1600},"
    public static final int BROADCAST_VIDEO_PROFILE_960P_1 = 7;//"{\"resolution\":[1712,960],\"framerate\":25,\"bitrate\":1900},"
    public static final int BROADCAST_VIDEO_PROFILE_1080P_0 = 8;//"{\"resolution\":[1440,1080],\"framerate\":25,\"bitrate\":1800},"
    public static final int BROADCAST_VIDEO_PROFILE_1080P_1 = 9;//"{\"resolution\":[1920,1080],\"framerate\":25,\"bitrate\":2200}"
    /*以下新增竖版旁路，（注意当使用竖版分辨率时，仅支持部分布局模板，含：
     CANVAS_LAYOUT_PATTERN_FLOAT_2_1DR 8
     CANVAS_LAYOUT_PATTERN_FLOAT_2_1DL 9
     CANVAS_LAYOUT_EX_PATTERN_FLOAT_2_1TR 30
     CANVAS_LAYOUT_EX_PATTERN_FLOAT_2_1TL 31
     ）*/
    public static final int BROADCAST_VIDEO_PROFILE_480P_1_VERTICAL = 10;
    public static final int BROADCAST_VIDEO_PROFILE_540P_1_VERTICAL = 11;
    public static final int BROADCAST_VIDEO_PROFILE_720P_1_VERTICAL = 12;
    public static final int BROADCAST_VIDEO_PROFILE_1080P_1_VERTICAL = 13;

    //旁路布局 layout
    public static final int CANVAS_LAYOUT_PATTERN_GRID_1 = 0;//强制仅一人铺满（即使房间内有多个互动流，也只混一个）
    public static final int CANVAS_LAYOUT_PATTERN_GRID_2_H = 1;//左右两格
    public static final int CANVAS_LAYOUT_PATTERN_GRID_3_E = 2;//正品字
    public static final int CANVAS_LAYOUT_PATTERN_GRID_3_D = 3;//倒品字
    public static final int CANVAS_LAYOUT_PATTERN_GRID_4_M = 4;//2行x2列
    public static final int CANVAS_LAYOUT_PATTERN_GRID_5_D = 5;//2行，上2下3
    public static final int CANVAS_LAYOUT_PATTERN_GRID_6_E = 6;//2行x3列
    public static final int CANVAS_LAYOUT_PATTERN_GRID_9_E = 7;//3行x3列
    public static final int CANVAS_LAYOUT_PATTERN_FLOAT_2_1DR = 8;//主次悬浮，大屏铺满，小屏悬浮右下角 (小窗宽=画布宽度/5，比例为4:3)
    public static final int CANVAS_LAYOUT_PATTERN_FLOAT_2_1DL = 9;//主次悬浮，大屏铺满，小屏悬浮左下角 (小窗宽=画布宽度/5，比例为4:3)
    public static final int CANVAS_LAYOUT_PATTERN_FLOAT_3_2DL = 10;//大屏铺满，2小屏悬浮右上角 (小窗宽=画布宽度/6，比例为4:3)
    public static final int CANVAS_LAYOUT_PATTERN_FLOAT_6_5D = 11;//主次悬浮，大屏铺满，一行5个悬浮于下面 (小窗宽=画布宽度/5，比例为4:3)
    public static final int CANVAS_LAYOUT_PATTERN_FLOAT_6_5T = 12;//主次悬浮，大屏铺满，一行5个悬浮于上面 (小窗宽=画布宽度/5，比例为4:3)
    public static final int CANVAS_LAYOUT_PATTERN_TILED_5_1T4D = 13;//主次平铺，一行4个位于底部
    public static final int CANVAS_LAYOUT_PATTERN_TILED_5_1D4T = 14;//主次平铺，一行4个位于顶部
    public static final int CANVAS_LAYOUT_PATTERN_TILED_5_1L4R = 15;//主次平铺，一列4个位于右边
    public static final int CANVAS_LAYOUT_PATTERN_TILED_5_1R4L = 16;//主次平铺，一列4个位于左边
    public static final int CANVAS_LAYOUT_PATTERN_TILED_6_1T5D = 17;//主次平铺，一行5个位于底部
    public static final int CANVAS_LAYOUT_PATTERN_TILED_6_1D5T = 18;//主次平铺，一行5个位于顶部
    public static final int CANVAS_LAYOUT_PATTERN_TILED_9_1L8R = 19;//主次平铺，右边为（2列x4行=8个块）
    public static final int CANVAS_LAYOUT_PATTERN_TILED_9_1R8L = 20;//主次平铺，左边为（2列x4行=8个块）
    public static final int CANVAS_LAYOUT_PATTERN_TILED_13_1L12R = 21;//主次平铺，右边为（3列x4行=12个块）
    public static final int CANVAS_LAYOUT_PATTERN_TILED_17_1TL16GRID = 22;//主次平铺，1V16，主屏在左上角
    public static final int CANVAS_LAYOUT_PATTERN_TILED_9_1D8T = 23;//主次平铺，主屏在下，8个（2行x4列）在上
    public static final int CANVAS_LAYOUT_PATTERN_TILED_13_1TL12GRID = 24;//主次平铺，主屏在左上角，其余12个均铺于其他剩余区域
    public static final int CANVAS_LAYOUT_PATTERN_TILED_17_1TL16GRID_E = 25;//主次平铺，主屏在左上角，其余16个均铺于其他剩余区域
    public static final int CANVAS_LAYOUT_PATTERN_CUSTOM = 27;//自定义，当使用坐标布局接口时，请使用此
    public static final int CANVAS_LAYOUT_EX_PATTERN_GRID_12_E = 28;//3行4列等分布局
    public static final int CANVAS_LAYOUT_EX_PATTERN_GRID_16_E = 29;//4行4列等分布局
    public static final int CANVAS_LAYOUT_EX_PATTERN_FLOAT_2_1TR = 30;// 主次悬浮，大屏铺满，小屏悬浮右上角 (小窗宽=画布宽度/5，比例为4:3)支持竖版布局，参考PaaS需求： paas pm
    public static final int CANVAS_LAYOUT_EX_PATTERN_FLOAT_2_1TL = 31;// 主次悬浮，大屏铺满，小屏悬浮左上角 (小窗宽=画布宽度/5，比例为4:3)支持竖版布局，参考PaaS需求： paas pm

    /**
     * 混流自适应布局mode
     * 传输给MCU时会做减去1000的处理，期间用来处理部分转换逻辑，简化上层调用
     */
    public static final int CANVAS_ADAPTIVE_LAYOUT_UNDEFINED = 1000;
    public static final int CANVAS_ADAPTIVE_LAYOUT_GRID_MODE = 1001;// 均分模式
    public static final int CANVAS_ADAPTIVE_LAYOUT_TILED_MODE = 1002;// 平铺模式
    public static final int CANVAS_ADAPTIVE_LAYOUT_FLOAT_MODE = 1003;// 悬浮模式
    public static final int CANVAS_ADAPTIVE_LAYOUT_TILED_TOP_MAX16 = 1004;// 文档布局模式, 窗格位于主屏上方，最多16窗格
    public static final int CANVAS_ADAPTIVE_LAYOUT_MAX = 1005;

    private String mRoomId = "";//互动房间ID
    private String mAccessToken = "";
    private String TOKEN = "";
    private String mBroadcastId = "";//旁路房间id
    private String mPushUrl = "";//旁路推流地址
    private String mMode = MODE_RTC;//
    private String mRole = ROLE_HOST;//
    private CopyOnWriteArraySet<String> mPermissions = new CopyOnWriteArraySet<>();
    private OnMessageListener mMessageListener;
    private VhallConnectService.OnConnectStateChangedListener mOnConnectChangedListener;
    private Room.RoomDelegate roomDelegate;
    /**
     * 混流配置
     */
    private JSONObject mMixConfig = null;

    private Handler mHandler;

    /**
     * onAddInternalStream接收的流类型：云渲染
     */
    public static final String MIX_INTERNALSTREAM_TYPE_DOC_CLOUD_RENDER = "doc-cloud-render";

    public interface InitCallback {
        void onSuccess();

        void onFailure(int errorCode, String errorMsg);
    }

    /**
     * 互动直播过程中事件监听
     */
    public interface OnMessageListener {
        void onMessage(JSONObject msg);//事件处理

        void onRefreshMemberState();//更新房间内用户状态

        void onRefreshMembers(JSONObject obj);//更新房间内人数

    }

    public void setOnMessageListener(OnMessageListener listener) {
        this.mMessageListener = listener;
    }

    public void setOnConnectChangedListener(VhallConnectService.OnConnectStateChangedListener listener) {
        mOnConnectChangedListener = listener;
    }

    /**
     * 设置混流主屏
     *
     * @param streamId
     * @param callback
     */
    public void setMixLayoutMainScreen(String streamId, FinishCallback callback) {
        if (null != mRoom) {
            mRoom.setMixLayoutMainScreen(streamId, callback);

            JSONObject logConfig = new JSONObject();
            try {
                logConfig.put("stream_id", streamId);
            } catch (JSONException e) {
                e.printStackTrace();
            }
            makeBroadcastReport(LogReporter.LOG_REPORT_RTC_MAINSCREEN, logConfig);
        }
    }

    /**
     * 收到的InternalStream 是否是云渲染流
     *
     * @param config 来源：{@link Room.RoomDelegate#onDidInternalStreamAdded(Room, Stream)}
     * @return
     */
    public boolean isInternalStreamDocCloudRender(JSONObject config) {
        return null != config && config.has("type") && config.optString("type").equals(MIX_INTERNALSTREAM_TYPE_DOC_CLOUD_RENDER);
    }

    private Room.RoomDelegate delegate = new Room.RoomDelegate() {
        @Override
        public void onDidConnect(Room room, JSONObject jsonObject) {
            if (roomDelegate != null) {
                roomDelegate.onDidConnect(room, jsonObject);
            }
        }

        @Override
        public void onDidError(Room room, Room.VHRoomErrorStatus vhRoomErrorStatus, String s) {
            if (roomDelegate != null) {
                roomDelegate.onDidError(room, vhRoomErrorStatus, s);
            }
        }

        @Override
        public void onDidPublishStream(Room room, Stream stream) {
            VHInteractiveApi.onStateChanged(mRoomId, mAccessToken, String.valueOf(mLocalStream.streamId), 1, null);
            if (roomDelegate != null) {
                roomDelegate.onDidPublishStream(room, stream);
            }
        }

        @Override
        public void onDidInternalStreamAdded(Room room, Stream stream) {
            if (roomDelegate != null) {
                roomDelegate.onDidInternalStreamAdded(room, stream);
            }
        }

        @Override
        public void onDidInternalStreamRemoved(Room room, Stream stream) {
            if (roomDelegate != null) {
                roomDelegate.onDidInternalStreamRemoved(room, stream);
            }
        }

        @Override
        public void onDidInternalStreamFailed(Room room, Stream stream, JSONObject msg) {
            if (roomDelegate != null) {
                roomDelegate.onDidInternalStreamFailed(room, stream, msg);
            }
        }

        @Override
        public void onDidUnPublishStream(Room room, Stream stream) {
            if (roomDelegate != null) {
                roomDelegate.onDidUnPublishStream(room, stream);
            }
        }

        @Override
        public void onDidSubscribeStream(Room room, Stream stream) {
            if (roomDelegate != null) {
                roomDelegate.onDidSubscribeStream(room, stream);
            }
        }

        @Override
        public void onDidUnSubscribeStream(Room room, Stream stream) {
            if (roomDelegate != null) {
                roomDelegate.onDidUnSubscribeStream(room, stream);
            }
        }

        @Override
        public void onDidChangeStatus(Room room, Room.VHRoomStatus vhRoomStatus) {
            if (roomDelegate != null) {
                roomDelegate.onDidChangeStatus(room, vhRoomStatus);
            }
            if (vhRoomStatus.getValue() == Room.VHRoomStatus.VHRoomStatusDisconnected.getValue() ||
                    vhRoomStatus.getValue() == Room.VHRoomStatus.VHRoomStatusError.getValue()) {
                makeConnectReport(LogReporter.LOG_REPORT_RTC_DISCONNECT, "Room.VHRoomStatus=" + vhRoomStatus.getValue());
            }
        }

        @Override
        public void onDidAddStream(Room room, Stream stream) {
            if (roomDelegate != null) {
                roomDelegate.onDidAddStream(room, stream);
            }
        }

        @Override
        public void onDidRemoveStream(Room room, Stream stream) {
            if (roomDelegate != null) {
                roomDelegate.onDidRemoveStream(room, stream);
            }
        }

        @Override
        public void onDidUpdateOfStream(Stream stream, JSONObject jsonObject) {
            if (roomDelegate != null) {
                roomDelegate.onDidUpdateOfStream(stream, jsonObject);
            }
        }

        @Override
        public void onReconnect(int i, int i1) {
            if (roomDelegate != null) {
                roomDelegate.onReconnect(i, i1);
            }
        }

        @Override
        public void onStreamMixed(JSONObject jsonObject) {
            if (roomDelegate != null) {
                roomDelegate.onStreamMixed(jsonObject);
            }
        }
    };

    public VHInteractive(Context context, Room.RoomDelegate listener) {
        this(context, listener, true, true);
    }

    public VHInteractive(Context context, Room.RoomDelegate listener, boolean softEchoCanceller) {
        this(context, listener, softEchoCanceller, true);
    }

    /**
     * @param context
     * @param listener
     * @param softEchoCanceller         软硬件回声消除设置 true 软件回声消除  false 硬件回声消除
     * @param enableCpuoverusedetection 是否启动CPU/网络超载检测以自动降低分辨率/码率
     */
    public VHInteractive(Context context, Room.RoomDelegate listener, boolean softEchoCanceller, boolean enableCpuoverusedetection) {
        super(context, softEchoCanceller, enableCpuoverusedetection);
        this.roomDelegate = listener;
        setListener(delegate);
        mHandler = new Handler(Looper.getMainLooper());
    }

    /**
     * @param roomid
     * @param accessToken
     * @param callback
     */
    public void init(final String roomid, final String accessToken, final InitCallback callback) {
        init(roomid, accessToken, null, callback);
    }

    /**
     * @param roomid
     * @param broadcastid 旁路直播id
     * @param accessToken
     * @param callback
     */
    public void init(final String roomid, final String accessToken,final String broadcastid, final InitCallback callback) {
        init(roomid, accessToken, broadcastid,MODE_RTC,ROLE_HOST, callback);
    }
    /**
     * @param roomid
     * @param broadcastid 旁路直播id
     * @param accessToken
     * @param mode MODE_RTC 实时通信场景，例如互动连麦 MODE_LIVE 无延迟直播房间 直播场景，例如无延迟直播
     *             应用场景模式，选填，可选值参考 VHInteractiveRoomMode。支持版本：2.4.0及以上
     * @param role ROLE_HOST   可以发布和订阅互动流  ROLE_AUDIENCE 只能订阅互动流，无法发布
     *            用户角色，选填，可选值参考下文VHInteractiveRoomRole。当mode为rtc模式时，不需要配置role。支持版本：2.4.0及以上
     * @param callback
     */
    public void init(final String roomid, final String accessToken, final String broadcastid, final String mode, final String role, final InitCallback callback) {
        mRoomId = roomid;
        mMode = mode;//
        mRole = role;//
        if (TextUtils.isEmpty(mMode)) {
            mMode = MODE_RTC;
        }
        if (!mMode.equals(MODE_LIVE)) {
            mRole = ROLE_HOST;
        }
        mAccessToken = accessToken;
        getRoomInfo(roomid, accessToken, broadcastid, mode, role, callback);
    }

    private void getRoomInfo(final String roomid, final String accessToken,final String broadcastid,final  String mode,final String role,final InitCallback callback) {
        VHInteractiveApi.getRoomInfo(roomid, accessToken, broadcastid, mode, role, new Callback() {
            @Override
            public void onFailure(final Call call, final IOException e) {
                L.e(TAG, e.getMessage());
                trackInitEvent(LogReporter.LOG_ERROR_NET);
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        if (callback != null)
                            callback.onFailure(-1, "error network,please try later！");
                    }
                });
            }

            @Override
            public void onResponse(Call call, Response response) throws IOException {
                final String content = response.body().string();
                mHandler.post(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            JSONObject result = new JSONObject(content);
                            String msg = result.optString("msg");
                            int code = result.optInt("code");
                            if (code == 200) {
                                mBroadcastId = broadcastid;
                                JSONObject data = result.optJSONObject("data");
                                String room_type = data.optString("type");
                                if (!room_type.equals(VRTCParams.RTCTYPE_VRTC)) {
                                    if (callback != null)
                                        callback.onFailure(-1, "room type error");
                                    return;
                                }
                                TOKEN = data.getString("inav_token");
                                mPushUrl = data.getString("pushUrl");
                                JSONObject logObj = data.optJSONObject("log_info");
                                JSONArray permissions = data.optJSONArray("permission");
                                if (permissions != null && permissions.length() > 0) {
                                    for (int i = 0; i < permissions.length(); i++) {
                                        mPermissions.add((String) permissions.get(i));
                                    }
                                }
                                //设置日志地址
                                LogInfo.getInstance().roomId = mRoomId;
                                if (logObj != null) {
                                    LogInfo.getInstance().initBaseData(logObj);
                                }
                                setDataReport(new JSONObject(LogInfo.getInstance().toString()));
//                                Room.setLogReportData(new JSONObject(LogInfo.getInstance().toString()));
                                /* verify token */
                                if (!Utils.verifyToken(TOKEN) && null != callback) {
//                                    callback.onFailure(-1, "token error: " + TOKEN);
                                    trackInitEvent();
//                                    return;
                                }
                                if (callback != null)
                                    callback.onSuccess();
                                trackInitEvent();
                            } else {
                                if (callback != null)
                                    callback.onFailure(code, msg);
                                trackInitEvent(code + ":" + msg);
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                            if (callback != null)
                                callback.onFailure(-1, "Request data exception");
                            trackInitEvent(LogReporter.LOG_ERROR_EXCEPTION);
                        }
                    }
                });
            }
        });
    }

    public void addExtraLogParam(String s) {
        LogInfo.getInstance().setExtraData(s);
    }

    /**
     * 进入房间
     *
     * @param attribute 自定义信息，其他用户订阅时，可以从{@link Stream#getUserAttributes()}中获取
     */
    public void enterRoom(String attribute) {
        if (!TextUtils.isEmpty(TOKEN)) {
            super.enterRoom(TOKEN, attribute);
            VhallSDK.getInstance().join(this);

            makeConnectReport(LogReporter.LOG_REPORT_RTC_CONNECT, null);
        } else {
            VLog.e(TAG, "token is null, please init first");
        }
    }

    @Override
    public void publish() {
        if (isPushAvailable()) {
            super.publish();
        } else {
            Log.e(TAG, "Do not have publish permission.");
        }
    }

    @Override
    public void unpublish() {
        super.unpublish();
        if (mLocalStream != null) {
            VHInteractiveApi.onStateChanged(mRoomId, mAccessToken, String.valueOf(mLocalStream.streamId), 2, null);
        }
    }

    @Override
    public void leaveRoom() {
        super.leaveRoom();
        VhallSDK.getInstance().leave(this);
        VHInteractiveApi.onLeave(mRoomId, mAccessToken);

        makeConnectReport(LogReporter.LOG_REPORT_RTC_DISCONNECT, "call leaveRoom");
    }

    public void requestPublish(Callback callback) {
        if (isReqPushAvailable())
            VHInteractiveApi.reqPush(mRoomId, mAccessToken, callback);
    }

    /**
     * 审核上麦
     *
     * @param userid 申请人ID
     * @param type   1 同意(默认) 2 不同意
     */
    public void checkPublishRequest(String userid, int type, Callback callback) {
        if (isCheckReqAvailable())
            VHInteractiveApi.checkPushReq(mRoomId, mAccessToken, userid, type, callback);
    }

    /**
     * 邀请上麦
     *
     * @param userid
     */
    public void invitePublish(String userid, Callback callback) {
        if (isInviteAvailable())
            VHInteractiveApi.invitePush(mRoomId, mAccessToken, userid, callback);
    }

    /**
     * 拒绝上麦
     * 1 上麦(默认) 2 下麦 3 拒绝上麦
     */
    public void refusePublish() {
        if (mLocalStream != null)
            VHInteractiveApi.onStateChanged(mRoomId, mAccessToken, String.valueOf(mLocalStream.streamId), 3, null);
    }

    /**
     * 踢流，请下麦
     *
     * @param userid
     */
    public void kickoutStream(String userid, Callback callback) {
        if (isKickoutStreamAvailable())
            VHInteractiveApi.kickoutPush(mRoomId, mAccessToken, userid, 1, callback);
    }

    /**
     * 踢出房间
     *
     * @param userid
     */
    public void kickoutRoom(String userid, Callback callback) {
        if (isKickoutRoomAvailable())
            VHInteractiveApi.kickoutRoom(mRoomId, mAccessToken, userid, 1, callback);
    }

    /**
     * 强制离开房间，一般用于自己异常退出互动房间后，再进入提示已在房间中使用，不会拉入黑名单
     *
     * @param userid
     */
    public void forceLeaveRoom(String userid, Callback callback) {
        if (userid == null)
            userid = VhallSDK.getInstance().getmUserId();
        VHInteractiveApi.forceLeaveRoom(mRoomId, mAccessToken, userid, callback);
    }

    /**
     * @param type     1发起，2结束发起
     * @param callback
     */
    public void broadcastRoom(int type, Callback callback) {
        broadcastRoom(type, BROADCAST_VIDEO_PROFILE_720P_1, CANVAS_LAYOUT_PATTERN_TILED_5_1T4D, callback);
    }

    public void broadcastRoom(int type, int layout, Callback callback) {
        broadcastRoom(type, BROADCAST_VIDEO_PROFILE_720P_1, layout, callback);
    }

    public void broadcastRoom(int type, int dpi, int layout, Callback callback) {
        broadcastRoom(type, dpi, layout, false, callback);
    }

    /**
     * @param type     1发起，2结束发起
     * @param dpi      BROADCAST_VIDEO_PROFILE_720P_1
     * @param layout   CANVAS_LAYOUT_PATTERN_TILED_5_1T4D
     *                 "precast_pic_exist": false, //选填，是否有占坑图标，默认为否。 用于标识填充布局模板种的多余块，图片为统一设计小人头图标，暂不支持自定义图片。 该配置项变化时，会立即生效
     * @param callback
     */
    public void broadcastRoom(int type, int dpi, int layout, boolean precast_pic_exist, Callback callback) {
        JSONObject config = new JSONObject();
        try {
            config.put("profile", Room.getProfile(dpi));
            config.put("layoutMode", layout);
            config.put("precast_pic_exist", precast_pic_exist);
            broadcastRoom(type, config, callback);
        } catch (JSONException e) {
            e.printStackTrace();
            if (callback != null)
                callback.onFailure(null, new IOException(e.getMessage()));
        }
    }

    /**
     * @param type     1发起，2结束发起
     * @param config
     * @param callback configRoomBroadCast 配置旁路混流参数 流程，1、通知微吼云旁路状态改变、2、调用sdk接口修改旁路、 3、失败后通知微吼云恢复旁路状态
     *                 http://wiki.vhallops.com/pages/viewpage.action?title=Room&spaceKey=media#Room-%E9%85%8D%E7%BD%AE%E6%97%81%E8%B7%AF%E5%8F%82%E6%95%B0
     *                 // 基本参数，满足大部分旁路配置需求
     *                 config.put("profile",Room.getProfile(BROADCAST_VIDEO_PROFILE_720P_1)); //{"resolution":[640,480],"framerate":25,"bitrate":700}
     *                 config.put("layoutMode",CANVAS_LAYOUT_PATTERN_TILED_5_1T4D);
     *                 设置成功后会自动推旁路
     */
    public void broadcastRoom(int type, JSONObject config, Callback callback) {

        final JSONObject tempConfig;
        if (null != config) {
            tempConfig = config;
        } else {
            tempConfig = new JSONObject();
        }

        if (tempConfig.has("layoutMode")) {
            try {
                int tempLayoutMode = tempConfig.getInt("layoutMode");
                if (tempLayoutMode >= 1000) {
                    tempConfig.remove("layoutMode");
                    tempConfig.put("adaptiveLayoutMode", tempLayoutMode - 1000);
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        if (TextUtils.isEmpty(mBroadcastId) || TextUtils.isEmpty(mPushUrl)) {
            if (callback != null)
                callback.onFailure(null, new IOException("未设置旁路信息"));
            return;
        }

        if (!isBrocastRoomAvailable()) {
            if (callback != null)
                callback.onFailure(null, new IOException("请开通设置旁路权限"));
            return;
        }

        if (type == 1) { //开启旁路
            try {
                tempConfig.put("publishUrl", mPushUrl);

                if (!tempConfig.has("nickName")) {
                    JSONObject nickName = new JSONObject();
                    nickName.put("display", true);
                    tempConfig.put("nickName", nickName);
                }

                boundPreConfigColor(tempConfig);
                configRoomBroadCast(tempConfig, null, new FinishCallback() {
                    @Override
                    public void onFinish(int code, String message) {
                        mMixConfig = tempConfig;
                        if (code != 200) {
                            if (callback != null)
                                callback.onFailure(null, new IOException(message));
                        } else {
                            if (callback != null) {
                                try {
                                    callback.onResponse(null, null);
                                } catch (IOException e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    }
                });

                makeBroadcastReport(LogReporter.LOG_REPORT_RTC_BROADCAST, tempConfig);
            } catch (JSONException e) {
                e.printStackTrace();
                if (callback != null)
                    callback.onFailure(null, new IOException(e.getMessage()));
            }
        } else if (type == 2) { //关闭旁路
            stopRoomBroadCast(null, new FinishCallback() {
                @Override
                public void onFinish(int code, String message) {
                    if (code != 200) {
                        if (callback != null)
                            callback.onFailure(null, new IOException(message));
                    } else {
                        if (callback != null) {
                            try {
                                callback.onResponse(null, null);
                            } catch (IOException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                }
            });
        }
    }

    private void boundPreConfigColor(JSONObject baseConfig) {
        if (null != _bgColorJSON) {
            try {
                baseConfig.put("backgroundColor", _bgColorJSON.opt("backgroundColor"));
                _bgColorJSON = null;
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        if (null != _boardColorJSON) {
            try {
                baseConfig.put("border", _boardColorJSON.opt("border"));
                _boardColorJSON = null;
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * @param layout   CANVAS_LAYOUT_PATTERN_TILED_5_1T4D
     * @param callback * 配置旁路混流参数
     *                 http://wiki.vhallops.com/pages/viewpage.action?title=Room&spaceKey=media#Room-%E9%85%8D%E7%BD%AE%E6%97%81%E8%B7%AF%E5%8F%82%E6%95%B0
     *                 // 基本参数，满足大部分旁路配置需求
     *                 config.put("profile",Room.getProfile(Room.BROADCAST_VIDEO_PROFILE_720P_1));
     *                 config.put("publishUrl",SettingModel.broadCastUrl);
     *                 config.put("layoutMode",SettingModel.broadCastLayout);
     *                 设置成功后会自动推旁路
     */
    public void broadcastLayout(int layout, Callback callback) {
        if (!isBrocastRoomAvailable()) {
            if (callback != null)
                callback.onFailure(null, new IOException("请开通设置旁路权限"));
            return;
        }

        this.setMixLayoutMode(layout, null, new FinishCallback() {
            @Override
            public void onFinish(int code, String message) {
                if (code != 200) {
                    if (callback != null)
                        callback.onFailure(null, new IOException(message));
                } else {
                    if (callback != null) {
                        try {
                            callback.onResponse(null, null);
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        });
    }

    @Override
    public void setMixLayoutMode(int layoutMode, String mode, FinishCallback finish) {
        super.setMixLayoutMode(layoutMode, mode, finish);
        //refresh local config cache
        if (null != mMixConfig) {
            if (layoutMode >= 1000) {
                try {
                    mMixConfig.remove("layoutMode");
                    mMixConfig.put("adaptiveLayoutMode", layoutMode - 1000);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            } else {
                try {
                    mMixConfig.put("layoutMode", layoutMode);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    @Override
    public void subscribeStream(Stream stream) {
        super.subscribeStream(stream);

        JSONObject logConfig = new JSONObject();
        try {
            logConfig.put("stream_id", stream.streamId);
            logConfig.put("attr", stream.getAttributes());
        } catch (JSONException e) {
            e.printStackTrace();
        }
        makeBroadcastReport(LogReporter.LOG_REPORT_RTC_SUBSCRIBE, logConfig);
    }

    @Override
    public void setRoomBroadCastBackgroundImage(String url, int cropType, FinishCallback callback) {
        super.setRoomBroadCastBackgroundImage(url, cropType, callback);

        JSONObject logConfig = new JSONObject();
        try {
            logConfig.put("img", url);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        makeBroadcastReport(LogReporter.LOG_REPORT_RTC_BROADCAST_BG_IMAGE, logConfig);
    }

    /**
     * 设置旁路背景色
     *
     * @param hexColor 支持格式： 0x000000、#000000、000000
     * @param callback
     */
    public void setRoomBroadCastBackgroundColor(String hexColor, FinishCallback callback) {
        String bgColor = "0x333338";//default color
        if (!TextUtils.isEmpty(hexColor)) {
            bgColor = hexColor;
            if (hexColor.startsWith("#")) {
                bgColor = hexColor.replace("#", "0x");
            } else if (!hexColor.startsWith("0x")) {
                bgColor = "0x".concat(hexColor);
            }
        }
        if (null != this.mRoom) {
            try {
                if (null != mMixConfig) {
                    mMixConfig.put("backgroundColor", bgColor);
                    this.mRoom.configRoomBroadCast(mMixConfig, null, callback);
                } else {
                    _bgColorJSON = new JSONObject();
                    _bgColorJSON.put("backgroundColor", bgColor);
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        } else {
            Log.e(TAG, "Room is null");
        }
    }

    private JSONObject _bgColorJSON = null;
    private JSONObject _boardColorJSON = null;

    public void resetRoomBroadCastBackgroundColor(FinishCallback callback) {
        setRoomBroadCastBackgroundColor("0x333338", callback);
    }

    /**
     * 设置旁路背景色
     *
     * @param hexColor 支持格式： 0x000000、#000000、000000
     * @param width default = 4px, max = 8px
     * @param callback
     */
    private void setRoomBroadCastBorderColor(boolean enable, int width, String hexColor, boolean transparent, FinishCallback callback) {
        if (null != this.mRoom) {
            if (!enable) {
                try {
                    if (null != mMixConfig) {
                        JSONObject border = new JSONObject();
                        border.put("exist", false);
                        mMixConfig.put("border", border);
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            } else {
                String bgColor = "0x555657";//default color
                if (!TextUtils.isEmpty(hexColor)) {
                    bgColor = hexColor;
                    if (hexColor.startsWith("#")) {
                        bgColor = hexColor.replace("#", "0x");
                    } else if (!hexColor.startsWith("0x")) {
                        bgColor = "0x".concat(hexColor);
                    }
                }
                if (width > 8) {
                    width = 8;
                } else if (width < 1) {
                    width = 0;
                }
                try {
                    JSONObject border = new JSONObject();
                    border.put("exist", true);
                    border.put("width", width);
                    border.put("color", bgColor);
                    border.put("transparency", transparent ? 100 : 0);

                    if (null != mMixConfig) {
                        mMixConfig.put("border", border);
                    } else {
                        _boardColorJSON = new JSONObject();
                        _boardColorJSON.put("border", border);
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            if (null != mMixConfig) {
                this.mRoom.configRoomBroadCast(mMixConfig, null, callback);
            }
        } else {
            Log.e(TAG, "Room is null");
        }
    }

    public void setRoomBroadCastBorderColor(int width, String hexColor, FinishCallback callback) {
        setRoomBroadCastBorderColor(width, hexColor, false, callback);
    }

    /**
     * @param width       [0, 8]
     * @param hexColor    支持格式： 0x000000、#000000、000000
     * @param transparent 边框是否透明【当前仅支持不透明与全透明】
     */
    public void setRoomBroadCastBorderColor(int width, String hexColor, boolean transparent, FinishCallback callback) {
        setRoomBroadCastBorderColor(true, width, hexColor, transparent, callback);
    }

    public void resetRoomBroadCastBorderColor(FinishCallback callback) {
        setRoomBroadCastBorderColor(false, 0, null, false, callback);
    }

    /**
     * 获取权限列表
     *
     * @return
     */
    public Set<String> getPermissions() {
        return mPermissions;
    }

    @Override
    public String getChannelId() {
        return mRoomId;
    }

    @Override
    public String getAccessToken() {
        return mAccessToken;
    }

    @Override
    public void onConnectStateChanged(ConnectServer.State state, int serverType) {
        if (mOnConnectChangedListener != null)
            mOnConnectChangedListener.onStateChanged(state, serverType);
    }

    @Override
    public void onMessage(String msg) {
        if (TextUtils.isEmpty(msg))
            return;
        Log.i(TAG, "onMsg:" + msg);
        try {
            JSONObject result = new JSONObject(msg);
            String event = result.optString("event");
            if (mMessageListener != null) {
                /**
                 * 收到消息通知用户刷新用户状态
                 * 收到消息立刻通知更新用户状态
                 * 1.房间内用户状态为：
                 *  1 推流中 2 观看中 3 受邀中 4 申请中 5 被踢出
                 */
                mMessageListener.onRefreshMemberState();
            }
            if (!event.equals(TYPE_INAV)) {
                /**
                 * 非互动消息（上下线消息）
                 * 因为最外层加了消息解析，故更新逻辑
                 */
                JSONObject text;
                if (!TextUtils.isEmpty(result.optString("text"))) {
                    text = new JSONObject(URLDecoder.decode(result.optString("text")));
                } else {
                    text = result;
                }
                if (mMessageListener != null && text != null)
                    mMessageListener.onRefreshMembers(text);
                return;
            }
            JSONObject data = result.getJSONObject("data");
            String dataEvent = data.getString("inav_event");
            String userId = data.optString("third_party_user_id");
            switch (dataEvent) {
                case VHInteractive.apply_inav_publish://申请上麦消息
                    if (userId.equals(VhallSDK.getInstance().mUserId))//自己不审核自己上麦申请
                        break;
                    //判断当前用户是否有审核权限
                    if (isCheckReqAvailable() && mMessageListener != null)
                        mMessageListener.onMessage(data);
                    break;
                case VHInteractive.audit_inav_publish://上麦审核结果
                    if (userId.equals(VhallSDK.getInstance().mUserId)) {
                        int status = data.getInt("status");
                        if (status == 1)//同意
                            mPermissions.add(publish_inav_stream);
                        else
                            mPermissions.remove(publish_inav_stream);
                        if (mMessageListener != null)
                            mMessageListener.onMessage(data);
                    }
                    break;
                case VHInteractive.askfor_inav_publish://邀请上麦消息
                    if (userId.equals(VhallSDK.getInstance().mUserId)) {
                        mPermissions.add(publish_inav_stream);
                        if (mMessageListener != null)
                            mMessageListener.onMessage(data);
                    }
                    break;
                case VHInteractive.kick_inav_stream://请下麦消息
                    if (userId.equals(VhallSDK.getInstance().mUserId)) {
                        unpublish();
                        mPermissions.remove(publish_inav_stream);
                        if (mMessageListener != null)
                            mMessageListener.onMessage(data);
                    }
                    break;
                case VHInteractive.kick_inav://请出房间消息
                    if (userId.equals(VhallSDK.getInstance().mUserId)) {
                        mPermissions.remove(publish_inav_stream);
                        leaveRoom();
                        if (mMessageListener != null)
                            mMessageListener.onMessage(data);
                    }
                    break;
                case VHInteractive.force_leave_inav://强制下线消息
                    if (userId.equals(VhallSDK.getInstance().mUserId)) {
                        leaveRoom();
                        if (mMessageListener != null)
                            mMessageListener.onMessage(data);
                    }
                    break;
                case VHInteractive.user_publish_callback:
                    if (mMessageListener != null)
                        mMessageListener.onMessage(data);
                    break;
                case VHInteractive.inav_close:
                    leaveRoom();
                    if (mMessageListener != null)
                        mMessageListener.onMessage(data);
                    break;

            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

    }

    /**
     * 获取当前互动直播间内人员
     *
     * @param callback
     */
    public void getMembers(Callback callback) {
        if (TextUtils.isEmpty(mRoomId))
            return;
        VHInteractiveApi.getRoomMember(mRoomId, mAccessToken, callback);
    }


    public boolean isKickoutStreamAvailable() {
        if (mPermissions != null && mPermissions.contains(kick_inav_stream))
            return true;
        else
            return false;
    }

    public boolean isKickoutRoomAvailable() {
        if (mPermissions != null && mPermissions.contains(kick_inav))
            return true;
        else
            return false;
    }

    public boolean isBrocastRoomAvailable() {
        if (mPermissions != null && mPermissions.contains(publish_inav_another))
            return true;
        else
            return false;
    }

    public String getToken() {
        return TOKEN;
    }

    public boolean isReqPushAvailable() {
        if (mPermissions != null && mPermissions.contains(apply_inav_publish))
            return true;
        else
            return false;
    }

    public boolean isPushAvailable() {
        if (mMode.equals(MODE_LIVE) && mRole.equals(ROLE_AUDIENCE)) {
            return false;
        }

        if (mPermissions != null && mPermissions.contains(publish_inav_stream))
            return true;
        else
            return false;
    }

    public boolean isInviteAvailable() {
        if (mPermissions != null && mPermissions.contains(askfor_inav_publish))
            return true;
        else
            return false;
    }

    public boolean isCheckReqAvailable() {
        if (mPermissions != null && mPermissions.contains(audit_inav_publish))
            return true;
        else
            return false;
    }

    @Override
    public void release() {
        super.release();
        VhallSDK.getInstance().leave(this);
        VHInteractiveApi.onLeave(mRoomId, mAccessToken);
    }


    private void trackInitEvent(String error) {
        JSONObject params = new JSONObject();
        try {
            params.put("inavId", mRoomId);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        LogReporter.getInstance().setErr(error);
        LogReporter.getInstance().onCollection(LogReporter.LOG_EVENT_INITILSS, false, params);
    }

    private void trackInitEvent() {
        JSONObject params = new JSONObject();
        try {
            params.put("inavId", mRoomId);
        } catch (JSONException e) {
            e.printStackTrace();
        }
        LogReporter.getInstance().onCollection(LogReporter.LOG_EVENT_INITILSS, params);
    }

    private void makeConnectReport(String event, String reason) {
        JSONObject pushConfig = new JSONObject();
        try {
            pushConfig.put("inav_id", mRoomId);
            pushConfig.put("user_id", VhallSDK.getInstance().mUserId);
            pushConfig.put("third_user_id", VhallSDK.getInstance().mUserId);
            if (!TextUtils.isEmpty(reason)) {
                pushConfig.put("reason", reason);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
        LogReporter.getInstance().onCollectionWithBData(event, pushConfig);
    }

    private void makeBroadcastReport(String event, JSONObject param) {
        LogReporter.getInstance().onCollectionWithBData(event, param);
    }
}
