package com.vhall.business;

import android.content.Context;
import android.graphics.Bitmap;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;
import android.view.SurfaceView;
import android.view.View;

import com.vhall.business.data.RequestCallback;
import com.vhall.business.data.WebinarInfo;
import com.vhall.business.data.WebinarInfoRemoteDataSource;
import com.vhall.business.data.source.UserInfoRepository;
import com.vhall.business.data.source.WebinarInfoRepository;
import com.vhall.business.data.source.local.UserInfoLocalDataSource;
import com.vhall.business.data.source.remote.UserInfoRemoteDataSource;
import com.vhall.player.Constants;
import com.vhall.player.VHPlayerListener;
import com.vhall.player.stream.play.impl.VHVideoPlayerView;
import com.vhall.player.vod.VodPlayerView;

import java.util.List;

import static com.vhall.business.ErrorCode.ERROR_INIT;
import static com.vhall.business.common.Constants.TYPE_FLASH;
import static com.vhall.business.common.Constants.TYPE_H5;
import static com.vhall.business.common.Constants.TYPE_H5_NEW;

/**
 * Created by zwp on 2018/12/27
 */
public class WatchPlayback extends Playback {
    private static final String TAG = "WatchPlayback";
    private Playback mPlayer;


    //观看回放过程中的文档白板事件监听
    public interface DocumentEventCallback {
        void onEvent(String key, List<MessageServer.MsgInfo> msgInfos);

        void onEvent(MessageServer.MsgInfo msgInfo);

        void onEvent(String event, String type, View documentView);

        void onError(int errorCode, int innerError, String errorMsg);
    }

    private String type;

    private WatchPlayback(Builder builder) {
        this.context = builder.context;
        if (builder.surfaceView != null) {
            this.surfaceView = builder.surfaceView;
        } else if (builder.vodPlayerView != null) {
            this.vodPlayerView = builder.vodPlayerView;
        }
        this.listener = builder.listener;
        this.docEventCallback = builder.docCallback;
        this.chatCallback = builder.chatCallback;
        this.messageCallback = builder.messageCallback;
        this.mDelivery = new Handler(Looper.getMainLooper());
    }

    @Override
    public void setWebinarInfo(WebinarInfo webinarInfo) {
        super.setWebinarInfo(webinarInfo);
        qualities = webinarInfo.qualities;
        waterMarkUrl = webinarInfo.watermark.imgUrl;
        waterMarkGravity = webinarInfo.watermark.imgPosition;
        waterMarkAlpha = webinarInfo.watermark.imgAlpha;
        Playback.Builder builder = new Playback.Builder();
        builder.callback(this.listener)
                .context(this.context)
                .chatCallback(this.chatCallback)
                .messageCallback(this.messageCallback)
                .docCallback(this.docEventCallback);
        if (this.surfaceView != null) {
            builder.surfaceView(this.surfaceView);
        } else if (this.vodPlayerView != null) {
            builder.vodPlayView(this.vodPlayerView);
        }
        if (webinarInfo.is_new_version == 3) {
            type = TYPE_H5_NEW;
        } else {
            type = TextUtils.isEmpty(webinarInfo.vss_token) ? TYPE_FLASH : TYPE_H5;
        }
        mPlayer = builder.build(type);
        mPlayer.setWebinarInfo(webinarInfo);
    }

    public static class Builder {
        Context context;
        VHPlayerListener listener;
        DocumentEventCallback docCallback;
        SurfaceView surfaceView;
        VodPlayerView vodPlayerView;
        ChatServer.Callback chatCallback;
        MessageServer.Callback messageCallback;

        public Builder chatCallback(ChatServer.Callback chatCallback) {
            this.chatCallback = chatCallback;
            return this;
        }
        public Builder messageCallback(MessageServer.Callback messageCallback) {
            this.messageCallback = messageCallback;
            return this;
        }

        public Builder context(Context context) {
            this.context = context;
            return this;
        }

        public Builder surfaceView(SurfaceView surfaceView) {
            this.surfaceView = surfaceView;
            return this;
        }

        public Builder vodPlayView(VodPlayerView vodPlayerView) {
            this.vodPlayerView = vodPlayerView;
            return this;
        }

        public Builder callback(VHPlayerListener listener) {
            this.listener = listener;
            return this;
        }

        @Deprecated
        public Builder docCallback(DocumentEventCallback docCallback) {
            this.docCallback = docCallback;
            return this;
        }

        public WatchPlayback build() {
            if (surfaceView == null && vodPlayerView == null) {
                throw new IllegalArgumentException("containerLayout can not be null");
            }
            if (context == null) {
                if (surfaceView != null) {
                    context = surfaceView.getContext();
                } else {
                    context = vodPlayerView.getContext();
                }
            }
            return new WatchPlayback(this);
        }
    }

    @Override
    public void setScaleType(int scaleType) {
        super.setScaleType(scaleType);
        if (mPlayer != null) {
            mPlayer.setScaleType(scaleType);
        }
    }

    @Override
    public boolean isPlaying() {
        if (mPlayer != null) {
            return mPlayer.isPlaying();
        }
        return false;
    }

    @Override
    public void mute() {
        if (null != mPlayer) {
            mPlayer.mute();
        }
    }

    @Override
    public void unMute() {
        if (null != mPlayer) {
            mPlayer.unMute();
        }
    }

    @Override
    public void setDefinition(String dpi) {
        mCurrentDpi = dpi;
        if (mPlayer == null) {
            start();
        } else {
            mPlayer.setDefinition(mCurrentDpi);
        }
    }

    @Override
    public void start() {
        if (!isAvaliable()) {
            if (listener != null) {
                listener.onError(ERROR_NOT_INIT, 0, VhallSDK.mContext.getString(R.string.error_video_msg_init));
            }
            return;
        }
        if (webinarInfo.status != WebinarInfo.VIDEO && webinarInfo.status != WebinarInfo.MEDIA) {
            if (listener != null)
                listener.onError(ERROR_STATE, 0, String.format(context.getString(R.string.playing_status), webinarInfo.getStatusStr()));
            return;
        }

        if (mPlayer != null) {
            mPlayer.start();
        }
    }

    @Override
    public void onPause() {
        if (mPlayer != null) {
            mPlayer.onPause();
        }
    }

    @Override
    public void onResume() {
        if (mPlayer != null) {
            mPlayer.onResume();
        }
    }

    @Override
    public void stop() {
        if (mPlayer != null) {
            mPlayer.stop();
        }
    }

    @Override
    public void destroy() {
        if (mPlayer != null) {
            mPlayer.destroy();
        }
        flag = false;
    }

    @Override
    public float setSpeed(float speed) {
        if (mPlayer != null) {
            return mPlayer.setSpeed(speed);
        }
        return 0;
    }

    @Override
    public boolean setVideoBackgroundColor(int color) {
        if (mPlayer != null)
            return mPlayer.setVideoBackgroundColor(color);
        return false;
    }

    @Override
    public boolean setVideoBackgroundImage(Bitmap bitmap) {
        if (bitmap == null) {
            return false;
        }
        if (mPlayer != null)
            return mPlayer.setVideoBackgroundImage(bitmap);
        return false;
    }
    @Override
    public boolean takeVideoScreenshot(VHVideoPlayerView.ScreenShotCallback callback) {
        if (mPlayer != null)
            return mPlayer.takeVideoScreenshot(callback);
        return false;
    }
    @Override
    public void startPlay(String url) {
        releasePlayer();
        if (mPlayer != null) {
            mPlayer.startPlay(url);
        }
    }

    @Override
    public void releasePlayer() {
        try {
            if (mPlayer != null) {
                mPlayer.releasePlayer();
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Override
    public long getCurrentPosition() {
        if (mPlayer != null) {
            return mPlayer.getCurrentPosition();
        }
        return 0;
    }

    @Override
    public long getDuration() {
        if (mPlayer != null) {
            return mPlayer.getDuration();
        }
        return 0;
    }

    @Override
    public void seekTo(long position) {
        if (mPlayer != null) {
            mPlayer.seekTo(position);
        }
    }

    @Override
    public Constants.State getPlayerState() {
        if (mPlayer != null) {
            return mPlayer.getPlayerState();
        }
        return Constants.State.NONE;
    }

    @Override
    public void initWH(int width, int height) {

    }

    @Override
    public String getOriginalUrl() {
        if (webinarInfo.getCast_screen() == 1 && mPlayer != null) {
            return mPlayer.getOriginalUrl();
        }
        return null;
    }


    @Override
    public void sendComment(String content, final RequestCallback callback) {
        if (!VhallSDK.isLogin()) {
            VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_PARAM, context.getString(R.string.ERROR_MSG_IS_LOGIN));
            return;
        }
        if (!isAvaliable() || TextUtils.isEmpty(webinarInfo.webinar_id)) {
            VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_INIT, context.getString(R.string.error_video_info_init));
            return;
        }
        if (webinarInfo.chatforbid) {
            callback.onError(ErrorCode.ERROR_BANNED, context.getString(R.string.error_chat_forbid_open));
            return;
        }
        if (webinarInfo.filters != null) { // 添加信息过滤
            for (int i = 0; i < webinarInfo.filters.size(); i++) {
                if (content.contains(webinarInfo.filters.get(i))) {
                    VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_KEY_FILTERS, VhallSDK.mContext.getString(R.string.error_chat_forbid));
                    return;
                }
            }
        }

        /**
         *  flash评论 h5聊天
         */

        if (mPlayer != null) {
            mPlayer.sendComment(content, callback);
        }
    }

    @Override
    public void requestCommentHistory(String webinar_id, int limit, int pos, final ChatServer.ChatRecordCallback callback) {
        if (checkCommentParameter(limit, pos, callback)) {
            if (mPlayer != null) {
                mPlayer.requestCommentHistory(webinar_id, limit, pos, callback);
            }
        }
    }

    /**
     * 获取当前房间聊天列表
     *
     * @param page        获取条目节点，默认为1
     * @param limit       获取条目数量，最大100
     * @param msgId      获取条目数量，聊天记录 锚点消息id,此参数存在时anchor_path 参数必须存在
     * @param anchor_path 锚点方向，up 向上查找，down 向下查找,此参数存在时 msg_id 参数必须存在
     * @param is_role     0：不筛选主办方 1：筛选主办方 默认是0
     * 注：h5 活动不支持此接口
     */
    @Override
    public void requestCommentHistory(String webinar_id, int limit, int page, String msgId, String anchor_path, String is_role, ChatServer.ChatRecordCallback callback) {
        if (checkCommentParameter(limit, page, callback)) {
            if (mPlayer != null) {
                mPlayer.requestCommentHistory(webinar_id, limit, page, msgId, anchor_path, is_role, callback);
            }
        }
    }

    private boolean checkCommentParameter(int limit, int pos, final ChatServer.ChatRecordCallback callback) {
        if (callback == null) {
            throw new IllegalArgumentException(ErrorCode.CALLBACK_ERROR_STR);
        }

        if (!isAvaliable() || TextUtils.isEmpty(webinarInfo.join_id)) {
            callback.onFailed(ErrorCode.ERROR_INIT, context.getString(R.string.error_video_info_init));
            return false;
        }
        if (webinarInfo.status == WebinarInfo.LIVE) {
            callback.onFailed(ErrorCode.ERROR_PARAM, context.getString(R.string.playing));
            return false;
        }
        if (limit <= 0 || pos < 0) {
            callback.onFailed(ErrorCode.ERROR_PARAM, ErrorCode.ERROR_PARAM_STR);
            return false;
        }
        return true;
    }

}
