package com.vhall.business;

import android.content.Context;
import android.graphics.Bitmap;
import android.text.TextUtils;
import android.view.View;
import android.widget.RelativeLayout;

import com.vhall.business.data.RequestCallback;
import com.vhall.business.data.WebinarInfo;
import com.vhall.business.module.message.IMessageCallBack;
import com.vhall.business.module.notice.NoticeMessageCallBack;
import com.vhall.player.VHPlayerListener;
import com.vhall.player.stream.play.IVHVideoPlayer;
import com.vhall.player.stream.play.impl.VHVideoPlayerView;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;

import static com.vhall.business.ErrorCode.ERROR_NO_SUPPORT;
import static com.vhall.business.MessageServer.EVENT_NOTICE;
import static com.vhall.business.common.Constants.TYPE_FLASH;
import static com.vhall.business.common.Constants.TYPE_H5;
import static com.vhall.business.common.Constants.TYPE_H5_NEW;
import static com.vhall.business.module.message.IMessageCallBack.KEY_MESSAGE_NOTICE;

/**
 * Created by zwp on 2018/12/26
 */
public class WatchLive extends Live {

    private static final String TAG = "WatchLive";
    private Live watchLivePlayer;

    @Override
    public String getDefinition() {
        if (watchLivePlayer != null) {
            return watchLivePlayer.getDefinition();
        }
        return null;
    }

    @Override
    public void setDefinition(String definition) {
        if (watchLivePlayer != null) {
            watchLivePlayer.setDefinition(definition);
        }
    }

    @Override
    public void setPCSwitchDefinition() {
        if (watchLivePlayer != null) {
            watchLivePlayer.setPCSwitchDefinition();
        }
    }

    @Override
    public boolean isPlaying() {
        if (watchLivePlayer != null) {
            return watchLivePlayer.isPlaying();
        }
        return false;
    }

    @Override
    public void mute() {
        if (null != watchLivePlayer) {
            watchLivePlayer.mute();
        }
    }

    @Override
    public void unMute() {
        if (null != watchLivePlayer) {
            watchLivePlayer.unMute();
        }
    }

    @Override
    public void setScaleType(int scaleType) {
        if (watchLivePlayer != null) {
            watchLivePlayer.setScaleType(scaleType);
        }

    }

    @Override
    public void setWebinarInfo(WebinarInfo webinarInfo) {
        super.setWebinarInfo(webinarInfo);
        /**
         * 容器非空时，如videoView 为null自动创建
         */
        Live.Builder builder = new Live.Builder();
        builder.playView(this.videoView)
                .callback(this.listener)
                .bufferDelay(this.buffSeconds)
                .connectTimeoutMils(this.connectTimeout)
                .messageCallback(this.messageCallback)
                .context(this.context)
                .containerLayout(this.videoContainer)
                .chatCallback(this.chatCallback);
        String type;

        if (webinarInfo.is_new_version == 3) {
            type = TYPE_H5_NEW;
        } else {
            type = TextUtils.isEmpty(webinarInfo.vss_token) ? TYPE_FLASH : TYPE_H5;
        }
        watchLivePlayer = builder.buildLive(type);
        watchLivePlayer.setWebinarInfo(webinarInfo);

        if (webinarInfo.notice != null && !TextUtils.isEmpty(webinarInfo.notice.content)) {
            MessageServer.MsgInfo messageInfo = new MessageServer.MsgInfo();
            messageInfo.content = webinarInfo.notice.content;
            messageInfo.publish_release_time = webinarInfo.notice.publish_release_time;
            messageInfo.duration = webinarInfo.notice.duration;
            messageInfo.event = EVENT_NOTICE;
            if (messageCallback != null) {
                messageCallback.onEvent(messageInfo);
            }
            List<IMessageCallBack> messageCallBackByKey = NewH5ImManager.getInstance().getMessageCallBackByKey(KEY_MESSAGE_NOTICE);
            if (messageCallBackByKey != null && messageCallBackByKey.size() > 0) {
                for (IMessageCallBack iMessageCallBack : messageCallBackByKey) {
                    if (iMessageCallBack instanceof NoticeMessageCallBack) {
                        ((NoticeMessageCallBack) iMessageCallBack).noticeSend(messageInfo);
                    }
                }
            }
        }
    }


    private WatchLive(Builder builder) {
        this.context = builder.context;
        this.videoContainer = builder.videoContainer;
        this.listener = builder.listener;
        this.videoView = builder.videoView;
        this.buffSeconds = builder.buffSeconds;
        this.connectTimeout = builder.connectTimeout;
        this.messageCallback = builder.messageCallback;
        this.chatCallback = builder.chatCallback;
    }

    public static class Builder {
        Context context;
        RelativeLayout videoContainer;
        VHPlayerListener listener;
        int buffSeconds = 6;
        int connectTimeout = 5000;
        MessageServer.Callback messageCallback;
        ChatServer.Callback chatCallback;
        IVHVideoPlayer videoView;

        public Builder context(Context context) {
            this.context = context;
            return this;
        }

        public Builder containerLayout(RelativeLayout container) {
            this.videoContainer = container;
            return this;
        }

        public Builder callback(VHPlayerListener listener) {
            this.listener = listener;
            return this;
        }

        public Builder messageCallback(MessageServer.Callback messageCallback) {
            this.messageCallback = messageCallback;
            return this;
        }

        public Builder chatCallback(ChatServer.Callback chatCallback) {
            this.chatCallback = chatCallback;
            return this;
        }

        public Builder bufferDelay(int buffSeconds) {
            if (buffSeconds > 0) {
                this.buffSeconds = buffSeconds;
            }
            return this;
        }


        public Builder connectTimeoutMils(int millisecond) {
            if (millisecond > 0) {
                this.connectTimeout = millisecond;
            }
            return this;
        }

        public Builder playView(IVHVideoPlayer playView) {
            this.videoView = playView;
            return this;
        }

        public WatchLive build() {
            if (context == null) {
                throw new IllegalArgumentException("context can not be null");
            }
            if (videoContainer == null && videoView == null) {
                throw new IllegalArgumentException("containerLayout and playView can not both be null");
            }
            return new WatchLive(this);
        }
    }


    @Override
    public void start() {
        if (watchLivePlayer != null) {
            if (watchLivePlayer.isPlaying()) {
                return;
            }
            if (!isAvaliable()) {
                listener.onError(ERROR_NOT_INIT, 0, context.getString(R.string.error_video_msg_init));
                return;
            }
            watchLivePlayer.start();
        }
    }

    @Override
    public void stop() {
        if (watchLivePlayer == null) {
            return;
        }
        watchLivePlayer.stop();
    }

    @Override
    public void destroy() {
        if (watchLivePlayer != null) {
            watchLivePlayer.destroy();
        }
        destoryPlayview();
    }

    @Override
    public void startPlay(String url) {
        if (watchLivePlayer != null) {
            watchLivePlayer.startPlay(url);
        }
    }

    @Override
    public boolean setVideoBackgroundColor(int color) {
        if (watchLivePlayer != null)
            return watchLivePlayer.setVideoBackgroundColor(color);

        return false;
    }

    @Override
    public boolean setVideoBackgroundImage(Bitmap bitmap) {
        if (bitmap == null) {
            return false;
        }
        if (watchLivePlayer != null)
            return watchLivePlayer.setVideoBackgroundImage(bitmap);
        return false;
    }

    @Override
    public boolean takeVideoScreenshot(VHVideoPlayerView.ScreenShotCallback callback) {
        if (callback == null) {
            return false;
        }
        if (watchLivePlayer != null)
            return watchLivePlayer.takeVideoScreenshot(callback);
        return false;
    }

    //化蝶有效
    public boolean setLiveSubtitle(boolean open){
        if (watchLivePlayer != null && (watchLivePlayer instanceof WatchLiveH5New)){
            return ((WatchLiveH5New)watchLivePlayer).setLiveSubtitle(open);
        }

        return true;
    }

    @Deprecated
    @Override
    public void releasePlayer() {
        if (watchLivePlayer != null) {
            watchLivePlayer.releasePlayer();
        }
    }


    private void destoryPlayview() {
        if (videoView != null) {
            if (videoContainer != null) {
                videoContainer.removeView((View) videoView);
            }
        }
    }

    @Override
    public void sendChat(String content, RequestCallback callback) {

        if (!isAvaliable()) {
            callback.onError(ErrorCode.ERROR_INIT, context.getString(R.string.error_msg_init));
            return;
        }
        if (webinarInfo.chatforbid) {
            callback.onError(ErrorCode.ERROR_BANNED, context.getString(R.string.error_chat_forbid_open));
            return;
        }
        if (webinarInfo.filters != null) {
            for (int i = 0; i < webinarInfo.filters.size(); i++) {
                if (content.contains(webinarInfo.filters.get(i))) {
                    VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_KEY_FILTERS, context.getString(R.string.error_chat_forbid));
                    return;
                }
            }
        }

        if (watchLivePlayer != null) {
            watchLivePlayer.sendChat(content, callback);
        }
    }

    @Override
    public void sendCustom(JSONObject content, RequestCallback callback) {
        if (watchLivePlayer != null) {
            watchLivePlayer.sendCustom(content, callback);
        }
    }

    @Override
    public void sendQuestion(String content, RequestCallback callback) {
        if (!VhallSDK.isLogin()) {
            return;
        }
        if (watchLivePlayer != null) {
            if (TextUtils.isEmpty(content)) {
                VhallCallback.ErrorCallback(callback, ErrorCode.ERROR_PARAM, "内容不能为空");
                return;
            }
            watchLivePlayer.sendQuestion(content, callback);
        }
    }

    @Override
    public void onRaiseHand(String webinarId, int type, RequestCallback callback) {
        if (!VhallSDK.isLogin()) {
            callback.onError(ErrorCode.ERROR_ISLOGIN, ErrorCode.ERROR_MSG_ISLOGIN);
            return;
        }
        if (watchLivePlayer == null) {
            callback.onError(ErrorCode.ERROR_INIT, ErrorCode.ERROR_MSG_INIT);
            return;
        }
        if (webinarInfo == null || TextUtils.isEmpty(webinarInfo.join_id)) {
            callback.onError(ErrorCode.ERROR_PARAM, ErrorCode.ERROR_MESSAGE_NOT_EMPTY_STR);
            return;
        }
        watchLivePlayer.onRaiseHand(webinarId, type, callback);
    }

    /**
     * 收到邀请后 反馈处理状态
     *
     * @param type     1接受，2拒绝，3超时失败
     * @param callback 结果回调
     */
    @Override
    public void replyInvitation(String webinarId, int type, RequestCallback callback) {
        if (!VhallSDK.isLogin()) {
            callback.onError(ErrorCode.ERROR_ISLOGIN, ErrorCode.ERROR_MSG_ISLOGIN);
            return;
        }
        if (webinarInfo == null || TextUtils.isEmpty(webinarInfo.join_id)) {
            callback.onError(ErrorCode.ERROR_PARAM, ErrorCode.ERROR_PARAM_STR);
            return;
        }
        if (watchLivePlayer != null) {
            watchLivePlayer.replyInvitation(webinarId, type, callback);
        } else {
            callback.onError(ErrorCode.ERROR_INIT, ErrorCode.ERROR_MSG_INIT);
        }
    }

    /**
     * @param show_all           false 显示当次直播聊天最多为20条,true显示所有聊天最条为100条，
     * @param chatRecordCallback 回调
     */
    @Override
    public void acquireChatRecord(boolean show_all, ChatServer.ChatRecordCallback chatRecordCallback) {
        if (watchLivePlayer != null) {
            watchLivePlayer.acquireChatRecord(show_all, chatRecordCallback);
        }
    }

    @Deprecated
    @Override
    public void acquireChatRecord(int page, ChatServer.ChatRecordCallback chatRecordCallback) {
        if (watchLivePlayer != null) {
            watchLivePlayer.acquireChatRecord(page, chatRecordCallback);
        }
    }
    /**
     * 获取当前房间聊天列表
     *
     * @param page        获取条目节点，默认为1
     * @param limit       获取条目数量，最大100
     * @param msgId      获取条目数量，聊天记录 锚点消息id,此参数存在时anchor_path 参数必须存在
     * @param anchorPath 锚点方向，up 向上查找，down 向下查找,此参数存在时 msg_id 参数必须存在
     * @param isRole     0：不筛选主办方 1：筛选主办方 默认是0
     * 注：h5 活动不支持此接口
     */
    @Override
    public void acquireChatRecord(int page, int limit, String msgId, String anchorPath, String isRole, ChatServer.ChatRecordCallback chatRecordCallback) {
        if (watchLivePlayer != null) {
            watchLivePlayer.acquireChatRecord(page, limit, msgId, anchorPath, isRole, chatRecordCallback);
        }
    }

    @Override
    public HashMap<String, Integer> getDefinitionAvailable() {
        if (watchLivePlayer != null) {
            return watchLivePlayer.getDefinitionAvailable();
        }
        return null;
    }

    @Override
    public String getOriginalUrl() {
        if (watchLivePlayer != null) {
            return watchLivePlayer.getOriginalUrl();
        }
        return null;
    }

    @Override
    public void connectChatServer() {
        if (watchLivePlayer != null) {
            watchLivePlayer.connectChatServer();
        }
    }

    @Override
    public void disconnectChatServer() {
        if (watchLivePlayer != null) {
            watchLivePlayer.disconnectChatServer();
        }
    }

    @Override
    public void connectMsgServer() {
        if (watchLivePlayer != null) {
            watchLivePlayer.connectMsgServer();
        }
    }

    @Override
    public void disconnectMsgServer() {
        if (watchLivePlayer != null) {
            watchLivePlayer.disconnectMsgServer();
        }
    }

    /**
     *
     * @param delayMs 文档延迟的毫秒数，小于0时则由sdk内部维护一个适合有延迟场景(看直播)下的值
     */
    @Override
    public void setOPSDelay(int delayMs) {
        if (null != watchLivePlayer) {
            watchLivePlayer.setOPSDelay(delayMs);
        }
    }
}
