package com.vhall.beautify;

import static java.lang.Math.abs;

import android.content.Context;
import android.content.pm.ActivityInfo;
import android.hardware.Camera;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.os.Bundle;
import android.util.Log;

import com.faceunity.core.callback.OperateCallback;
import com.faceunity.core.entity.FURenderInputData;
import com.faceunity.core.entity.FURenderOutputData;
import com.faceunity.core.enumeration.CameraFacingEnum;
import com.faceunity.core.enumeration.FUAITypeEnum;
import com.faceunity.core.enumeration.FUExternalInputEnum;
import com.faceunity.core.enumeration.FUInputBufferEnum;
import com.faceunity.core.enumeration.FUInputTextureEnum;
import com.faceunity.core.enumeration.FUTransformMatrixEnum;
import com.faceunity.core.faceunity.FUAIKit;
import com.faceunity.core.faceunity.FURenderKit;
import com.faceunity.core.faceunity.FURenderManager;
import com.faceunity.core.model.facebeauty.FaceBeauty;
import com.faceunity.core.model.facebeauty.FaceBeautyBlurTypeEnum;
import com.faceunity.core.utils.FULogger;
import com.faceunity.wrapper.faceunity;
import com.vhall.beautify.type.VHBeautifyCode;
import com.vhall.beautify.type.VHBeautifyConfig;

import org.jetbrains.annotations.NotNull;

import java.lang.ref.WeakReference;
import java.util.Arrays;

/**
 * @author：jooper Email：jooperge@163.com
 * 描述：相芯美颜实现层
 * 修改历史:
 * 1。各入参合法性由上层控制，避免底层冗余
 * <p>
 * 创建于： 2021/12/08
 */
public class VHBeautifyImpl implements IVHBeautifyService {

    private static final String TAG = "VH_BeautifyImpl";
    private FUExternalInputEnum externalInputType = FUExternalInputEnum.EXTERNAL_INPUT_TYPE_CAMERA;//数据源类型
    private FUInputTextureEnum inputTextureType = FUInputTextureEnum.FU_ADM_FLAG_COMMON_TEXTURE;//纹理类型
    private FUInputBufferEnum inputBufferType = FUInputBufferEnum.FU_FORMAT_NV21_BUFFER;//数据类型
    private final FURenderKit mFURenderKit = FURenderKit.getInstance();
    private final FUAIKit mFUAIKit = FUAIKit.getInstance();
    private static final String ERRMSG_AUTH_NULL = "SDK授权信息不能为空";
    private DefaultFaceBeautyDataFactory faceBeautyDataFactory;
    /**
     * 定制陀螺仪生命周期：bindFaceBeautySource() -> release()
     */
    private SensorManager mSensorManager = null;
    private Sensor mSensor = null;
    /**
     * ApplicationContext
     */
    private WeakReference<Context> mWeakContext = null;
    /**
     * 手机设备朝向
     */
    private int mDeviceOrientation = 90;
    /**
     * true: RTC    false:lss
     */
    private boolean renderOfRTC = false;
    /**
     * 数据源人脸方向
     */
    private final int DEFAULT_INPUTORIENTATION = 0;
    private int mInputOrientation = DEFAULT_INPUTORIENTATION;
    /**
     * 当前activity方向
     */
    private int mActivityOrientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
    /**
     * 前后置摄像头
     */
    private int mCameraFacing = Camera.CameraInfo.CAMERA_FACING_FRONT;

    /**
     * 美颜注册是否成功
     */
    private static boolean mRegisterSucceeded = false;

    @Override
    public void init(Context context, IVHBeautifyInitListener ivhBeautifyInitListener) {
        Bundle config = new Bundle();
        config.putByteArray("auth", new AuthUtil().byteArray());
        init(context, config, ivhBeautifyInitListener);
    }

    private void init(Context context, Bundle config, IVHBeautifyInitListener initListener) {
        Log.d(TAG, ">>>init FU beautify");

        byte[] auth = config.getByteArray("auth");
        if (null == auth) {
            Log.d(TAG, ERRMSG_AUTH_NULL);
            if (null != initListener) {
                initListener.onError(VHBeautifyCode.CODE_AUTHCONFIG_NULL, ERRMSG_AUTH_NULL);
            }
            return;
        }
        mWeakContext = new WeakReference<>(context.getApplicationContext());
        FURenderManager.setKitDebug(FULogger.LogLevel.ERROR);
        FURenderManager.setCoreDebug(FULogger.LogLevel.INFO);
        FURenderManager.registerFURender(context, auth, new OperateCallback() {
            @Override
            public void onSuccess(int code, @NotNull String msg) {
                Log.d(TAG, "success: " + code + " - " + msg);
                mRegisterSucceeded = true;
                if (null != initListener) {
                    initListener.onSuccess();
                }
            }

            @Override
            public void onFail(int errCode, @NotNull String msg) {
                int vhErrCode = ParamMappingHelper.mappingCode(errCode);
                Log.d(TAG, "vhErrCode: " + vhErrCode + "    thirdErrCode: " + errCode + "   msg: " + msg);
                mRegisterSucceeded = false;
                if (null != initListener) {
                    initListener.onError(vhErrCode, msg);
                }
            }
        });
    }

    private DefaultFaceBeautyDataFactory beautyDataFactory() {
        if (null == faceBeautyDataFactory) {
            faceBeautyDataFactory = new DefaultFaceBeautyDataFactory();
        }
        return faceBeautyDataFactory;
    }

    private void bindFaceBeautySource(FaceBeauty faceBeauty) {
        if (mRegisterSucceeded) {
            try {
                getSensorManager().registerListener(mSensorEventListener, getSensor(), SensorManager.SENSOR_DELAY_NORMAL);
            } catch (Exception e) {
                e.printStackTrace();
            }
            try {
                FURenderKit.getInstance().setFaceBeauty(faceBeauty);
            } catch (Throwable e) {
                e.printStackTrace();
            }
            loadFaceProcessor();
        }

    }

    @Override
    public int renderWithTexture(int width, int height, int texture) {
        FURenderInputData inputData = new FURenderInputData(width, height);
        inputData.setTexture(new FURenderInputData.FUTexture(inputTextureType, texture));
        FURenderInputData.FURenderConfig fuRenderConfig = inputData.getRenderConfig();
        fuRenderConfig.setExternalInputType(FUExternalInputEnum.EXTERNAL_INPUT_TYPE_CAMERA);

        calculateInputOrientation();
        fuRenderConfig.setInputOrientation(mInputOrientation);
        resetInputOrientation();
        fuRenderConfig.setDeviceOrientation(mDeviceOrientation);
        fuRenderConfig.setInputTextureMatrix(FUTransformMatrixEnum.CCROT0);
        fuRenderConfig.setCameraFacing(mCameraFacing == Camera.CameraInfo.CAMERA_FACING_FRONT ? CameraFacingEnum.CAMERA_FRONT : CameraFacingEnum.CAMERA_BACK);
        fuRenderConfig.setOutputMatrix(FUTransformMatrixEnum.CCROT0_FLIPVERTICAL);
        renderBefore(inputData);
        FURenderOutputData outputData = mFURenderKit.renderWithInput(inputData);
        renderAfter();
        if (null != outputData && null != outputData.getTexture() && 0 < outputData.getTexture().getTexId()) {
            return outputData.getTexture().getTexId();
        } else {
            faceunity.fuGetSystemError();
            Log.e(TAG, "faceunity异常--->[faceunity.fuGetSystemError()]，请检查控制台输出");
            return texture;
        }
    }

    private void calculateInputOrientation() {
        if (renderOfRTC) {
            if (mCameraFacing == Camera.CameraInfo.CAMERA_FACING_BACK) {
                mInputOrientation = 90;
            } else {
                mInputOrientation = 270;
            }
        } else {
            if (mCameraFacing == Camera.CameraInfo.CAMERA_FACING_BACK) {
                if (mDeviceOrientation == 0 | mDeviceOrientation == 180) {
                    mInputOrientation = isActvityPortrait() ? 180 : 270;
                } else {
                    mInputOrientation = isActvityPortrait() ? 0 : 90;
                }
            } else {
                resetInputOrientation();
            }
        }
    }

    private void resetInputOrientation() {
        if (isActvityPortrait()) {
            mInputOrientation = DEFAULT_INPUTORIENTATION;
        } else {
            mInputOrientation = 90;
        }
    }

    private boolean isActvityPortrait() {
        return mActivityOrientation == ActivityInfo.SCREEN_ORIENTATION_PORTRAIT;
    }

    @Override
    public void setRenderOfRTC(boolean renderOfRTC) {
        this.renderOfRTC = renderOfRTC;
    }

    @Override
    public String sdkModel() {
        return VHBeautifyConfig.MODEL_FACEUNITY;
    }

    //开启camera时过滤的帧数
    private int openCameraIgnoreFrame = 0;

    @Override
    public void onCameraSwitch(int cameraFacing) {
        mCameraFacing = cameraFacing;
    }

    @Override
    public void try2Restore() {
        VHBeautifyKit.getInstance().setBeautifyEnable(VHBeautifyKit.getInstance().isBeautifyEnable());
    }

    @Override
    public void loadFaceProcessor() {
        try {
            FUAIKit.getInstance().loadAIProcessor(DemoConfig.BUNDLE_AI_FACE, FUAITypeEnum.FUAITYPE_FACEPROCESSOR);
        } catch (IllegalStateException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void judgeDeviceLevel(WeakReference<Context> activityWeakReference) {
        if (null != activityWeakReference && null != activityWeakReference.get()) {
            DemoConfig.DEVICE_LEVEL = FuDeviceUtils.judgeDeviceLevel(activityWeakReference.get());
        }
    }

    @Override
    public void setFilterIntensity(double filterIntensity) {
        if (checkFURenderKit()) {
            mFURenderKit.getFaceBeauty().setFilterIntensity(filterIntensity);
        }
    }

    @Override
    public void setFilter(String filterName, double filterIntensity) {
        if (checkFURenderKit()) {
            mFURenderKit.getFaceBeauty().setFilterName(ParamMappingHelper.mapping(filterName));
            mFURenderKit.getFaceBeauty().setFilterIntensity(filterIntensity);
        }
    }

    @Override
    public void updateParamIntensity(String key, double value) {
        if (key.startsWith("vh_")) {
            beautyDataFactory().updateParamIntensity(ParamMappingHelper.mapping(key), value);
        } else {
            beautyDataFactory().updateParamIntensity(key, value);
        }
    }

    @Override
    public double getParamIntensity(String key) {
        return beautyDataFactory().getParamIntensity(key);
    }

    @Override
    public void onBeautifyEnable(boolean enable) {
        if (enable) {
            bindFaceBeautySource(currentFaceBeauty());
        }
    }

    @Override
    public void setActivityOrientation(int activityOrientation) {
        mActivityOrientation = activityOrientation;
    }

    private FaceBeauty currentFaceBeauty() {
        FaceBeauty faceBeauty = FURenderKit.getInstance().getFaceBeauty();
        return null == faceBeauty ? DefaultFaceBeautyDataFactory.defaultFaceBeauty : faceBeauty;
    }

    @Override
    public void release() {
        renderOfRTC = false;
        if (null != mFURenderKit) {
            mFURenderKit.release();
        }
        if (null != mFUAIKit) {
            mFUAIKit.releaseAllAIProcessor();
        }
        if (null != mSensorManager) {
            mSensorManager.unregisterListener(mSensorEventListener);
        }
    }

    /**
     * @return true:可用
     */
    private boolean checkFURenderKit() {
        boolean isFURenderKitEnable = null != mFURenderKit && null != mFURenderKit.getFaceBeauty();
        if (!isFURenderKitEnable) {
            Log.e(TAG, ">>>FURenderKit is null");
        }
        return isFURenderKitEnable;
    }

    //上下镜像
    private float[] TEXTURE_MATRIX_CCRO_FLIPV_0 = {
            1.0f, 0.0f, 0.0f, 0.0f, 0.0f, -1.0f, 0.0f, 0.0f,
            0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f
    };

    //渲染矩阵
    private float[] TEXTURE_MATRIX = {
            1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f, 0.0f, 0.0f,
            0.0f, 0.0f, 1.0f, 0.0f, 0.0f, 0.0f, 0.0f, 1.0f
    };

    protected volatile FURenderInputData currentFURenderInputData = new FURenderInputData(0, 0);

    private final Object mFURenderInputDataLock = new Object();
    private float[] defaultFUTexMatrix = Arrays.copyOf(TEXTURE_MATRIX, TEXTURE_MATRIX.length);//默认FURender图形矩阵
    private float[] defaultFUMvpMatrix = Arrays.copyOf(TEXTURE_MATRIX, TEXTURE_MATRIX.length);//默认FURender图形矩阵
    private float[] currentFUTexMatrix = Arrays.copyOf(TEXTURE_MATRIX, TEXTURE_MATRIX.length);//最终渲染FURender图形矩阵
    private float[] currentFUMvpMatrix = Arrays.copyOf(TEXTURE_MATRIX, TEXTURE_MATRIX.length);//最终渲染FURender图形矩阵
    private float[] originTexMatrix = Arrays.copyOf(TEXTURE_MATRIX, TEXTURE_MATRIX.length); //原始图形纹理矩阵
    private float[] originMvpMatrix = Arrays.copyOf(TEXTURE_MATRIX, TEXTURE_MATRIX.length);//原始图形绑定矩阵
    private float[] smallViewMatrix = Arrays.copyOf(TEXTURE_MATRIX, TEXTURE_MATRIX.length);//小窗图形绑定矩阵

    private void renderBefore(FURenderInputData inputData) {
        checkSpecialDevice(inputData);
        if (DemoConfig.DEVICE_LEVEL > FuDeviceUtils.DEVICE_LEVEL_MID)//高性能设备 并且 人脸场景 -> 才会走磨皮策略
            cheekFaceNum();
        inputData.setImageBuffer(null);
    }

    private void renderAfter() {
        if (null != VHBeautifyKit.getInstance().getFaceDetectionListener() &&
                VHBeautifyKit.getInstance().isFaceDetectionEnable()) {
            VHBeautifyKit.getInstance().getFaceDetectionListener().onFaceDetection(mFUAIKit.isTracking());
        }
    }

    /**
     * 检查当前人脸数量
     */
    private void cheekFaceNum() {
        //根据有无人脸 + 设备性能 判断开启的磨皮类型
        float faceProcessorGetConfidenceScore = mFUAIKit.getFaceProcessorGetConfidenceScore(0);
        if (faceProcessorGetConfidenceScore >= 0.95) {
            //高端手机并且检测到人脸开启均匀磨皮，人脸点位质
            if (mFURenderKit != null && mFURenderKit.getFaceBeauty() != null && mFURenderKit.getFaceBeauty().getBlurType() != FaceBeautyBlurTypeEnum.EquallySkin) {
                mFURenderKit.getFaceBeauty().setBlurType(FaceBeautyBlurTypeEnum.EquallySkin);
                mFURenderKit.getFaceBeauty().setEnableBlurUseMask(true);
            }
        } else {
            if (mFURenderKit != null && mFURenderKit.getFaceBeauty() != null && mFURenderKit.getFaceBeauty().getBlurType() != FaceBeautyBlurTypeEnum.FineSkin) {
                mFURenderKit.getFaceBeauty().setBlurType(FaceBeautyBlurTypeEnum.FineSkin);
                mFURenderKit.getFaceBeauty().setEnableBlurUseMask(false);
            }
        }
    }

    /**
     * 检查是否特殊设备
     *
     * @param inputData
     */
    private void checkSpecialDevice(FURenderInputData inputData) {
        if (FuDeviceUtils.isNexus6P()) {
            if (inputData.getRenderConfig().getCameraFacing() == CameraFacingEnum.CAMERA_FRONT) {
                inputData.getRenderConfig().setInputTextureMatrix(FUTransformMatrixEnum.CCROT90_FLIPVERTICAL);
                inputData.getRenderConfig().setInputBufferMatrix(FUTransformMatrixEnum.CCROT90_FLIPVERTICAL);
            }
        }
    }

    private FURenderInputData buildFURenderInputData() {
        synchronized (mFURenderInputDataLock) {
            currentFURenderInputData.clone();
            //切换相机的时候忽略几帧不处理，
            if (openCameraIgnoreFrame > 0) {
                openCameraIgnoreFrame--;
                currentFURenderInputData.setImageBuffer(null);
                currentFURenderInputData.setTexture(null);
            }
            return currentFURenderInputData;
        }
    }

    private SensorManager getSensorManager() {
        if (null == mSensorManager && null != mWeakContext) {
            mSensorManager = (SensorManager) mWeakContext.get().getSystemService(Context.SENSOR_SERVICE);
        }
        return mSensorManager;
    }

    private Sensor getSensor() {
        if (null == mSensor) {
            mSensor = mSensorManager.getDefaultSensor(Sensor.TYPE_ACCELEROMETER);
        }
        return mSensor;
    }

    private final SensorEventListener mSensorEventListener = new SensorEventListener() {
        @Override
        public void onSensorChanged(SensorEvent event) {
            if (event.sensor.getType() == Sensor.TYPE_ACCELEROMETER) {
                float x = event.values[0];
                float y = event.values[1];
                if (abs(x) > 3 || abs(y) > 3) {
                    if (abs(x) > abs(y)) {
                        mDeviceOrientation = x > 0 ? 0 : 180;
                    } else {
                        mDeviceOrientation = y > 0 ? 90 : 270;
                    }
                }
            }
        }

        @Override
        public void onAccuracyChanged(Sensor sensor, int accuracy) {

        }
    };
}