/*
 * Decompiled with CFR 0.152.
 */
package org.tron.keystore;

import java.nio.charset.StandardCharsets;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.UUID;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.spongycastle.crypto.Digest;
import org.spongycastle.crypto.digests.SHA256Digest;
import org.spongycastle.crypto.generators.PKCS5S2ParametersGenerator;
import org.spongycastle.crypto.generators.SCrypt;
import org.spongycastle.crypto.params.KeyParameter;
import org.tron.common.crypto.ECKey;
import org.tron.common.crypto.Hash;
import org.tron.common.utils.ByteArray;
import org.tron.keystore.CipherException;
import org.tron.keystore.WalletFile;

public class Wallet {
    private static final int N_LIGHT = 4096;
    private static final int P_LIGHT = 6;
    private static final int N_STANDARD = 262144;
    private static final int P_STANDARD = 1;
    private static final int R = 8;
    private static final int DKLEN = 32;
    private static final int CURRENT_VERSION = 3;
    private static final String CIPHER = "aes-128-ctr";
    protected static final String AES_128_CTR = "pbkdf2";
    protected static final String SCRYPT = "scrypt";

    public static WalletFile create(String password, ECKey ecKeyPair, int n, int p) throws CipherException {
        byte[] salt = Wallet.generateRandomBytes(32);
        byte[] derivedKey = Wallet.generateDerivedScryptKey(password.getBytes(StandardCharsets.UTF_8), salt, n, 8, p, 32);
        byte[] encryptKey = Arrays.copyOfRange(derivedKey, 0, 16);
        byte[] iv = Wallet.generateRandomBytes(16);
        byte[] privateKeyBytes = ecKeyPair.getPrivKeyBytes();
        byte[] cipherText = Wallet.performCipherOperation(1, iv, encryptKey, privateKeyBytes);
        byte[] mac = Wallet.generateMac(derivedKey, cipherText);
        return Wallet.createWalletFile(ecKeyPair, cipherText, iv, salt, mac, n, p);
    }

    public static WalletFile createStandard(String password, ECKey ecKeyPair) throws CipherException {
        return Wallet.create(password, ecKeyPair, 262144, 1);
    }

    public static WalletFile createLight(String password, ECKey ecKeyPair) throws CipherException {
        return Wallet.create(password, ecKeyPair, 4096, 6);
    }

    private static WalletFile createWalletFile(ECKey ecKeyPair, byte[] cipherText, byte[] iv, byte[] salt, byte[] mac, int n, int p) {
        WalletFile walletFile = new WalletFile();
        walletFile.setAddress(org.tron.core.Wallet.encode58Check(ecKeyPair.getAddress()));
        WalletFile.Crypto crypto = new WalletFile.Crypto();
        crypto.setCipher(CIPHER);
        crypto.setCiphertext(ByteArray.toHexString(cipherText));
        walletFile.setCrypto(crypto);
        WalletFile.CipherParams cipherParams = new WalletFile.CipherParams();
        cipherParams.setIv(ByteArray.toHexString(iv));
        crypto.setCipherparams(cipherParams);
        crypto.setKdf(SCRYPT);
        WalletFile.ScryptKdfParams kdfParams = new WalletFile.ScryptKdfParams();
        kdfParams.setDklen(32);
        kdfParams.setN(n);
        kdfParams.setP(p);
        kdfParams.setR(8);
        kdfParams.setSalt(ByteArray.toHexString(salt));
        crypto.setKdfparams(kdfParams);
        crypto.setMac(ByteArray.toHexString(mac));
        walletFile.setCrypto(crypto);
        walletFile.setId(UUID.randomUUID().toString());
        walletFile.setVersion(3);
        return walletFile;
    }

    private static byte[] generateDerivedScryptKey(byte[] password, byte[] salt, int n, int r, int p, int dkLen) throws CipherException {
        return SCrypt.generate((byte[])password, (byte[])salt, (int)n, (int)r, (int)p, (int)dkLen);
    }

    private static byte[] generateAes128CtrDerivedKey(byte[] password, byte[] salt, int c, String prf) throws CipherException {
        if (!prf.equals("hmac-sha256")) {
            throw new CipherException("Unsupported prf:" + prf);
        }
        PKCS5S2ParametersGenerator gen = new PKCS5S2ParametersGenerator((Digest)new SHA256Digest());
        gen.init(password, salt, c);
        return ((KeyParameter)gen.generateDerivedParameters(256)).getKey();
    }

    private static byte[] performCipherOperation(int mode, byte[] iv, byte[] encryptKey, byte[] text) throws CipherException {
        try {
            IvParameterSpec ivParameterSpec = new IvParameterSpec(iv);
            Cipher cipher = Cipher.getInstance("AES/CTR/NoPadding");
            SecretKeySpec secretKeySpec = new SecretKeySpec(encryptKey, "AES");
            cipher.init(mode, (Key)secretKeySpec, ivParameterSpec);
            return cipher.doFinal(text);
        }
        catch (InvalidAlgorithmParameterException | InvalidKeyException | NoSuchAlgorithmException | BadPaddingException | IllegalBlockSizeException | NoSuchPaddingException e) {
            throw new CipherException("Error performing cipher operation", e);
        }
    }

    private static byte[] generateMac(byte[] derivedKey, byte[] cipherText) {
        byte[] result = new byte[16 + cipherText.length];
        System.arraycopy(derivedKey, 16, result, 0, 16);
        System.arraycopy(cipherText, 0, result, 16, cipherText.length);
        return Hash.sha3(result);
    }

    public static ECKey decrypt(String password, WalletFile walletFile) throws CipherException {
        byte[] derivedKey;
        Wallet.validate(walletFile);
        WalletFile.Crypto crypto = walletFile.getCrypto();
        byte[] mac = ByteArray.fromHexString(crypto.getMac());
        byte[] iv = ByteArray.fromHexString(crypto.getCipherparams().getIv());
        byte[] cipherText = ByteArray.fromHexString(crypto.getCiphertext());
        WalletFile.KdfParams kdfParams = crypto.getKdfparams();
        if (kdfParams instanceof WalletFile.ScryptKdfParams) {
            WalletFile.ScryptKdfParams scryptKdfParams = (WalletFile.ScryptKdfParams)crypto.getKdfparams();
            int dklen = scryptKdfParams.getDklen();
            int n = scryptKdfParams.getN();
            int p = scryptKdfParams.getP();
            int r = scryptKdfParams.getR();
            byte[] salt = ByteArray.fromHexString(scryptKdfParams.getSalt());
            derivedKey = Wallet.generateDerivedScryptKey(password.getBytes(StandardCharsets.UTF_8), salt, n, r, p, dklen);
        } else if (kdfParams instanceof WalletFile.Aes128CtrKdfParams) {
            WalletFile.Aes128CtrKdfParams aes128CtrKdfParams = (WalletFile.Aes128CtrKdfParams)crypto.getKdfparams();
            int c = aes128CtrKdfParams.getC();
            String prf = aes128CtrKdfParams.getPrf();
            byte[] salt = ByteArray.fromHexString(aes128CtrKdfParams.getSalt());
            derivedKey = Wallet.generateAes128CtrDerivedKey(password.getBytes(StandardCharsets.UTF_8), salt, c, prf);
        } else {
            throw new CipherException("Unable to deserialize params: " + crypto.getKdf());
        }
        byte[] derivedMac = Wallet.generateMac(derivedKey, cipherText);
        if (!Arrays.equals(derivedMac, mac)) {
            throw new CipherException("Invalid password provided");
        }
        byte[] encryptKey = Arrays.copyOfRange(derivedKey, 0, 16);
        byte[] privateKey = Wallet.performCipherOperation(2, iv, encryptKey, cipherText);
        return ECKey.fromPrivate(privateKey);
    }

    static void validate(WalletFile walletFile) throws CipherException {
        WalletFile.Crypto crypto = walletFile.getCrypto();
        if (walletFile.getVersion() != 3) {
            throw new CipherException("Wallet version is not supported");
        }
        if (!crypto.getCipher().equals(CIPHER)) {
            throw new CipherException("Wallet cipher is not supported");
        }
        if (!crypto.getKdf().equals(AES_128_CTR) && !crypto.getKdf().equals(SCRYPT)) {
            throw new CipherException("KDF type is not supported");
        }
    }

    static byte[] generateRandomBytes(int size) {
        byte[] bytes = new byte[size];
        new SecureRandom().nextBytes(bytes);
        return bytes;
    }
}

