/*
 * Decompiled with CFR 0.152.
 */
package org.tron.common.runtime.vm;

import java.math.BigInteger;
import java.util.ArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.spongycastle.util.encoders.Hex;
import org.springframework.util.StringUtils;
import org.tron.common.crypto.Hash;
import org.tron.common.runtime.config.VMConfig;
import org.tron.common.runtime.utils.MUtil;
import org.tron.common.runtime.vm.DataWord;
import org.tron.common.runtime.vm.EnergyCost;
import org.tron.common.runtime.vm.LogInfo;
import org.tron.common.runtime.vm.MessageCall;
import org.tron.common.runtime.vm.OpCode;
import org.tron.common.runtime.vm.PrecompiledContracts;
import org.tron.common.runtime.vm.program.Program;
import org.tron.common.runtime.vm.program.Stack;
import org.tron.common.utils.ByteUtil;

public class VM {
    private static final Logger logger = LoggerFactory.getLogger((String)"VM");
    private static final BigInteger _32_ = BigInteger.valueOf(32L);
    private static final String ENERGY_LOG_FORMATE = "{}    Op: [{}]  Energy: [{}] Deep: [{}]  Hint: [{}]";
    private static final BigInteger MEM_LIMIT = BigInteger.valueOf(0x300000L);
    private final VMConfig config;

    public VM() {
        this.config = VMConfig.getInstance();
    }

    public VM(VMConfig config) {
        this.config = config;
    }

    private void checkMemorySize(OpCode op, BigInteger newMemSize) {
        if (newMemSize.compareTo(MEM_LIMIT) > 0) {
            throw Program.Exception.memoryOverflow(op);
        }
    }

    private long calcMemEnergy(EnergyCost energyCosts, long oldMemSize, BigInteger newMemSize, long copySize, OpCode op) {
        long energyCost = 0L;
        this.checkMemorySize(op, newMemSize);
        long memoryUsage = (newMemSize.longValueExact() + 31L) / 32L * 32L;
        if (memoryUsage > oldMemSize) {
            long memWords = memoryUsage / 32L;
            long memWordsOld = oldMemSize / 32L;
            long memEnergy = (long)energyCosts.getMEMORY() * memWords + memWords * memWords / 512L - ((long)energyCosts.getMEMORY() * memWordsOld + memWordsOld * memWordsOld / 512L);
            energyCost += memEnergy;
        }
        if (copySize > 0L) {
            long copyEnergy = (long)energyCosts.getCOPY_ENERGY() * ((copySize + 31L) / 32L);
            energyCost += copyEnergy;
        }
        return energyCost;
    }

    public void step(Program program) {
        if (this.config.vmTrace()) {
            program.saveOpTrace();
        }
        try {
            DataWord size;
            OpCode op = OpCode.code(program.getCurrentOp());
            if (op == null) {
                throw Program.Exception.invalidOpCode(program.getCurrentOp());
            }
            if (!(VMConfig.allowTvmTransferTrc10() || op != OpCode.CALLTOKEN && op != OpCode.TOKENBALANCE && op != OpCode.CALLTOKENVALUE && op != OpCode.CALLTOKENID)) {
                throw Program.Exception.invalidOpCode(program.getCurrentOp());
            }
            program.setLastOp(op.val());
            program.verifyStackSize(op.require());
            program.verifyStackOverflow(op.require(), op.ret());
            long oldMemSize = program.getMemSize();
            Stack stack = program.getStack();
            String hint = "";
            long energyCost = op.getTier().asInt();
            EnergyCost energyCosts = EnergyCost.getInstance();
            DataWord adjustedCallEnergy = null;
            switch (op) {
                case STOP: {
                    energyCost = energyCosts.getSTOP();
                    break;
                }
                case SUICIDE: {
                    energyCost = energyCosts.getSUICIDE();
                    DataWord suicideAddressWord = (DataWord)stack.get(stack.size() - 1);
                    if (!this.isDeadAccount(program, suicideAddressWord) || program.getBalance(program.getContractAddress()).isZero()) break;
                    energyCost += (long)energyCosts.getNEW_ACCT_SUICIDE();
                    break;
                }
                case SSTORE: {
                    DataWord newValue = (DataWord)stack.get(stack.size() - 2);
                    DataWord oldValue = program.storageLoad((DataWord)stack.peek());
                    if (oldValue == null && !newValue.isZero()) {
                        energyCost = energyCosts.getSET_SSTORE();
                        break;
                    }
                    if (oldValue != null && newValue.isZero()) {
                        program.futureRefundEnergy(energyCosts.getREFUND_SSTORE());
                        energyCost = energyCosts.getCLEAR_SSTORE();
                        break;
                    }
                    energyCost = energyCosts.getRESET_SSTORE();
                    break;
                }
                case SLOAD: {
                    energyCost = energyCosts.getSLOAD();
                    break;
                }
                case TOKENBALANCE: 
                case BALANCE: {
                    energyCost = energyCosts.getBALANCE();
                    break;
                }
                case MSTORE: {
                    energyCost = this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.peek(), new DataWord(32)), 0L, op);
                    break;
                }
                case MSTORE8: {
                    energyCost = this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.peek(), new DataWord(1)), 0L, op);
                    break;
                }
                case MLOAD: {
                    energyCost = this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.peek(), new DataWord(32)), 0L, op);
                    break;
                }
                case RETURN: 
                case REVERT: {
                    energyCost = (long)energyCosts.getSTOP() + this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.peek(), (DataWord)stack.get(stack.size() - 2)), 0L, op);
                    break;
                }
                case SHA3: {
                    energyCost = (long)energyCosts.getSHA3() + this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.peek(), (DataWord)stack.get(stack.size() - 2)), 0L, op);
                    size = (DataWord)stack.get(stack.size() - 2);
                    long chunkUsed = (size.longValueSafe() + 31L) / 32L;
                    energyCost += chunkUsed * (long)energyCosts.getSHA3_WORD();
                    break;
                }
                case CALLDATACOPY: 
                case RETURNDATACOPY: {
                    energyCost = this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.peek(), (DataWord)stack.get(stack.size() - 3)), ((DataWord)stack.get(stack.size() - 3)).longValueSafe(), op);
                    break;
                }
                case CODECOPY: {
                    energyCost = this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.peek(), (DataWord)stack.get(stack.size() - 3)), ((DataWord)stack.get(stack.size() - 3)).longValueSafe(), op);
                    break;
                }
                case EXTCODESIZE: {
                    energyCost = energyCosts.getEXT_CODE_SIZE();
                    break;
                }
                case EXTCODECOPY: {
                    energyCost = (long)energyCosts.getEXT_CODE_COPY() + this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.get(stack.size() - 2), (DataWord)stack.get(stack.size() - 4)), ((DataWord)stack.get(stack.size() - 4)).longValueSafe(), op);
                    break;
                }
                case CALL: 
                case CALLCODE: 
                case DELEGATECALL: 
                case STATICCALL: 
                case CALLTOKEN: {
                    int opOff;
                    DataWord value;
                    energyCost = energyCosts.getCALL();
                    DataWord callEnergyWord = (DataWord)stack.get(stack.size() - 1);
                    DataWord callAddressWord = (DataWord)stack.get(stack.size() - 2);
                    DataWord dataWord = value = op.callHasValue() ? (DataWord)stack.get(stack.size() - 3) : DataWord.ZERO;
                    if ((op == OpCode.CALL || op == OpCode.CALLTOKEN) && this.isDeadAccount(program, callAddressWord) && !value.isZero()) {
                        energyCost += (long)energyCosts.getNEW_ACCT_CALL();
                    }
                    if (!value.isZero()) {
                        energyCost += (long)energyCosts.getVT_CALL();
                    }
                    int n = opOff = op.callHasValue() ? 4 : 3;
                    if (op == OpCode.CALLTOKEN) {
                        ++opOff;
                    }
                    BigInteger in = VM.memNeeded((DataWord)stack.get(stack.size() - opOff), (DataWord)stack.get(stack.size() - opOff - 1));
                    BigInteger out = VM.memNeeded((DataWord)stack.get(stack.size() - opOff - 2), (DataWord)stack.get(stack.size() - opOff - 3));
                    this.checkMemorySize(op, in.max(out));
                    if ((energyCost += this.calcMemEnergy(energyCosts, oldMemSize, in.max(out), 0L, op)) > program.getEnergyLimitLeft().longValueSafe()) {
                        throw new Program.OutOfEnergyException("Not enough energy for '%s' operation executing: opEnergy[%d], programEnergy[%d]", op.name(), energyCost, program.getEnergyLimitLeft().longValueSafe());
                    }
                    DataWord getEnergyLimitLeft = program.getEnergyLimitLeft().clone();
                    getEnergyLimitLeft.sub(new DataWord(energyCost));
                    adjustedCallEnergy = program.getCallEnergy(op, callEnergyWord, getEnergyLimitLeft);
                    energyCost += adjustedCallEnergy.longValueSafe();
                    break;
                }
                case CREATE: {
                    energyCost = (long)energyCosts.getCREATE() + this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.get(stack.size() - 2), (DataWord)stack.get(stack.size() - 3)), 0L, op);
                    break;
                }
                case LOG0: 
                case LOG1: 
                case LOG2: 
                case LOG3: 
                case LOG4: {
                    int nTopics = op.val() - OpCode.LOG0.val();
                    BigInteger dataSize = ((DataWord)stack.get(stack.size() - 2)).value();
                    BigInteger dataCost = dataSize.multiply(BigInteger.valueOf(energyCosts.getLOG_DATA_ENERGY()));
                    if (program.getEnergyLimitLeft().value().compareTo(dataCost) < 0) {
                        throw new Program.OutOfEnergyException("Not enough energy for '%s' operation executing: opEnergy[%d], programEnergy[%d]", op.name(), dataCost.longValueExact(), program.getEnergyLimitLeft().longValueSafe());
                    }
                    energyCost = (long)(energyCosts.getLOG_ENERGY() + energyCosts.getLOG_TOPIC_ENERGY() * nTopics) + (long)energyCosts.getLOG_DATA_ENERGY() * ((DataWord)stack.get(stack.size() - 2)).longValue() + this.calcMemEnergy(energyCosts, oldMemSize, VM.memNeeded((DataWord)stack.peek(), (DataWord)stack.get(stack.size() - 2)), 0L, op);
                    this.checkMemorySize(op, VM.memNeeded((DataWord)stack.peek(), (DataWord)stack.get(stack.size() - 2)));
                    break;
                }
                case EXP: {
                    DataWord exp = (DataWord)stack.get(stack.size() - 2);
                    int bytesOccupied = exp.bytesOccupied();
                    energyCost = (long)energyCosts.getEXP_ENERGY() + (long)(energyCosts.getEXP_BYTE_ENERGY() * bytesOccupied);
                    break;
                }
            }
            program.spendEnergy(energyCost, op.name());
            program.checkCPUTimeLimit(op.name());
            switch (op) {
                case STOP: {
                    program.setHReturn(ByteUtil.EMPTY_BYTE_ARRAY);
                    program.stop();
                    break;
                }
                case ADD: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " + " + word2.value();
                    }
                    word1.add(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case MUL: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " * " + word2.value();
                    }
                    word1.mul(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case SUB: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " - " + word2.value();
                    }
                    word1.sub(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case DIV: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " / " + word2.value();
                    }
                    word1.div(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case SDIV: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.sValue() + " / " + word2.sValue();
                    }
                    word1.sDiv(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case MOD: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " % " + word2.value();
                    }
                    word1.mod(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case SMOD: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.sValue() + " #% " + word2.sValue();
                    }
                    word1.sMod(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case EXP: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " ** " + word2.value();
                    }
                    word1.exp(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case SIGNEXTEND: {
                    DataWord word1 = program.stackPop();
                    BigInteger k = word1.value();
                    if (k.compareTo(_32_) < 0) {
                        DataWord word2 = program.stackPop();
                        if (logger.isDebugEnabled()) {
                            hint = word1 + "  " + word2.value();
                        }
                        word2.signExtend(k.byteValue());
                        program.stackPush(word2);
                    }
                    program.step();
                    break;
                }
                case NOT: {
                    DataWord word1 = program.stackPop();
                    word1.bnot();
                    if (logger.isDebugEnabled()) {
                        hint = "" + word1.value();
                    }
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case LT: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " < " + word2.value();
                    }
                    if (word1.value().compareTo(word2.value()) < 0) {
                        word1.and(DataWord.ZERO);
                        word1.getData()[31] = 1;
                    } else {
                        word1.and(DataWord.ZERO);
                    }
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case SLT: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.sValue() + " < " + word2.sValue();
                    }
                    if (word1.sValue().compareTo(word2.sValue()) < 0) {
                        word1.and(DataWord.ZERO);
                        word1.getData()[31] = 1;
                    } else {
                        word1.and(DataWord.ZERO);
                    }
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case SGT: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.sValue() + " > " + word2.sValue();
                    }
                    if (word1.sValue().compareTo(word2.sValue()) > 0) {
                        word1.and(DataWord.ZERO);
                        word1.getData()[31] = 1;
                    } else {
                        word1.and(DataWord.ZERO);
                    }
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case GT: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " > " + word2.value();
                    }
                    if (word1.value().compareTo(word2.value()) > 0) {
                        word1.and(DataWord.ZERO);
                        word1.getData()[31] = 1;
                    } else {
                        word1.and(DataWord.ZERO);
                    }
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case EQ: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " == " + word2.value();
                    }
                    if (word1.xor(word2).isZero()) {
                        word1.and(DataWord.ZERO);
                        word1.getData()[31] = 1;
                    } else {
                        word1.and(DataWord.ZERO);
                    }
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case ISZERO: {
                    DataWord word1 = program.stackPop();
                    if (word1.isZero()) {
                        word1.getData()[31] = 1;
                    } else {
                        word1.and(DataWord.ZERO);
                    }
                    if (logger.isDebugEnabled()) {
                        hint = "" + word1.value();
                    }
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case AND: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " && " + word2.value();
                    }
                    word1.and(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case OR: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " || " + word2.value();
                    }
                    word1.or(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case XOR: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = word1.value() + " ^ " + word2.value();
                    }
                    word1.xor(word2);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case BYTE: {
                    DataWord result;
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    if (word1.value().compareTo(_32_) < 0) {
                        byte tmp = word2.getData()[word1.intValue()];
                        word2.and(DataWord.ZERO);
                        word2.getData()[31] = tmp;
                        result = word2;
                    } else {
                        result = new DataWord();
                    }
                    if (logger.isDebugEnabled()) {
                        hint = "" + result.value();
                    }
                    program.stackPush(result);
                    program.step();
                    break;
                }
                case ADDMOD: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    DataWord word3 = program.stackPop();
                    word1.addmod(word2, word3);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case MULMOD: {
                    DataWord word1 = program.stackPop();
                    DataWord word2 = program.stackPop();
                    DataWord word3 = program.stackPop();
                    word1.mulmod(word2, word3);
                    program.stackPush(word1);
                    program.step();
                    break;
                }
                case SHA3: {
                    DataWord memOffsetData = program.stackPop();
                    DataWord lengthData = program.stackPop();
                    byte[] buffer = program.memoryChunk(memOffsetData.intValueSafe(), lengthData.intValueSafe());
                    byte[] encoded = Hash.sha3(buffer);
                    DataWord word = new DataWord(encoded);
                    if (logger.isDebugEnabled()) {
                        hint = word.toString();
                    }
                    program.stackPush(word);
                    program.step();
                    break;
                }
                case ADDRESS: {
                    DataWord address = program.getContractAddress();
                    if (logger.isDebugEnabled()) {
                        hint = "address: " + Hex.toHexString((byte[])address.getLast20Bytes());
                    }
                    program.stackPush(address);
                    program.step();
                    break;
                }
                case BALANCE: {
                    DataWord address = program.stackPop();
                    DataWord balance = program.getBalance(address);
                    if (logger.isDebugEnabled()) {
                        hint = "address: " + Hex.toHexString((byte[])address.getLast20Bytes()) + " balance: " + balance.toString();
                    }
                    program.stackPush(balance);
                    program.step();
                    break;
                }
                case ORIGIN: {
                    DataWord originAddress = program.getOriginAddress();
                    if (logger.isDebugEnabled()) {
                        hint = "address: " + Hex.toHexString((byte[])originAddress.getLast20Bytes());
                    }
                    program.stackPush(originAddress);
                    program.step();
                    break;
                }
                case CALLER: {
                    DataWord callerAddress = program.getCallerAddress();
                    callerAddress = new DataWord(callerAddress.getLast20Bytes());
                    if (logger.isDebugEnabled()) {
                        hint = "address: " + Hex.toHexString((byte[])callerAddress.getLast20Bytes());
                    }
                    program.stackPush(callerAddress);
                    program.step();
                    break;
                }
                case CALLVALUE: {
                    DataWord callValue = program.getCallValue();
                    if (logger.isDebugEnabled()) {
                        hint = "value: " + callValue;
                    }
                    program.stackPush(callValue);
                    program.step();
                    break;
                }
                case CALLTOKENVALUE: {
                    DataWord tokenValue = program.getTokenValue();
                    if (logger.isDebugEnabled()) {
                        hint = "tokenValue: " + tokenValue;
                    }
                    program.stackPush(tokenValue);
                    program.step();
                    break;
                }
                case CALLTOKENID: {
                    DataWord _tokenId = program.getTokenId();
                    if (logger.isDebugEnabled()) {
                        hint = "tokenId: " + _tokenId;
                    }
                    program.stackPush(_tokenId);
                    program.step();
                    break;
                }
                case CALLDATALOAD: {
                    DataWord dataOffs = program.stackPop();
                    DataWord value = program.getDataValue(dataOffs);
                    if (logger.isDebugEnabled()) {
                        hint = "data: " + value;
                    }
                    program.stackPush(value);
                    program.step();
                    break;
                }
                case CALLDATASIZE: {
                    DataWord dataSize = program.getDataSize();
                    if (logger.isDebugEnabled()) {
                        hint = "size: " + dataSize.value();
                    }
                    program.stackPush(dataSize);
                    program.step();
                    break;
                }
                case CALLDATACOPY: {
                    DataWord memOffsetData = program.stackPop();
                    DataWord dataOffsetData = program.stackPop();
                    DataWord lengthData = program.stackPop();
                    byte[] msgData = program.getDataCopy(dataOffsetData, lengthData);
                    if (logger.isDebugEnabled()) {
                        hint = "data: " + Hex.toHexString((byte[])msgData);
                    }
                    program.memorySave(memOffsetData.intValueSafe(), msgData);
                    program.step();
                    break;
                }
                case RETURNDATASIZE: {
                    DataWord dataSize = program.getReturnDataBufferSize();
                    if (logger.isDebugEnabled()) {
                        hint = "size: " + dataSize.value();
                    }
                    program.stackPush(dataSize);
                    program.step();
                    break;
                }
                case RETURNDATACOPY: {
                    DataWord memOffsetData = program.stackPop();
                    DataWord dataOffsetData = program.stackPop();
                    DataWord lengthData = program.stackPop();
                    byte[] msgData = program.getReturnDataBufferData(dataOffsetData, lengthData);
                    if (msgData == null) {
                        throw new Program.ReturnDataCopyIllegalBoundsException(dataOffsetData, lengthData, program.getReturnDataBufferSize().longValueSafe());
                    }
                    if (logger.isDebugEnabled()) {
                        hint = "data: " + Hex.toHexString((byte[])msgData);
                    }
                    program.memorySave(memOffsetData.intValueSafe(), msgData);
                    program.step();
                    break;
                }
                case EXTCODESIZE: 
                case CODESIZE: {
                    DataWord address;
                    int length;
                    if (op == OpCode.CODESIZE) {
                        length = program.getCode().length;
                    } else {
                        address = program.stackPop();
                        length = program.getCodeAt(address).length;
                    }
                    DataWord codeLength = new DataWord(length);
                    if (logger.isDebugEnabled()) {
                        hint = "size: " + length;
                    }
                    program.stackPush(codeLength);
                    program.step();
                    break;
                }
                case CODECOPY: 
                case EXTCODECOPY: {
                    int lengthData;
                    DataWord address;
                    byte[] fullCode = ByteUtil.EMPTY_BYTE_ARRAY;
                    if (op == OpCode.CODECOPY) {
                        fullCode = program.getCode();
                    }
                    if (op == OpCode.EXTCODECOPY) {
                        address = program.stackPop();
                        fullCode = program.getCodeAt(address);
                    }
                    int memOffset = program.stackPop().intValueSafe();
                    int codeOffset = program.stackPop().intValueSafe();
                    int sizeToBeCopied = (long)codeOffset + (long)(lengthData = program.stackPop().intValueSafe()) > (long)fullCode.length ? (fullCode.length < codeOffset ? 0 : fullCode.length - codeOffset) : lengthData;
                    byte[] codeCopy = new byte[lengthData];
                    if (codeOffset < fullCode.length) {
                        System.arraycopy(fullCode, codeOffset, codeCopy, 0, sizeToBeCopied);
                    }
                    if (logger.isDebugEnabled()) {
                        hint = "code: " + Hex.toHexString((byte[])codeCopy);
                    }
                    program.memorySave(memOffset, codeCopy);
                    program.step();
                    break;
                }
                case GASPRICE: {
                    DataWord energyPrice = new DataWord(0);
                    if (logger.isDebugEnabled()) {
                        hint = "price: " + energyPrice.toString();
                    }
                    program.stackPush(energyPrice);
                    program.step();
                    break;
                }
                case BLOCKHASH: {
                    int blockIndex = program.stackPop().intValueSafe();
                    DataWord blockHash = program.getBlockHash(blockIndex);
                    if (logger.isDebugEnabled()) {
                        hint = "blockHash: " + blockHash;
                    }
                    program.stackPush(blockHash);
                    program.step();
                    break;
                }
                case COINBASE: {
                    DataWord coinbase = program.getCoinbase();
                    if (logger.isDebugEnabled()) {
                        hint = "coinbase: " + Hex.toHexString((byte[])coinbase.getLast20Bytes());
                    }
                    program.stackPush(coinbase);
                    program.step();
                    break;
                }
                case TIMESTAMP: {
                    DataWord timestamp = program.getTimestamp();
                    if (logger.isDebugEnabled()) {
                        hint = "timestamp: " + timestamp.value();
                    }
                    program.stackPush(timestamp);
                    program.step();
                    break;
                }
                case NUMBER: {
                    DataWord number = program.getNumber();
                    if (logger.isDebugEnabled()) {
                        hint = "number: " + number.value();
                    }
                    program.stackPush(number);
                    program.step();
                    break;
                }
                case DIFFICULTY: {
                    DataWord difficulty = program.getDifficulty();
                    if (logger.isDebugEnabled()) {
                        hint = "difficulty: " + difficulty;
                    }
                    program.stackPush(difficulty);
                    program.step();
                    break;
                }
                case GASLIMIT: {
                    DataWord energyLimit = new DataWord(0);
                    if (logger.isDebugEnabled()) {
                        hint = "energylimit: " + energyLimit;
                    }
                    program.stackPush(energyLimit);
                    program.step();
                    break;
                }
                case POP: {
                    program.stackPop();
                    program.step();
                    break;
                }
                case DUP1: 
                case DUP2: 
                case DUP3: 
                case DUP4: 
                case DUP5: 
                case DUP6: 
                case DUP7: 
                case DUP8: 
                case DUP9: 
                case DUP10: 
                case DUP11: 
                case DUP12: 
                case DUP13: 
                case DUP14: 
                case DUP15: 
                case DUP16: {
                    int n = op.val() - OpCode.DUP1.val() + 1;
                    DataWord word_1 = (DataWord)stack.get(stack.size() - n);
                    program.stackPush(word_1.clone());
                    program.step();
                    break;
                }
                case SWAP1: 
                case SWAP2: 
                case SWAP3: 
                case SWAP4: 
                case SWAP5: 
                case SWAP6: 
                case SWAP7: 
                case SWAP8: 
                case SWAP9: 
                case SWAP10: 
                case SWAP11: 
                case SWAP12: 
                case SWAP13: 
                case SWAP14: 
                case SWAP15: 
                case SWAP16: {
                    int n = op.val() - OpCode.SWAP1.val() + 2;
                    stack.swap(stack.size() - 1, stack.size() - n);
                    program.step();
                    break;
                }
                case LOG0: 
                case LOG1: 
                case LOG2: 
                case LOG3: 
                case LOG4: {
                    if (program.isStaticCall()) {
                        throw new Program.StaticCallModificationException();
                    }
                    DataWord address = program.getContractAddress();
                    DataWord memStart = stack.pop();
                    DataWord memOffset = stack.pop();
                    int nTopics = op.val() - OpCode.LOG0.val();
                    ArrayList<DataWord> topics = new ArrayList<DataWord>();
                    for (int i = 0; i < nTopics; ++i) {
                        DataWord topic = stack.pop();
                        topics.add(topic);
                    }
                    byte[] data = program.memoryChunk(memStart.intValueSafe(), memOffset.intValueSafe());
                    LogInfo logInfo = new LogInfo(address.getLast20Bytes(), topics, data);
                    if (logger.isDebugEnabled()) {
                        hint = logInfo.toString();
                    }
                    program.getResult().addLogInfo(logInfo);
                    program.step();
                    break;
                }
                case MLOAD: {
                    DataWord addr = program.stackPop();
                    Object data = program.memoryLoad(addr);
                    if (logger.isDebugEnabled()) {
                        hint = "data: " + data;
                    }
                    program.stackPush((DataWord)data);
                    program.step();
                    break;
                }
                case MSTORE: {
                    DataWord addr = program.stackPop();
                    DataWord value = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = "addr: " + addr + " value: " + value;
                    }
                    program.memorySave(addr, value);
                    program.step();
                    break;
                }
                case MSTORE8: {
                    DataWord addr = program.stackPop();
                    DataWord value = program.stackPop();
                    byte[] byteVal = new byte[]{value.getData()[31]};
                    program.memorySave(addr.intValueSafe(), byteVal);
                    program.step();
                    break;
                }
                case SLOAD: {
                    DataWord key = program.stackPop();
                    DataWord val = program.storageLoad(key);
                    if (logger.isDebugEnabled()) {
                        hint = "key: " + key + " value: " + val;
                    }
                    if (val == null) {
                        val = key.and(DataWord.ZERO);
                    }
                    program.stackPush(val);
                    program.step();
                    break;
                }
                case SSTORE: {
                    if (program.isStaticCall()) {
                        throw new Program.StaticCallModificationException();
                    }
                    DataWord addr = program.stackPop();
                    DataWord value = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = "[" + program.getContractAddress().toPrefixString() + "] key: " + addr + " value: " + value;
                    }
                    program.storageSave(addr, value);
                    program.step();
                    break;
                }
                case JUMP: {
                    DataWord pos = program.stackPop();
                    int nextPC = program.verifyJumpDest(pos);
                    if (logger.isDebugEnabled()) {
                        hint = "~> " + nextPC;
                    }
                    program.setPC(nextPC);
                    break;
                }
                case JUMPI: {
                    DataWord pos = program.stackPop();
                    DataWord cond = program.stackPop();
                    if (!cond.isZero()) {
                        int nextPC = program.verifyJumpDest(pos);
                        if (logger.isDebugEnabled()) {
                            hint = "~> " + nextPC;
                        }
                        program.setPC(nextPC);
                        break;
                    }
                    program.step();
                    break;
                }
                case PC: {
                    int pc = program.getPC();
                    DataWord pcWord = new DataWord(pc);
                    if (logger.isDebugEnabled()) {
                        hint = pcWord.toString();
                    }
                    program.stackPush(pcWord);
                    program.step();
                    break;
                }
                case MSIZE: {
                    int memSize = program.getMemSize();
                    DataWord wordMemSize = new DataWord(memSize);
                    if (logger.isDebugEnabled()) {
                        hint = "" + memSize;
                    }
                    program.stackPush(wordMemSize);
                    program.step();
                    break;
                }
                case GAS: {
                    DataWord energy = program.getEnergyLimitLeft();
                    if (logger.isDebugEnabled()) {
                        hint = "" + energy;
                    }
                    program.stackPush(energy);
                    program.step();
                    break;
                }
                case PUSH1: 
                case PUSH2: 
                case PUSH3: 
                case PUSH4: 
                case PUSH5: 
                case PUSH6: 
                case PUSH7: 
                case PUSH8: 
                case PUSH9: 
                case PUSH10: 
                case PUSH11: 
                case PUSH12: 
                case PUSH13: 
                case PUSH14: 
                case PUSH15: 
                case PUSH16: 
                case PUSH17: 
                case PUSH18: 
                case PUSH19: 
                case PUSH20: 
                case PUSH21: 
                case PUSH22: 
                case PUSH23: 
                case PUSH24: 
                case PUSH25: 
                case PUSH26: 
                case PUSH27: 
                case PUSH28: 
                case PUSH29: 
                case PUSH30: 
                case PUSH31: 
                case PUSH32: {
                    program.step();
                    int nPush = op.val() - OpCode.PUSH1.val() + 1;
                    Object data = program.sweep(nPush);
                    if (logger.isDebugEnabled()) {
                        hint = "" + Hex.toHexString((byte[])data);
                    }
                    program.stackPush((byte[])data);
                    break;
                }
                case JUMPDEST: {
                    program.step();
                    break;
                }
                case CREATE: {
                    if (program.isStaticCall()) {
                        throw new Program.StaticCallModificationException();
                    }
                    DataWord value = program.stackPop();
                    DataWord inOffset = program.stackPop();
                    DataWord inSize = program.stackPop();
                    program.createContract(value, inOffset, inSize);
                    program.step();
                    break;
                }
                case TOKENBALANCE: {
                    DataWord tokenId = program.stackPop();
                    DataWord address = program.stackPop();
                    DataWord tokenBalance = program.getTokenBalance(address, tokenId);
                    program.stackPush(tokenBalance);
                    program.step();
                    break;
                }
                case CALL: 
                case CALLCODE: 
                case DELEGATECALL: 
                case STATICCALL: 
                case CALLTOKEN: {
                    program.stackPop();
                    DataWord codeAddress = program.stackPop();
                    DataWord value = op.callHasValue() ? program.stackPop() : DataWord.ZERO;
                    if (program.isStaticCall() && (op == OpCode.CALL || op == OpCode.CALLTOKEN) && !value.isZero()) {
                        throw new Program.StaticCallModificationException();
                    }
                    if (!value.isZero()) {
                        adjustedCallEnergy.add(new DataWord(energyCosts.getSTIPEND_CALL()));
                    }
                    DataWord tokenId = new DataWord(0);
                    if (op == OpCode.CALLTOKEN) {
                        tokenId = program.stackPop();
                    }
                    DataWord inDataOffs = program.stackPop();
                    DataWord inDataSize = program.stackPop();
                    DataWord outDataOffs = program.stackPop();
                    DataWord outDataSize = program.stackPop();
                    if (logger.isDebugEnabled()) {
                        hint = "addr: " + Hex.toHexString((byte[])codeAddress.getLast20Bytes()) + " energy: " + adjustedCallEnergy.shortHex() + " inOff: " + inDataOffs.shortHex() + " inSize: " + inDataSize.shortHex();
                        logger.debug(ENERGY_LOG_FORMATE, new Object[]{String.format("%5s", "[" + program.getPC() + "]"), String.format("%-12s", op.name()), program.getEnergyLimitLeft().value(), program.getCallDeep(), hint});
                    }
                    program.memoryExpand(outDataOffs, outDataSize);
                    MessageCall msg = new MessageCall(op, adjustedCallEnergy, codeAddress, value, inDataOffs, inDataSize, outDataOffs, outDataSize, tokenId);
                    PrecompiledContracts.PrecompiledContract contract = PrecompiledContracts.getContractForAddress(codeAddress);
                    if (!op.callIsStateless()) {
                        program.getResult().addTouchAccount(codeAddress.getLast20Bytes());
                    }
                    if (contract != null) {
                        program.callToPrecompiledAddress(msg, contract);
                    } else {
                        program.callToAddress(msg);
                    }
                    program.step();
                    break;
                }
                case RETURN: 
                case REVERT: {
                    DataWord offset = program.stackPop();
                    size = program.stackPop();
                    byte[] hReturn = program.memoryChunk(offset.intValueSafe(), size.intValueSafe());
                    program.setHReturn(hReturn);
                    if (logger.isDebugEnabled()) {
                        hint = "data: " + Hex.toHexString((byte[])hReturn) + " offset: " + offset.value() + " size: " + size.value();
                    }
                    program.step();
                    program.stop();
                    if (op != OpCode.REVERT) break;
                    program.getResult().setRevert();
                    break;
                }
                case SUICIDE: {
                    if (program.isStaticCall()) {
                        throw new Program.StaticCallModificationException();
                    }
                    DataWord address = program.stackPop();
                    program.suicide(address);
                    program.getResult().addTouchAccount(address.getLast20Bytes());
                    if (logger.isDebugEnabled()) {
                        hint = "address: " + Hex.toHexString((byte[])program.getContractAddress().getLast20Bytes());
                    }
                    program.stop();
                    break;
                }
            }
            program.setPreviouslyExecutedOp(op.val());
        }
        catch (RuntimeException e) {
            logger.info("VM halted: [{}]", (Object)e.getMessage());
            program.spendAllEnergy();
            program.resetFutureRefund();
            program.stop();
            throw e;
        }
        finally {
            program.fullTrace();
        }
    }

    public void play(Program program) {
        try {
            if (program.byTestingSuite()) {
                return;
            }
            while (!program.isStopped()) {
                this.step(program);
            }
        }
        catch (Program.JVMStackOverFlowException | Program.OutOfTimeException e) {
            throw e;
        }
        catch (RuntimeException e) {
            if (StringUtils.isEmpty((Object)e.getMessage())) {
                program.setRuntimeFailure(new RuntimeException("Unknown Exception"));
            } else {
                program.setRuntimeFailure(e);
            }
        }
        catch (StackOverflowError soe) {
            logger.info("\n !!! StackOverflowError: update your java run command with -Xss !!!\n", (Throwable)soe);
            throw new Program.JVMStackOverFlowException();
        }
    }

    private boolean isDeadAccount(Program program, DataWord address) {
        return program.getContractState().getAccount(MUtil.convertToTronAddress(address.getLast20Bytes())) == null;
    }

    private static BigInteger memNeeded(DataWord offset, DataWord size) {
        return size.isZero() ? BigInteger.ZERO : offset.value().add(size.value());
    }
}

