/*
 * Decompiled with CFR 0.152.
 */
package com.github.triceo.robozonky.common.secrets;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableEntryException;
import java.security.cert.CertificateException;
import java.security.spec.InvalidKeySpecException;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KeyStoreHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(KeyStoreHandler.class);
    private static final String KEYSTORE_TYPE = "JCEKS";
    private static final String KEY_TYPE = "PBE";
    private final AtomicBoolean dirty;
    private final char[] password;
    private final File keyStoreFile;
    private final KeyStore keyStore;
    private final KeyStore.ProtectionParameter protectionParameter;
    private final SecretKeyFactory keyFactory;

    private static SecretKeyFactory getSecretKeyFactory() {
        try {
            return SecretKeyFactory.getInstance(KEY_TYPE);
        }
        catch (NoSuchAlgorithmException ex) {
            throw new IllegalStateException("Should not happen.", ex);
        }
    }

    public static KeyStoreHandler create(File keyStoreFile, char[] password) throws IOException, KeyStoreException {
        if (keyStoreFile == null) {
            throw new FileNotFoundException(null);
        }
        if (keyStoreFile.exists()) {
            throw new FileAlreadyExistsException(keyStoreFile.getAbsolutePath());
        }
        KeyStore ks = KeyStore.getInstance(KEYSTORE_TYPE);
        try {
            ks.load(null, password);
        }
        catch (NoSuchAlgorithmException | CertificateException ex) {
            throw new IllegalStateException("Should not happen.", ex);
        }
        SecretKeyFactory skf = KeyStoreHandler.getSecretKeyFactory();
        KeyStoreHandler ksh = new KeyStoreHandler(ks, password, keyStoreFile, skf, true);
        ksh.save();
        return ksh;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStoreHandler open(File keyStoreFile, char[] password) throws IOException, KeyStoreException {
        if (!keyStoreFile.exists()) {
            throw new FileNotFoundException(keyStoreFile.getAbsolutePath());
        }
        KeyStore ks = KeyStore.getInstance(KEYSTORE_TYPE);
        try (FileInputStream fis = new FileInputStream(keyStoreFile);){
            ks.load(fis, password);
            KeyStoreHandler keyStoreHandler = new KeyStoreHandler(ks, password, keyStoreFile, KeyStoreHandler.getSecretKeyFactory());
            return keyStoreHandler;
        }
        catch (NoSuchAlgorithmException | CertificateException ex) {
            throw new IllegalStateException("Should not happen.", ex);
        }
    }

    private KeyStoreHandler(KeyStore keyStore, char[] password, File keyStoreFile, SecretKeyFactory keyFactory) {
        this(keyStore, password, keyStoreFile, keyFactory, false);
    }

    private KeyStoreHandler(KeyStore keyStore, char[] password, File keyStoreFile, SecretKeyFactory keyFactory, boolean isDirty) {
        this.keyStore = keyStore;
        this.password = password;
        this.protectionParameter = new KeyStore.PasswordProtection("NO_PASSWORD".toCharArray());
        this.keyStoreFile = keyStoreFile;
        this.keyFactory = keyFactory;
        this.dirty = new AtomicBoolean(isDirty);
    }

    public boolean set(String alias, char[] value) {
        try {
            SecretKey secret = this.keyFactory.generateSecret(new PBEKeySpec(value));
            KeyStore.SecretKeyEntry skEntry = new KeyStore.SecretKeyEntry(secret);
            this.keyStore.setEntry(alias, skEntry, this.protectionParameter);
            this.dirty.set(true);
            return true;
        }
        catch (KeyStoreException | InvalidKeySpecException ex) {
            LOGGER.debug("Failed storing '{}'.", (Object)alias, (Object)ex);
            return false;
        }
    }

    public Optional<char[]> get(String alias) {
        try {
            KeyStore.SecretKeyEntry skEntry = (KeyStore.SecretKeyEntry)this.keyStore.getEntry(alias, this.protectionParameter);
            if (skEntry == null) {
                return Optional.empty();
            }
            PBEKeySpec keySpec = (PBEKeySpec)this.keyFactory.getKeySpec(skEntry.getSecretKey(), PBEKeySpec.class);
            return Optional.of(keySpec.getPassword());
        }
        catch (KeyStoreException | NoSuchAlgorithmException | InvalidKeySpecException ex) {
            throw new IllegalStateException("Should not happen.", ex);
        }
        catch (UnrecoverableEntryException ex) {
            LOGGER.debug("Unrecoverable entry '{}'.", (Object)alias, (Object)ex);
            return Optional.empty();
        }
    }

    public boolean delete(String alias) {
        try {
            this.keyStore.deleteEntry(alias);
            this.dirty.set(true);
            return true;
        }
        catch (KeyStoreException ex) {
            LOGGER.debug("Entry '{}' not deleted.", (Object)alias, (Object)ex);
            return false;
        }
    }

    public boolean isDirty() {
        return this.dirty.get();
    }

    public void save() throws IOException {
        try (BufferedOutputStream os = new BufferedOutputStream(new FileOutputStream(this.keyStoreFile));){
            try {
                this.keyStore.store(os, this.password);
                this.dirty.set(false);
            }
            catch (KeyStoreException | NoSuchAlgorithmException | CertificateException ex) {
                throw new IllegalStateException("Should not happen.", ex);
            }
        }
    }
}

