package com.proximities.sdk;

import android.app.Activity;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.os.SystemClock;
import android.text.Html;

import androidx.work.ExistingPeriodicWorkPolicy;
import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;
import androidx.work.WorkRequest;

import com.google.android.gms.maps.model.LatLng;
import com.proximities.sdk.GeofenceBroadcastReceiver;
import com.proximities.sdk.ProximitiesConfig;

import com.proximities.sdk.ProximitiesWorker;
import com.proximities.sdk.R;
import com.proximities.sdk.interfaces.FavoriteActionsInterface;
import com.proximities.sdk.interfaces.GetFavoritesInterface;
import com.proximities.sdk.interfaces.LoyaltyInterface;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.concurrent.TimeUnit;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 19/11/14.
 */
public abstract class Utils {

    private static final String WORK_TAG = "prxsc_worker_tag";

    public static String getCurrentLanguage(Context ctx) {
        return ctx.getResources().getConfiguration().locale.getLanguage();
    }

    public static String getAppId(Context ctx) {
        return getMetaData(ctx).getString("proximities:appId");
    }

    public static String getAppSecret(Context ctx) {
        return getMetaData(ctx).getString("proximities:appSecret");
    }

    /**
     * Get application meta-data of a package name.
     *
     * @param context application context.
     * @return meta-data, may be empty but never null.
     */
    public static Bundle getMetaData(Context context) {
        Bundle config;
        try {
            config = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA).metaData;
            if (config == null) {
                config = new Bundle();
            }
        } catch (Exception e) {
            /*
             * NameNotFoundException or in some rare scenario an undocumented "RuntimeException: Package
             * manager has died.", probably caused by a system app process crash.
             */
            config = new Bundle();
        }
        return config;
    }

    public static boolean isColorDark(int color){
        double darkness = 1-(0.299* Color.red(color) + 0.587*Color.green(color) + 0.114*Color.blue(color))/255;
        return darkness >= 0.1;
    }

    public static void endCampaignDisplay(Activity activity, boolean isMultiCampaign){
        Intent intent = null;
        try {
            if(ProximitiesConfig.getInstance() != null && ProximitiesConfig.getInstance().isUsingCurrentActivity()){
                intent = new Intent(activity.getApplicationContext(), Class.forName(ProximitiesConfig.getInstance().getCurrentActivity()));
                intent.setFlags(Intent.FLAG_ACTIVITY_REORDER_TO_FRONT);
            } else {
                intent = new Intent(activity.getApplicationContext(), Class.forName(ProximitiesConfig.getMainActivity(activity.getApplicationContext())));
            }
        } catch (ClassNotFoundException e) {
            e.printStackTrace();
        }
        if(intent != null){
            activity.startActivity(intent);
        }
        activity.finish();
        if(!isMultiCampaign){
            activity.overridePendingTransition(R.anim.null_animation, R.anim.zoom_out);
        }
    }

    static String getCurrentDate(){
        Calendar calendar = Calendar.getInstance();
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        return format.format(calendar.getTime());
    }

    static double computeDistanceBetween(double latitudeFrom, double longitudeFrom, double latitudeTo, double longitudeTo) {
        return computeAngleBetween(latitudeFrom, longitudeFrom, latitudeTo, longitudeTo) * 6371009.0D;
    }

    private static double computeAngleBetween(double latitudeFrom, double longitudeFrom, double latitudeTo, double longitudeTo) {
        return distanceRadians(Math.toRadians(latitudeFrom), Math.toRadians(longitudeFrom), Math.toRadians(latitudeTo), Math.toRadians(longitudeTo));
    }

    private static double distanceRadians(double lat1, double lng1, double lat2, double lng2) {
        return arcHav(havDistance(lat1, lat2, lng1 - lng2));
    }

    private static double arcHav(double x) {
        return 2.0D * Math.asin(Math.sqrt(x));
    }

    private static double havDistance(double lat1, double lat2, double dLng) {
        return hav(lat1 - lat2) + hav(dLng) * Math.cos(lat1) * Math.cos(lat2);
    }

    private static double hav(double x) {
        double sinHalf = Math.sin(x * 0.5D);
        return sinHalf * sinHalf;
    }

    static String fromHtml(String source){
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            return Html.fromHtml(source, Html.FROM_HTML_MODE_LEGACY).toString();
        } else {
            return Html.fromHtml(source).toString();
        }
    }

    static void startPeriodicJob(Context context){
        PeriodicWorkRequest request = new PeriodicWorkRequest.Builder(ProximitiesWorker.class,
                PeriodicWorkRequest.MIN_PERIODIC_INTERVAL_MILLIS,
                TimeUnit.MILLISECONDS,
                PeriodicWorkRequest.MIN_PERIODIC_FLEX_MILLIS,
                TimeUnit.MILLISECONDS).build();
        WorkManager.getInstance(context).enqueueUniquePeriodicWork(WORK_TAG, ExistingPeriodicWorkPolicy.KEEP, request);
    }

    public static void refreshLoyalty(LoyaltyInterface callback, int campaignId){
        ProximitiesNetworkManager.getInstance().refreshLoyalty(callback, campaignId);
    }

    public static void saveCampaign(FavoriteActionsInterface callback, int campaignId){
        ProximitiesNetworkManager.getInstance().saveCampaign(callback, campaignId);
    }

    public static void deleteCampaign(FavoriteActionsInterface callback, int campaignId){
        ProximitiesNetworkManager.getInstance().deleteCampaign(callback, campaignId);
    }

    public static void getFavorites(GetFavoritesInterface callback){
        ProximitiesNetworkManager.getInstance().getFavorites(callback);
    }
}
