package com.proximities.sdk.util;

import android.content.Context;
import android.content.SharedPreferences;

import com.google.gson.Gson;
import com.proximities.sdk.json.model.transmitter.Transmitter;

import java.util.ArrayList;
import java.util.List;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 07/12/14.
 */
public abstract class ProximitiesPrefs {

    private static final String PREF_NAME = "sweepin_prefs";

    private static final String START_BEACON_MANAGER   = "start_beacon_manager";
    private static final String START_LOCATION_MANAGER = "start_location_manager";
    private static final String SERVICE_IS_STOPPED = "service_is_stopped";

    private static final String USER_EMAIl_KEY = "user_email";
    private static final String ANONYMOUS_ID_KEY = "anonymous_id";
    private static final String ENABLE_SQLITE = "enable_sqlite";
    private static final String CURRENT_LAT_KEY = "curr_lat";
    private static final String CURRENT_LNG_KEY = "curr_lng";
    private static final String CATEGORIES_FILTERED_ID = "categories_filtered_id";
    private static final String CATEGORIES_FILTERED_NAME = "categories_filtered_name";
    private static final String DISABLE_ANIM = "disable_anim";
    private static final String DISABLE_FAVORITES = "disable_fav";
    private static final String APP_FIRST_LAUNCH = "app_first_launch";
    private static final String CURRENT_ACTIVITY = "current_activity";
    private static final String USE_CURRENT_ACTIVITY = "use_current_activity";
    private static final String MAIN_ACTIVITY = "main_activity";
    private static final String SHARE_FRAGMENT = "share_fragment";

    private static final String NOTIFICATION_ICON = "notification_icon";
    private static final String NOTIFICATION_CONTENT = "notification_content";
    private static final String NOTIFICATION_LIGHT = "notification_light";
    private static final String BACKGROUND_SCAN_PERIOD = "background_scan_period";
    private static final String BACKGROUND_BETWEEN_SCAN_PERIOD = "background_between_scan_period";
    private static final String FOREGROUND_SCAN_PERIOD = "foreground_scan_period";
    private static final String FOREGROUND_BETWEEN_SCAN_PERIOD = "foreground_between_scan_period";
    private static final String LAST_BEACON_DETECTED = "last_beacon_detected";
    private static final String LAST_TRANSMITTER_DETECTED = "last_transmitter_detected";
    private static final String LIST_UUIDS = "list_uuids";
    private static final String DETECT_ONLY_CLOSEST_BEACON = "detect_only_closest_beacon";


    public static boolean readStartBeaconManager(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(START_BEACON_MANAGER, false);
    }

    public static void writeStartBeaconManager(final Context c, boolean startBeaconManager) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(START_BEACON_MANAGER, startBeaconManager);
        editor.apply();
    }

    public static boolean readStartLocationManager(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(START_LOCATION_MANAGER, false);
    }

    public static void writeStartLocationManager(final Context c, boolean startLocationManager) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(START_LOCATION_MANAGER, startLocationManager);
        editor.apply();
    }

    public static boolean readServiceIsStopped(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(SERVICE_IS_STOPPED, false);
    }

    public static void writeServiceIsStopped(final Context c, boolean isServiceStopped) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(SERVICE_IS_STOPPED, isServiceStopped);
        editor.apply();
    }

    /**
     * Save that the application has been launched for the first time
     *
     * @param c     context of the application
     */
    public static void writeAppFirstLaunch(final Context c, String firstLauch) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(APP_FIRST_LAUNCH, firstLauch);
        editor.apply();
    }

    /**
     * Retrieve if it is the first launch of the application
     *
     * @param c context of the application
     */
    public static String readAppFirstLaunch(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(APP_FIRST_LAUNCH, "");
    }
    /**
     * Save the email of the user
     *
     * @param c     context of the application
     * @param email email as string to save
     */
    public static void writeUserEmail(final Context c, String email) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(USER_EMAIl_KEY, email);
        editor.apply();
    }

    /**
     * Retrieve the email of the user
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static String readUserEmail(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(USER_EMAIl_KEY, "");
    }

    /**
     * Just like {@link #readUserEmail} but for anonymous user
     *
     * @param c           context of the application
     * @param anonymousId anonymous identifier as string to save
     */
    public static void writeAnonymousId(final Context c, String anonymousId) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(ANONYMOUS_ID_KEY, anonymousId);
        editor.apply();
    }

    /**
     * Retrieve the email of the user
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static String readAnonymousId(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(ANONYMOUS_ID_KEY, "");
    }

    public static boolean readEnableSqlite(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(ENABLE_SQLITE, true);
    }

    public static void writeEnableSqlite(final Context c, boolean enableSqlite) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(ENABLE_SQLITE, enableSqlite);
        editor.apply();
    }

    public static boolean readDetectOnlyClosestBeacon(final Context c){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(DETECT_ONLY_CLOSEST_BEACON, false);
    }

    public static void writeDetectOnlyClosestBeacon(final Context c, boolean detectOnlyClosestBeacon){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(DETECT_ONLY_CLOSEST_BEACON, detectOnlyClosestBeacon);
        editor.apply();
    }

    /**
     * Just like {@link #readUserEmail} but in order to keep latitude
     *
     * @param c   context of the application
     * @param lat latitude of the user position
     */
    public static void writeCurrentLat(final Context c, float lat) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putFloat(CURRENT_LAT_KEY, lat);
        editor.apply();
    }

    /**
     * Retrieve user position
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static float readCurrentLat(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getFloat(CURRENT_LAT_KEY, 0.0f);
    }

    /**
     * Just like {@link #readUserEmail} but in order to keep longitude
     *
     * @param c   context of the application
     * @param lng longitude of the user position
     */
    public static void writeCurrentLng(final Context c, float lng) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putFloat(CURRENT_LNG_KEY, lng);
        editor.apply();
    }

    /**
     * Retrieve user position
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static float readCurrentLng(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getFloat(CURRENT_LNG_KEY, 0.0f);
    }

    /**
     * Just like {@link #writeCurrentLat(Context, float)}
     * plus {@link #writeCurrentLng(Context, float)}
     *
     * @param c   context of the application
     * @param lat latitude of the user position
     * @param lng longitude of the user position
     */
    public static void writeCurrentLatAndLng(final Context c, float lat, float lng) {
        if(c != null){
            final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
            final SharedPreferences.Editor editor = settings.edit();
            editor.putFloat(CURRENT_LAT_KEY, lat);
            editor.putFloat(CURRENT_LNG_KEY, lng);
            editor.apply();
        }
    }

    /**
     * Retrieve current Activity
     *
     * @param c context of the application
     * @return String containing current activity's name
     */
    public static String readCurrentActivity(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(CURRENT_ACTIVITY, "");
    }

    /**
     * Save current Activity
     *
     * @param c context of the application
     * @param currentActivity current activity's name
     *
     */
    public static void writeCurrentActivity(final Context c, String currentActivity){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(CURRENT_ACTIVITY, currentActivity);
        editor.apply();
    }

    /**
     * Tell us if sdk user retrieves current activity
     *
     * @param c context of the application
     * @return true if user retrieves current activity
     */
    public static boolean readKeepCurrentActivity(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(USE_CURRENT_ACTIVITY, false);
    }

    /**
     * Save sdk user's choice to retrieve current activity
     *
     * @param c context of the application
     * @param useCurrentActivity choose to retrieve current activity or not
     */
    public static void writeKeepCurrentActivity(final Context c, boolean useCurrentActivity) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(USE_CURRENT_ACTIVITY, useCurrentActivity);
        editor.apply();
    }

    /**
     * Retrieve main Activity
     *
     * @param c context of the application
     * @return String containing main activity's name
     */
    public static String readMainActivity(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(MAIN_ACTIVITY, "");
    }

    /**
     * Save main Activity
     *
     * @param c context of the application
     * @param mainActivity main activity's name
     *
     */
    public static void writeMainActivity(final Context c, String mainActivity){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(MAIN_ACTIVITY, mainActivity);
        editor.apply();
    }

    /**
     * Retrieve share fragment
     *
     * @param c context of the application
     * @return String containing share fragment's name
     */
    public static String readShareFragment(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(SHARE_FRAGMENT, "");

    }

    /**
     * Save share fragment
     *
     * @param c context of the application
     * @param shareFragment share fragment's name
     *
     */
    public static void writeShareFragment(final Context c, String shareFragment){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(SHARE_FRAGMENT, shareFragment);
        editor.apply();
    }

    // ----------------------------------------------------------------------
    // ----------------------------------------------------------------------

    public static boolean readDisableAnim(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(DISABLE_ANIM, false);
    }

    public static void writeDisableAnim(final Context c, boolean disableAnim) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(DISABLE_ANIM, disableAnim);
        editor.apply();
    }

    public static boolean readDisableFavorites(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(DISABLE_FAVORITES, false);
    }

    public static void writeDisableFavorites(final Context c, boolean disableFav) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(DISABLE_FAVORITES, disableFav);
        editor.apply();
    }

    /**
     * Retrieve the duration of the scan in milliseconds to detect beacons when the app is in foreground
     *
     * @param c context of the application
     * @return duration of the scan
     */
    public static int readForegroundScanPeriod(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(FOREGROUND_SCAN_PERIOD, 1100);
    }

    /**
     * Retrieve the duration between each Bluetooth LE scan cycle to detect beacons when the app is in foreground
     *
     * @param c context of the application
     * @return duration between the scan
     */
    public static int readForegroundBetweenScanPeriod(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(FOREGROUND_BETWEEN_SCAN_PERIOD, 0);
    }

    /**
     * Set the duration of the scan in milliseconds and the duration between each Bluetooth LE scan cycle to look for beacons (app in foreground)
     *
     * @param c   context of the application
     * @param foregroundScanPeriod duration of the scan
     * @param foregroundBetweenScanPeriod duration between each scan
     *
     */
    public static void writeForegroundScanPeriod(final Context c, int foregroundScanPeriod, int foregroundBetweenScanPeriod) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putInt(FOREGROUND_SCAN_PERIOD, foregroundScanPeriod);
        editor.putInt(FOREGROUND_BETWEEN_SCAN_PERIOD, foregroundBetweenScanPeriod);
        editor.apply();
    }

    /**
     * Retrieve the duration of the scan in milliseconds to detect beacons when the app is in background
     *
     * @param c context of the application
     * @return duration of the scan
     */
    public static int readBackgroundScanPeriod(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(BACKGROUND_SCAN_PERIOD, 5000);
    }

    /**
     * Retrieve the duration between each Bluetooth LE scan cycle to detect beacons when the app is in background
     *
     * @param c context of the application
     * @return duration between the scan
     */
    public static int readBackgroundBetweenScanPeriod(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(BACKGROUND_BETWEEN_SCAN_PERIOD, 25000);
    }

    /**
     * Set the duration of the scan in milliseconds and the duration between each Bluetooth LE scan cycle to look for beacons (app in background)
     *
     * @param backgroundScanPeriod duration of the scan
     * @param backgroundBetweenScanPeriod duration between each scan
     */
    public static void writeBackgroundScanPeriod(final Context c, int backgroundScanPeriod, int backgroundBetweenScanPeriod) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putInt(BACKGROUND_SCAN_PERIOD, backgroundScanPeriod);
        editor.putInt(BACKGROUND_BETWEEN_SCAN_PERIOD, backgroundBetweenScanPeriod);
        editor.apply();
    }

    /**
     * Retrieve the last beacon detected
     *
     * @param c context of the application
     * @return uuid-major-minor from the last beacon detected
     */
    public static String readLastBeaconDetected(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(LAST_BEACON_DETECTED, "");
    }

    /**
     * Set the the last beacon detected
     *
     * @param beacon uuid-major-minor from the last beacon detected
     */
    public static void writeLastBeaconDetected(final Context c, String beacon ) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(LAST_BEACON_DETECTED, beacon);
        editor.apply();
    }

    /**
     * Retrieve the last beacon detected
     *
     * @param c context of the application
     * @return transmitter from the last beacon detected
     */
    public static Transmitter readLastTransmitterDetected(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        Gson gson = new Gson();
        String json = settings.getString(LAST_TRANSMITTER_DETECTED, "");
        return gson.fromJson(json, Transmitter.class);
    }

    /**
     * Set the the last beacon detected
     *
     * @param transmitter transmitter from the last beacon detected
     */
    public static void writeLastTransmitterDetected(final Context c, Transmitter transmitter) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        Gson gson = new Gson();
        String json = gson.toJson(transmitter);
        editor.putString(LAST_TRANSMITTER_DETECTED, json);
        editor.apply();
    }

    /**
     * Set the uuids to add to the beacon scanner
     *
     * @param listOfUuids
     */
    public static void writeUuidsList(final Context c, List<String> listOfUuids ) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        List<String> list = new ArrayList<String>();
        list.addAll(listOfUuids);
        editor.putString(LIST_UUIDS, new Gson().toJson(list));
        editor.apply();
    }

    /**
     * Get the list of uuids added
     *
     * @return list of uuids
     */
    public static String[] readUuidsList(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        String text = settings.getString(LIST_UUIDS, null);
        return new Gson().fromJson(text, String[].class);
    }


}
