package com.proximities.sdk.corekit;

import android.content.Context;
import android.os.RemoteException;

import com.proximities.sdk.corekit.interfaces.IPSBeaconServicesInterface;
import com.proximities.sdk.corekit.interfaces.PSBeaconServicesInterface;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.BeaconManager;
import org.altbeacon.beacon.BeaconParser;
import org.altbeacon.beacon.MonitorNotifier;
import org.altbeacon.beacon.RangeNotifier;
import org.altbeacon.beacon.Region;
import org.altbeacon.beacon.powersave.BackgroundPowerSaver;

import java.util.Collection;
import java.util.List;

/**
 * Created by william on 01/12/16.
 */

public class CKBeaconManager {

    private static CKBeaconManager sharedInstance;

    public static CKBeaconManager getSharedInstance(Context context){
        if(sharedInstance == null){
            sharedInstance = new CKBeaconManager(context);
        }
        return sharedInstance;
    }

    private static final String CUSTOM_IBEACON_LAYOUT = "m:2-3=0215,i:4-19,i:20-21,i:22-23,p:24-24";
    private static final String CUSTOM_EDDYSTONE_LAYOUT = "s:0-1=feaa,m:2-2=00,p:3-3:-41,i:4-13,i:14-19";

    private BeaconManager beaconManager;
    private BackgroundPowerSaver backgroundPowerSaver;
    private PSBeaconServicesInterface mPSBeaconCallback;
    private IPSBeaconServicesInterface mIPSBeaconCallback;
    private boolean isBeaconManagerRunning = false;

    public CKBeaconManager(Context context){
        beaconManager = BeaconManager.getInstanceForApplication(context);
        this.beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(CUSTOM_EDDYSTONE_LAYOUT));
        this.beaconManager.getBeaconParsers().add(new BeaconParser().setBeaconLayout(CUSTOM_IBEACON_LAYOUT));
        backgroundPowerSaver = new BackgroundPowerSaver(context);
    }

    public void bindBeaconManager(BeaconConsumer beaconConsumer){
        beaconManager.bind(beaconConsumer);
    }

    public void unbindBeaconManager(BeaconConsumer beaconConsumer){
        beaconManager.unbind(beaconConsumer);
    }

    public void startBeaconServices(){
        if(!isBeaconManagerRunning){
            isBeaconManagerRunning = true;
            beaconManager.addMonitorNotifier(new MonitorNotifier() {
                @Override
                public void didEnterRegion(Region region) {
                    if(mPSBeaconCallback != null) mPSBeaconCallback.didEnterRegion(region);
                    if(mIPSBeaconCallback != null) mIPSBeaconCallback.didEnterRegion(region);
                }

                @Override
                public void didExitRegion(Region region) {
                    if(mPSBeaconCallback != null) mPSBeaconCallback.didExitRegion(region);
                    if(mIPSBeaconCallback != null) mIPSBeaconCallback.didExitRegion(region);
                }

                @Override
                public void didDetermineStateForRegion(int i, Region region) {
                    if(mPSBeaconCallback != null) mPSBeaconCallback.didDetermineStateForRegion(i, region);
                    if(mIPSBeaconCallback != null) mIPSBeaconCallback.didDetermineStateForRegion(i, region);
                }
            });
            beaconManager.addRangeNotifier(new RangeNotifier() {
                @Override
                public void didRangeBeaconsInRegion(Collection<Beacon> collection, Region region) {
                    if(mPSBeaconCallback != null) mPSBeaconCallback.didRangeBeaconsInRegion(collection, region);
                    if(mIPSBeaconCallback != null) mIPSBeaconCallback.didRangeBeaconsInRegion(collection, region);
                }
            });
        }
    }

    public void startMonitoringInRegion(List<Region> beaconRegions){
        for (Region r : beaconRegions) {
            try {
                beaconManager.startMonitoringBeaconsInRegion(r);
            } catch (RemoteException e) {
//                e.printStackTrace();
            }
        }
    }

    public void startRangingInRegion(List<Region> beaconRegions){
        for (Region r : beaconRegions) {
            try {
                beaconManager.startRangingBeaconsInRegion(r);
            } catch (RemoteException e) {
//                e.printStackTrace();
            }
        }
    }

    /**
     * Set the duration of the scan in milliseconds and the duration between each Bluetooth LE scan cycle to look for beacons (app in foreground)
     *
     * @param scanPeriodMillis time in milliseconds
     * @param waitTimeMillis time in milliseconds
     */
    public void setForegroundScanPeriod(long scanPeriodMillis, long waitTimeMillis) {
        beaconManager.setForegroundScanPeriod(scanPeriodMillis);
        beaconManager.setForegroundBetweenScanPeriod(waitTimeMillis);
        try {
            beaconManager.updateScanPeriods();
        } catch (RemoteException e) {
//            Log.e("ForegroundScan", "Scan period can't be updated");
        }
    }

    /**
     * Set the duration of the scan in milliseconds and the duration between each Bluetooth LE scan cycle to look for beacons (app in background)
     *
     * @param scanPeriodMillis time in milliseconds
     * @param waitTimeMillis time in milliseconds
     */
    public void setBackgroundScanPeriod(long scanPeriodMillis, long waitTimeMillis) {
        beaconManager.setBackgroundScanPeriod(scanPeriodMillis);
        beaconManager.setBackgroundBetweenScanPeriod(waitTimeMillis);
        try {
            beaconManager.updateScanPeriods();
        } catch (RemoteException e) {
//            Log.e("BackgroundScan", "scan period can't be updated");
        }
    }

    public boolean isBeaconManagerRunning() {
        return isBeaconManagerRunning;
    }

    public void attachPSBeaconInterface(PSBeaconServicesInterface callback){
        this.mPSBeaconCallback = callback;
    }

    public void attachIPSBeaconInterface(IPSBeaconServicesInterface callback){
        this.mIPSBeaconCallback = callback;
    }
}
