package com.proximities.sdk;

import android.Manifest;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.support.v4.content.ContextCompat;

import com.proximities.sdk.corekit.CKBeaconManager;
import com.proximities.sdk.corekit.interfaces.PSBeaconServicesInterface;
import com.proximities.sdk.interfaces.TransmitterInterface;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.json.model.transmitter_logs.TransmitterLog;
import com.proximities.sdk.json.model.transmitter_logs.TransmittersLogs;
import com.proximities.sdk.request.api.EntryExitLogRequest;
import com.proximities.sdk.request.api.TransmitterRequest;
import com.proximities.sdk.util.DetectedBeacon;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;
import com.proximities.sdk.util.Utils;

import org.altbeacon.beacon.Beacon;
import org.altbeacon.beacon.BeaconConsumer;
import org.altbeacon.beacon.Identifier;
import org.altbeacon.beacon.Region;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static com.proximities.sdk.util.LogUtils.LOGD;
import static com.proximities.sdk.util.LogUtils.LOGE;
import static com.proximities.sdk.util.LogUtils.makeLogTag;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 27/12/14.
 */
class ProximitiesBeaconManager implements PSBeaconServicesInterface, TransmitterInterface{

    private static final String TAG = makeLogTag(ProximitiesBeaconManager.class);

    private static final ProximitiesBeaconManager ourInstance = new ProximitiesBeaconManager();

    private static final String DEFAULT_UUID_ONE = "F46EFD73-819B-4766-AB45-CAE5A282CD59";
    private static final String DEFAULT_UUID_TWO = "12E4CE63-E9D1-40E9-A529-096DE63E504E";
    private static final String DEFAULT_NAMESPACE_ONE = "EDD1EBEAC04E5DEFA017";
    private static final long DURATION_BETWEEN_TRANSMITTER_REQUEST = 30000; //in milliseconds

    private Context context;
    private DatabaseManager dbManager;
    private CKBeaconManager coreKitBeaconManager;
    private List<DetectedBeacon> mDetectedBeacons;
    private List<Region> regions;
    private TransmitterRequest mTransmitterRequest;
    private EntryExitLogRequest mEntryExitLogRequest;
    private Beacon closestBeacon;
    private boolean isClosestBeaconModOn;

    private long timestamp;

    public static ProximitiesBeaconManager getInstance() {
        return ourInstance;
    }

    private ProximitiesBeaconManager() {}

    void initBeaconManager(Context ctx, BeaconConsumer consumer) {
        this.context = ctx;
        closestBeacon = null;
        isClosestBeaconModOn = ProximitiesPrefs.readDetectOnlyClosestBeacon(context);
        this.mDetectedBeacons = new ArrayList<>();
        if(ProximitiesPrefs.readEnableSqlite(context)){
            dbManager = DatabaseManager.getInstance();
        }
        coreKitBeaconManager = CKBeaconManager.getSharedInstance(ctx);
        coreKitBeaconManager.attachPSBeaconInterface(this);
        initRegions();
        mTransmitterRequest = new TransmitterRequest(context, this);
        mEntryExitLogRequest = new EntryExitLogRequest(context);
        coreKitBeaconManager.bindBeaconManager(consumer);
    }

    /**
     * Make a list with default regions
     */
    private void initRegions() {
        if(regions == null) regions = new ArrayList<>();
        else regions.clear();
        regions.add(new Region(DEFAULT_UUID_ONE, Identifier.parse(DEFAULT_UUID_ONE), null, null));
        regions.add(new Region(DEFAULT_UUID_TWO, Identifier.parse(DEFAULT_UUID_TWO), null, null));
        regions.add(new Region(DEFAULT_NAMESPACE_ONE, Identifier.parse(DEFAULT_NAMESPACE_ONE), null, null));

        if(ProximitiesPrefs.readUuidsList(context) != null && ProximitiesPrefs.readUuidsList(context).length > 0) {
            for(String uuid : ProximitiesPrefs.readUuidsList(context)){
                try {
                    regions.add(new Region(uuid, Identifier.parse(uuid), null, null));
                } catch (IllegalArgumentException exception){
                    LOGE(TAG, "Unable to add the uuid : " + uuid + ". Wrong format");
                }
            }
        }
    }

    void startServicesWhenReady(){
        coreKitBeaconManager.setBackgroundScanPeriod(ProximitiesPrefs.readBackgroundScanPeriod(context),ProximitiesPrefs.readBackgroundBetweenScanPeriod(context));
        coreKitBeaconManager.setForegroundScanPeriod(ProximitiesPrefs.readForegroundScanPeriod(context),ProximitiesPrefs.readForegroundBetweenScanPeriod(context));
        coreKitBeaconManager.startBeaconServices();
        coreKitBeaconManager.startMonitoringInRegion(regions);
        coreKitBeaconManager.startRangingInRegion(regions);
    }

    void unbindBeaconManager(BeaconConsumer beaconConsumer){
        if(coreKitBeaconManager != null){
            coreKitBeaconManager.unbindBeaconManager(beaconConsumer);
            if(ProximitiesConfig.isEntryExitLogEnabled() && ProximitiesPrefs.readLastTransmitterDetected(context) != null) {
                sendEntryExitLog(ProximitiesPrefs.readLastTransmitterDetected(context), ProximitiesConstants.ACTION_EXIT_LOG);
                ProximitiesPrefs.writeLastTransmitterDetected(context, null);
            }
        }
    }

    @Override
    public void onGetBeaconContent(Transmitter transmitter) {
        if(!ProximitiesPrefs.readDisableAnim(context)) {
            ArrayList<Campaign> campaigns = (ArrayList<Campaign>) transmitter.getCampaigns();
            if(campaigns.size() > 0) {
                removeDetectedBeaconWithCampaigns(transmitter);
                for(Campaign campaign : campaigns){
                    campaign.setPoi(transmitter.getPoi().get(0));
                }
                PSManager.getInstance().startCampaignsDisplay(campaigns, null, String.valueOf(transmitter.getId()));
            }
        }
    }

    @Override
    public void onGetBeaconError(Transmitter transmitter) {
        if(dbManager != null){
            switch(transmitter.getType()){
                case ProximitiesConstants.TYPE_EDDYSTONE:
                    dbManager.startCampaignFromDatabase(null, null, null, transmitter.getNamespace(), transmitter.getInstance());
                    break;
                case ProximitiesConstants.TYPE_IBEACON:
                    dbManager.startCampaignFromDatabase(transmitter.getUuid(), transmitter.getMajor(), transmitter.getMinor(), null, null);
                    break;
            }
        }
        removeAllDetectedBeacons();
    }

    @Override
    public void didEnterRegion(Region region) {
        LOGD(TAG, "I just saw a beacon for the first time! ");
    }

    @Override
    public void didExitRegion(Region region) {
        stopEntryExitLog();
        LOGD(TAG, "I no longer see a beacon");
    }

    @Override
    public void didDetermineStateForRegion(int i, Region region) {
        LOGD(TAG, "I have just switched from seeing/not seeing beacons: " + i);
    }

    @Override
    public void didRangeBeaconsInRegion(Collection<Beacon> beacons, Region region) {
        if (ContextCompat.checkSelfPermission(context, Manifest.permission.ACCESS_FINE_LOCATION) != PackageManager.PERMISSION_GRANTED) {
            Intent i = new Intent(context, ProximitiesService.class);
            context.stopService(i);
        } else {
            if (ProximitiesConfig.getOnScannedBeaconsListener() != null){
                ProximitiesConfig.getOnScannedBeaconsListener().onScannedBeacons(beacons);
            }
            requestBeaconsWithTimestamp(beacons);
        }
    }

    private void requestBeaconsWithTimestamp(Collection<Beacon> beacons){
        timestamp = System.currentTimeMillis();
        for (Beacon beacon : beacons) {
            if(isClosestBeaconModOn){
                if(closestBeacon == null || closestBeacon.getIdentifiers() == beacon.getIdentifiers() || closestBeacon.getDistance() > beacon.getDistance()){
                    closestBeacon = beacon;
                }
            } else {
                handleBeacon(beacon, beacons.size());
            }
        }

        if(isClosestBeaconModOn && closestBeacon != null){
            if(!beacons.contains(closestBeacon)){
                closestBeacon = null; // reset closest beacon if the current one is not detected anymore
            } else {
                handleBeacon(closestBeacon, beacons.size());
            }
        }
    }

    private void handleBeacon(Beacon beacon, int beaconListSize){
        Transmitter transmitter = new Transmitter();
        if (beacon.getServiceUuid() == 0xfeaa) {
            //is an Eddystone Beacon
            transmitter.setType(ProximitiesConstants.TYPE_EDDYSTONE);
            transmitter.setNamespace(beacon.getId1().toString().substring(2));
            transmitter.setInstance(beacon.getId2().toString().substring(2));

        } else {
            //is a ibeacon
            transmitter.setType(ProximitiesConstants.TYPE_IBEACON);
            transmitter.setUuid(beacon.getId1().toString());
            transmitter.setMajor(beacon.getId2().toString());
            transmitter.setMinor(beacon.getId3().toString());
        }

        if(ProximitiesConfig.isEntryExitLogEnabled() && beaconListSize == 1) {
            if(ProximitiesPrefs.readLastTransmitterDetected(context) == null || !areTransmittersEquals(transmitter, ProximitiesPrefs.readLastTransmitterDetected(context))) {
                if (ProximitiesPrefs.readLastTransmitterDetected(context) != null) {
                    sendEntryExitLog(ProximitiesPrefs.readLastTransmitterDetected(context), ProximitiesConstants.ACTION_EXIT_LOG);
                }
                ProximitiesPrefs.writeLastTransmitterDetected(context, transmitter);
                sendEntryExitLog(transmitter, ProximitiesConstants.ACTION_ENTRY_LOG);
            }
        }

        // this code limits the number of transmitter requests
        if(mDetectedBeacons != null && mDetectedBeacons.size() > 0){
            boolean isBeaconAlreadyDetected = false;
            for(int i = 0; i < mDetectedBeacons.size() ; i++){
                DetectedBeacon detectedBeacon = mDetectedBeacons.get(i);
                if(areTransmittersEquals(detectedBeacon.getTransmitter(), transmitter)){
                    isBeaconAlreadyDetected = true;
                    if(timestamp - detectedBeacon.getTimestamp() >= DURATION_BETWEEN_TRANSMITTER_REQUEST){
                        // if the last transmitter request on this beacon was done more than x seconds ago (x is the duration constant set up above)
                        mDetectedBeacons.get(i).setTimestamp(timestamp);
                        sendTransmitterRequest(transmitter);
                    }
                    break;
                }
            }
            if(!isBeaconAlreadyDetected){
                mDetectedBeacons.add(new DetectedBeacon(transmitter, timestamp));
                sendTransmitterRequest(transmitter);
            }
        } else {
            //if no beacon has been detected before
            mDetectedBeacons.add(new DetectedBeacon(transmitter, timestamp));
            sendTransmitterRequest(transmitter);
        }
    }

    private void sendTransmitterRequest(Transmitter transmitter){
        if(transmitter.getType().equals(ProximitiesConstants.TYPE_EDDYSTONE)){
            mTransmitterRequest.getEddystoneContent(transmitter);
        } else if(transmitter.getType().equals(ProximitiesConstants.TYPE_IBEACON)){
            mTransmitterRequest.getIBeaconContent(transmitter);
        }
    }

    private void sendEntryExitLog(Transmitter transmitter, String action){
        TransmittersLogs logs = new TransmittersLogs();
        logs.addLog(new TransmitterLog(transmitter, Utils.getCurrentDate(), action));
        mEntryExitLogRequest.sendTransmittersLogs(logs);
    }

    /*
       If a transmitter is linked to a campaign the user didn't see yet,
       the beacon is removed from the detected beacon list until the campaign is displayed.
     */
    private void removeDetectedBeaconWithCampaigns(Transmitter transmitter){
        for(DetectedBeacon detectedBeacon : mDetectedBeacons) {
            if(areTransmittersEquals(detectedBeacon.getTransmitter(), transmitter)){
                detectedBeacon.reduceTimestampBy(DURATION_BETWEEN_TRANSMITTER_REQUEST);
                break;
            }
        }
    }

    private void removeAllDetectedBeacons(){
        if(mDetectedBeacons != null){
            mDetectedBeacons.clear();
        }
    }

    private boolean areTransmittersEquals(Transmitter transmitter1, Transmitter transmitter2){
        if(transmitter1 == null || transmitter2 == null){
            return false;
        } else if(!transmitter1.getType().equals(transmitter2.getType())){
            return false;
        } else if(transmitter1.getType().equals(ProximitiesConstants.TYPE_EDDYSTONE)
                && transmitter1.getNamespace().equalsIgnoreCase(transmitter2.getNamespace())
                && transmitter1.getInstance().equalsIgnoreCase(transmitter2.getInstance())){
            return true;
        } else if(transmitter1.getType().equals(ProximitiesConstants.TYPE_IBEACON)
                && transmitter1.getUuid().equalsIgnoreCase(transmitter2.getUuid())
                && transmitter1.getMajor().equals(transmitter2.getMajor())
                && transmitter1.getMinor().equals(transmitter2.getMinor())){
            return true;
        } else {
            return false;
        }
    }

    private void stopEntryExitLog(){
        if(ProximitiesConfig.isEntryExitLogEnabled() && ProximitiesPrefs.readLastTransmitterDetected(context) != null) {
            sendEntryExitLog(ProximitiesPrefs.readLastTransmitterDetected(context), ProximitiesConstants.ACTION_EXIT_LOG);
            ProximitiesPrefs.writeLastTransmitterDetected(context, null);
        }
    }

}
