package com.proximities.sdk.database;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.graphics.Bitmap;
import android.util.Base64;

import com.bumptech.glide.Glide;
import com.bumptech.glide.request.animation.GlideAnimation;
import com.bumptech.glide.request.target.SimpleTarget;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;
import com.proximities.sdk.bridge.OnProximitiesService;
import com.proximities.sdk.json.model.transmitter.Campaign;
import com.proximities.sdk.json.model.transmitter.Transmitter;
import com.proximities.sdk.util.ProximitiesConstants;

import java.io.ByteArrayOutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;

/**
 * Created by william on 10/08/15.
 */
public class DatabaseHandler extends SQLiteOpenHelper {

    private Context context;
    public static final int DATABASE_VERSION = 2;
    public static final String DATABASE_NAME = "proximities_animations.db";
    private Campaign animToInsert;

    public static final String ANIMATIONS_TABLE_CREATE =

            "CREATE TABLE " + AnimationsTable.AnimationsEntry.TABLE_NAME + " (" +
                    AnimationsTable.AnimationsEntry.COLUMN_KEY + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    AnimationsTable.AnimationsEntry.COLUMN_BEACON_UUID + " TEXT, " +
                    AnimationsTable.AnimationsEntry.COLUMN_BEACON_MINOR + " INTEGER, "+
                    AnimationsTable.AnimationsEntry.COLUMN_BEACON_MAJOR + " INTEGER, " +
                    AnimationsTable.AnimationsEntry.COLUMN_BEACON_NAMESPACE + " TEXT, " +
                    AnimationsTable.AnimationsEntry.COLUMN_BEACON_INSTANCE + " TEXT, " +
                    AnimationsTable.AnimationsEntry.COLUMN_ANIMATION_OBJECT + " BLOB);";

    public static final String ANIMATIONS_TABLE_DROP = "DROP TABLE IF EXISTS " + AnimationsTable.AnimationsEntry.TABLE_NAME + ";";

    public static final String ANIMATIONS_LOGS_TABLE_CREATE =
            "CREATE TABLE " + LogsTable.LogsEntry.TABLE_NAME + " (" +
                    LogsTable.LogsEntry.COLUMN_KEY + " INTEGER PRIMARY KEY AUTOINCREMENT, " +
                    LogsTable.LogsEntry.COLUMN_ANIM_ID + " INTEGER, " +
                    LogsTable.LogsEntry.COLUMN_CREATED + " TEXT, "+
                    LogsTable.LogsEntry.COLUMN_ACTION + " TEXT);";

    public static final String ANIMATIONS_LOGS_TABLE_DROP = "DROP TABLE IF EXISTS " + LogsTable.LogsEntry.TABLE_NAME + ";";

    public DatabaseHandler(Context context) {
        super(context, DATABASE_NAME, null, DATABASE_VERSION);
        this.context = context;
    }

    @Override

    public void onCreate(SQLiteDatabase db) {
        db.execSQL(ANIMATIONS_TABLE_CREATE);
        db.execSQL(ANIMATIONS_LOGS_TABLE_CREATE);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        db.execSQL(ANIMATIONS_TABLE_DROP);
        db.execSQL(ANIMATIONS_LOGS_TABLE_DROP);
        onCreate(db);
    }

    public void insertLogsIntoDatabase(int animId, String action){
        SQLiteDatabase db = this.getWritableDatabase();
        ContentValues values = new ContentValues();
        values.put(LogsTable.LogsEntry.COLUMN_ANIM_ID, animId);
        Calendar calendar = Calendar.getInstance();
        SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        values.put(LogsTable.LogsEntry.COLUMN_CREATED, format.format(calendar.getTime()));
        values.put(LogsTable.LogsEntry.COLUMN_ACTION, action);
        db.insert(LogsTable.LogsEntry.TABLE_NAME, null, values);
        db.close();
    }

    public void insertAnimationsInDatabase(Transmitter trans, SQLiteDatabase db) {
        ContentValues values = new ContentValues();
        if (trans.getUuid() != null && !trans.getUuid().isEmpty()){
            values.put(AnimationsTable.AnimationsEntry.COLUMN_BEACON_UUID, trans.getUuid());
            values.put(AnimationsTable.AnimationsEntry.COLUMN_BEACON_MINOR, trans.getMinor());
            values.put(AnimationsTable.AnimationsEntry.COLUMN_BEACON_MAJOR, trans.getMajor());
        } else if (trans.getNamespace() != null && !trans.getNamespace().isEmpty()){
            values.put(AnimationsTable.AnimationsEntry.COLUMN_BEACON_NAMESPACE, trans.getNamespace());
            values.put(AnimationsTable.AnimationsEntry.COLUMN_BEACON_INSTANCE, trans.getInstance());
        }
        animToInsert = trans.getCampaigns().get(0);
        Glide.with(context).load(ProximitiesConstants.STATIC_CONTENT_HOST + animToInsert.getBanner())
                .asBitmap()
                .into(new SimpleTarget<Bitmap>() {
                    @Override
                    public void onResourceReady(Bitmap resource, GlideAnimation glideAnimation) {
                        ByteArrayOutputStream bao = new ByteArrayOutputStream();
                        resource.compress(Bitmap.CompressFormat.JPEG, 30, bao);
                        animToInsert.setBannerOffline(Base64.encodeToString(bao.toByteArray(), Base64.DEFAULT));
                    }
                });
        Glide.with(context).load(ProximitiesConstants.STATIC_CONTENT_HOST + animToInsert.getImage())
                .asBitmap()
                .into(new SimpleTarget<Bitmap>() {
                    @Override
                    public void onResourceReady(Bitmap resource, GlideAnimation glideAnimation) {
                        ByteArrayOutputStream bao = new ByteArrayOutputStream();
                        resource.compress(Bitmap.CompressFormat.JPEG, 30, bao);
                        animToInsert.setImageOffline(Base64.encodeToString(bao.toByteArray(), Base64.DEFAULT));
                    }
                });
        if(trans.getPoi() != null && trans.getPoi().size() > 0)
            animToInsert.setPoi(trans.getPoi().get(0));
        values.put(AnimationsTable.AnimationsEntry.COLUMN_ANIMATION_OBJECT, new Gson().toJson((ArrayList) trans.getCampaigns()).getBytes());
        db.insert(AnimationsTable.AnimationsEntry.TABLE_NAME, null, values);
    }

    public boolean isAnimationInLogs(int animId, SQLiteDatabase db){
        String query = "SELECT * FROM " + LogsTable.LogsEntry.TABLE_NAME +
                " WHERE " + LogsTable.LogsEntry.COLUMN_ANIM_ID + " LIKE '%" + animId + "%';";

        Cursor cursor = db.rawQuery(query, null);
        if(cursor.getCount() > 0) {
            cursor.close();
            return true;
        } else {
            cursor.close();
            return false;
        }
    }

    public void startAnimFromDatabase(String uuid, String minor, String major, String namespace, String instance, OnProximitiesService onProximitiesService){
        SQLiteDatabase db = this.getReadableDatabase();
        String query="";
        if (uuid != null && !uuid.isEmpty()) {
            query = "SELECT * FROM " + AnimationsTable.AnimationsEntry.TABLE_NAME +
                    " WHERE " + AnimationsTable.AnimationsEntry.COLUMN_BEACON_UUID + " LIKE '%" + uuid + "%' AND "
                    + AnimationsTable.AnimationsEntry.COLUMN_BEACON_MINOR + " LIKE '%" + minor + "%' AND "
                    + AnimationsTable.AnimationsEntry.COLUMN_BEACON_MAJOR + " LIKE '%" + major + "%'" +
                    " ORDER BY " + AnimationsTable.AnimationsEntry.COLUMN_KEY + " COLLATE NOCASE ASC;";
        }
        if (namespace != null && !namespace.isEmpty()){
            query = "SELECT * FROM " + AnimationsTable.AnimationsEntry.TABLE_NAME +
                    " WHERE " + AnimationsTable.AnimationsEntry.COLUMN_BEACON_NAMESPACE + " LIKE '%" + namespace+ "%' AND "
                    + AnimationsTable.AnimationsEntry.COLUMN_BEACON_INSTANCE + " LIKE '%" + instance + "%'" +
                    " ORDER BY " + AnimationsTable.AnimationsEntry.COLUMN_KEY + " COLLATE NOCASE ASC;";
        }

        Cursor cursor = db.rawQuery(query, null);
        if(cursor.getCount() > 0) {
            cursor.moveToFirst();
            byte[] animationBlob = cursor.getBlob(cursor.getColumnIndex(AnimationsTable.AnimationsEntry.COLUMN_ANIMATION_OBJECT));
            cursor.close();
            ArrayList<Campaign> campaigns = new Gson().fromJson(new String(animationBlob), new TypeToken<ArrayList<Campaign>>() {
            }.getType());
            if(!this.isAnimationInLogs(campaigns.get(0).getId(), db))
                onProximitiesService.startAnim(campaigns);
        }
        db.close();
    }
}