package com.proximities.sdk;

import android.Manifest;
import android.app.Application;
import android.app.PendingIntent;
import android.content.pm.PackageManager;
import android.location.Location;
import android.os.Bundle;
import android.support.v4.content.ContextCompat;
import android.util.Log;

import com.proximities.sdk.bridge.OnEventUserMoved;
import com.proximities.sdk.bridge.OnProximitiesService;
import com.proximities.sdk.request.api.ClosestPoiRequest;
import com.proximities.sdk.request.api.PartnerRequest;
import com.proximities.sdk.util.LocationChange;
import com.proximities.sdk.util.ProximitiesPrefs;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GooglePlayServicesUtil;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.common.api.PendingResult;
import com.google.android.gms.common.api.Status;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;
import org.greenrobot.eventbus.EventBus;

import static com.proximities.sdk.util.LogUtils.LOGD;
import static com.proximities.sdk.util.LogUtils.makeLogTag;

/**
 * Created by william on 05/04/15.
 */
public class ProximitiesGpsManager implements
        GoogleApiClient.ConnectionCallbacks,
        GoogleApiClient.OnConnectionFailedListener,
        LocationListener {

    private Application app;
    private static final String TAG = makeLogTag(ProximitiesGpsManager.class);
    private static final int MY_PERMISSIONS_REQUEST_ACCESS_FINE_LOCATION = 11111;

    private static final ProximitiesGpsManager ourInstance = new ProximitiesGpsManager();

    // Take a look at the following link
    // http://davistechyinfo.blogspot.fr/2014/09/android-location-updates-with.html
    private static final int POLLING_FREQ = 1000 * 30;
    private static final int FASTEST_UPDATE_FREQ = 1000 * 10;
    private static final int POLLING_FREQ_AWAY = 1000 * 60 * 5;
    private static final int FASTEST_UPDATE_FREQ_AWAY = 1000 * 60;
    private static final int THRESHOLD_DISTANCE_FREQ = 15;
    private static final int SMALLEST_DISPLACEMENT = 200;

    private static final int MIN_DIST_TO_CALL_WS = 30;
    private static final float MIN_LAST_READ_ACCURACY = 500.0f;
    private static final int ONE_MIN = 1000 * 60;
    private static final int TIME_TO_STOP_GPS_POLLING = 10000;
    private static final int GEOFENCE_RADIUS = 150;
    // time in milliseconds

    private static int mPollingFreq = POLLING_FREQ;
    private static int mFastestUpdateFreq = FASTEST_UPDATE_FREQ;
    private static int defaultPriority = LocationRequest.PRIORITY_BALANCED_POWER_ACCURACY;

    private GoogleApiClient googleApiClient;
    private LocationRequest locationRequest;
    private PendingResult<Status> result;
    private Location currentLocation;
    private boolean isRequested = false;
    private float distanceClosestPoi;
    private boolean isFarAway = false;
    private int coefFrequency = 0;

    private OnProximitiesService callback;

    private OnEventUserMoved callbackUser;

    private static final float MIN_ACCURACY = 25.0f;

    /*
     * Define a request code to send to Google Play services
     * This code is returned in Activity.onActivityResult
     */
    private final static int CONNECTION_FAILURE_RESOLUTION_REQUEST = 9000;

    public static ProximitiesGpsManager getInstance() {
        return ourInstance;
    }

    private ProximitiesGpsManager() {}

    public void initLocationManager(Application app, OnProximitiesService callback, OnEventUserMoved callbackUser) {
        this.callback = callback;
        this.callbackUser = callbackUser;

        if (googleApiClient == null) {
            this.app = app;
            googleApiClient = new GoogleApiClient.Builder(app)
                    .addApi(LocationServices.API)
                    .addConnectionCallbacks(this)
                    .addOnConnectionFailedListener(this)
                    .build();

            locationRequest = LocationRequest.create();
            locationRequest.setPriority(defaultPriority);
            createLocationRequest(SMALLEST_DISPLACEMENT, mPollingFreq, mFastestUpdateFreq);
            googleApiClient.connect();
        }
    }

    private void createLocationRequest(float smallestDisplacement, long interval, long fastestInterval){
        locationRequest.setInterval(interval);
        locationRequest.setFastestInterval(fastestInterval);
        locationRequest.setSmallestDisplacement(smallestDisplacement);
    }

    public void setLocationPriority(int priority){
        if(locationRequest != null) {
            locationRequest.setPriority(priority);
            startPolling();
        } else {
            defaultPriority = priority;
        }
    }

    /**
     * Explicitly set the fastest interval for location updates, in milliseconds.
     *
     * @param millis fastest interval for updates in milliseconds, exact
     */
    public void setFastestInterval(int millis) {
        mFastestUpdateFreq = millis;
    }

    /**
     * Set the desired interval for active location updates, in milliseconds.
     *
     * The location client will actively try to obtain location updates for your application at this interval, so it has
     * a direct influence on the amount of power used by your application. Choose your interval wisely.
     *
     * @param millis desired interval in millisecond, inexact
     */
    public void setInterval(int millis) {
        mPollingFreq = millis;
    }

    public boolean isConnected() {
        return googleApiClient != null && googleApiClient.isConnected();
    }

    public void removeLocationUpdates(){
        if(googleApiClient != null && googleApiClient.isConnected()){
            LocationServices.FusedLocationApi.removeLocationUpdates(googleApiClient, this);
            googleApiClient.disconnect();
        }
    }

    @Override
    public void onConnected(Bundle bundle) {
        LOGD(TAG, "Connected to play services");
        callback.onGpsReadyWithRealCoord(currentLocation != null);
        if(ContextCompat.checkSelfPermission(app, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED)
            LocationServices.FusedLocationApi.requestLocationUpdates(googleApiClient, locationRequest, this);
    }

    @Override
    public void onConnectionSuspended(int i) {
    }

    @Override
    public void onLocationChanged(Location location) {
        LOGD(TAG, "onLocationChanged");
        if(currentLocation == null || location.distanceTo(currentLocation) > GEOFENCE_RADIUS || isRequested) {
            if(isRequested) {
                isRequested = false;
            }
            currentLocation = location;
            ProximitiesPrefs.writeCurrentLatAndLng(app, (float) location.getLatitude(), (float) location.getLongitude());
            EventBus.getDefault().post(new LocationChange());
            executeRequest();
            getClosestPoi();
        }
    }

    public void updateLocationRequest(float distanceClosestPoi) {
        if(ContextCompat.checkSelfPermission(app, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED) {
            this.distanceClosestPoi = distanceClosestPoi;
            if (distanceClosestPoi > THRESHOLD_DISTANCE_FREQ) {
                int coef = Math.round(distanceClosestPoi) / THRESHOLD_DISTANCE_FREQ;
                if (coef != coefFrequency) {
                    coefFrequency = coef;
                    LocationServices.FusedLocationApi.removeLocationUpdates(googleApiClient, this);
                    createLocationRequest(0, coef * POLLING_FREQ_AWAY, coef * FASTEST_UPDATE_FREQ_AWAY);
                    isFarAway = true;
                    LocationServices.FusedLocationApi.requestLocationUpdates(googleApiClient, locationRequest, this);
                }
            } else if (distanceClosestPoi < THRESHOLD_DISTANCE_FREQ && isFarAway) {
                LocationServices.FusedLocationApi.removeLocationUpdates(googleApiClient, this);
                createLocationRequest(SMALLEST_DISPLACEMENT, mPollingFreq, mFastestUpdateFreq);
                isFarAway = false;
                LocationServices.FusedLocationApi.requestLocationUpdates(googleApiClient, locationRequest, this);
            }
        }
    }

    /**
     * Enable polling to retrieve periodically user position
     */
    public void startPolling() {
        if (isConnected()) {
            if(currentLocation != null){
                if(ContextCompat.checkSelfPermission(app, Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED){
                    isRequested = true;
                    LocationServices.FusedLocationApi.removeLocationUpdates(googleApiClient, this);
                    LocationServices.FusedLocationApi.requestLocationUpdates(googleApiClient, locationRequest, this);
                }
            }
        }
    }

    @Override
    public void onConnectionFailed(ConnectionResult connectionResult) { }

    private void executeRequest(){
        PartnerRequest partnerRequest = PartnerRequest.getInstance(app);
        partnerRequest.executeGet(ProximitiesPrefs.readCurrentLat(app), ProximitiesPrefs.readCurrentLng(app), true);
    }

    private void getClosestPoi(){
        ClosestPoiRequest closestPoiRequest = ClosestPoiRequest.getInstance(app);
        closestPoiRequest.executeGetForClosestPoi(currentLocation.getLatitude(), currentLocation.getLongitude());
    }

}
