package com.proximities.sdk.util;

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;

import com.google.gson.Gson;
import com.proximities.sdk.ProximitiesConfig;
import com.proximities.sdk.R;

/**
 * Created by Antoine Arnoult <arnoult.antoine@gmail.com> on 07/12/14.
 */
public abstract class ProximitiesPrefs {

    private static final String PREF_NAME = "sweepin_prefs";

    private static final String USER_EMAIl_KEY = "user_email";
    private static final String ANONYMOUS_ID_KEY = "anonymous_id";
    private static final String PASSWORD_KEY = "pass";
    private static final String USER_IMAGE_URL = "user_image_url";
    private static final String USER_COVER_URL = "user_cover_url";
    private static final String ID_KEY = "id";
    private static final String CURRENT_LAT_KEY = "curr_lat";
    private static final String CURRENT_LNG_KEY = "curr_lng";
    private static final String ASK_BLE_AGAIN = "ask_ble";
    private static final String CATEGORIES_FILTERED_ID = "categories_filtered_id";
    private static final String CATEGORIES_FILTERED_NAME = "categories_filtered_name";
    private static final String DISABLE_ANIM = "disable_anim";
    private static final String DISABLE_FAVORITES = "disable_fav";
    private static final String APP_FIRST_LAUNCH = "app_first_launch";
    private static final String CURRENT_ACTIVITY = "current_activity";
    private static final String USE_CURRENT_ACTIVITY = "use_current_activity";
    private static final String MAIN_ACTIVITY = "main_activity";
    private static final String SHARE_FRAGMENT = "share_fragment";
    private static final String TIME_VIDEO = "time_video";

    private static final String NOTIFICATION_ICON = "notification_icon";
    private static final String NOTIFICATION_CONTENT = "notification_content";
    private static final String NOTIFICATION_LIGHT = "notification_light";
    private static final String BACKGROUND_SCAN_PERIOD = "background_scan_period";
    private static final String BACKGROUND_BETWEEN_SCAN_PERIOD = "background_between_scan_period";
    private static final String FOREGROUND_SCAN_PERIOD = "foreground_scan_period";
    private static final String FOREGROUND_BETWEEN_SCAN_PERIOD = "foreground_between_scan_period";
    private static final String LAST_BEACON_DETECTED = "last_beacon_detected";

    /**
     * Save progress of the video
     *
     * @param c  context of the application
     * @param timeVideo progress time
     */
    public static void writeTimeVideo(final Context c, long timeVideo) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putLong(TIME_VIDEO, timeVideo);
        editor.apply();
    }

    /**
     * Retrieve progress of the video
     *
     * @param c context of the application
     * @return timeVideo progress time
     */
    public static long readTimeVideo(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getLong(TIME_VIDEO, 0);
    }

    /**
     * Save that the application has been launched for the first time
     *
     * @param c     context of the application
     */
    public static void writeAppFirstLaunch(final Context c, String firstLauch) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(APP_FIRST_LAUNCH, firstLauch);
        editor.apply();
    }

    /**
     * Retrieve if it is the first launch of the application
     *
     * @param c context of the application
     */
    public static String readAppFirstLaunch(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(APP_FIRST_LAUNCH, "");
    }
    /**
     * Save the email of the user
     *
     * @param c     context of the application
     * @param email email as string to save
     */
    public static void writeUserEmail(final Context c, String email) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(USER_EMAIl_KEY, email);
        editor.apply();
    }

    /**
     * Retrieve the email of the user
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static String readUserEmail(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(USER_EMAIl_KEY, "");
    }

    /**
     * Just like {@link #readUserEmail} but for anonymous user
     *
     * @param c           context of the application
     * @param anonymousId anonymous identifier as string to save
     */
    public static void writeAnonymousId(final Context c, String anonymousId) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(ANONYMOUS_ID_KEY, anonymousId);
        editor.apply();
    }

    /**
     * Retrieve the email of the user
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static String readAnonymousId(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(ANONYMOUS_ID_KEY, "");
    }

    /**
     * Just like {@link #readUserEmail} but in order to keep password
     *
     * @param c        context of the application
     * @param password user's password as string to save
     */
    public static void writePassword(final Context c, String password) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(PASSWORD_KEY, password);
        editor.apply();
    }

    /**
     * Retrieve user's password
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static String readPassword(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(PASSWORD_KEY, "");
    }

    /**
     * Just like {@link #readUserEmail} but in order to keep password
     *
     * @param c            context of the application
     * @param userImageUrl user's image url as string to save
     */
    public static void writeUserImageUrl(final Context c, String userImageUrl) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(USER_IMAGE_URL, userImageUrl);
        editor.apply();
    }

    /**
     * Retrieve user's image url
     *
     * @param c context of the application
     * @return user image url or empty
     */
    public static String readUserImageUrl(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(USER_IMAGE_URL, "");
    }

    /**
     * Just like {@link #readUserEmail} but in order to keep password
     *
     * @param c            context of the application
     * @param userCoverUrl user's image url as string to save
     */
    public static void writeUserCoverUrl(final Context c, String userCoverUrl) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(USER_COVER_URL, userCoverUrl);
        editor.apply();
    }

    /**
     * Retrieve user's image url
     *
     * @param c context of the application
     * @return user cover url or empty
     */
    public static String readUserCoverUrl(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(USER_COVER_URL, "");
    }

    /**
     * Just like {@link #readUserId} but in order to keep password
     *
     * @param c  context of the application
     * @param id user's id as long to save
     */
    public static void writeUserId(final Context c, long id) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putLong(ID_KEY, id);
        editor.apply();
    }

    /**
     * Retrieve user's id
     *
     * @param c context of the application
     * @return id user or empty
     */
    public static long readUserId(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getLong(ID_KEY, 0);
    }

    /**
     * Just like {@link #readUserEmail} but in order to keep latitude
     *
     * @param c   context of the application
     * @param lat latitude of the user position
     */
    public static void writeCurrentLat(final Context c, float lat) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putFloat(CURRENT_LAT_KEY, lat);
        editor.apply();
    }

    /**
     * Retrieve user position
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static float readCurrentLat(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getFloat(CURRENT_LAT_KEY, 0.0f);
    }

    /**
     * Just like {@link #readUserEmail} but in order to keep longitude
     *
     * @param c   context of the application
     * @param lng longitude of the user position
     */
    public static void writeCurrentLng(final Context c, float lng) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putFloat(CURRENT_LNG_KEY, lng);
        editor.apply();
    }

    /**
     * Retrieve user position
     *
     * @param c context of the application
     * @return email user or empty
     */
    public static float readCurrentLng(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getFloat(CURRENT_LNG_KEY, 0.0f);
    }

    /**
     * Just like {@link #writeCurrentLat(Context, float)}
     * plus {@link #writeCurrentLng(Context, float)}
     *
     * @param c   context of the application
     * @param lat latitude of the user position
     * @param lng longitude of the user position
     */
    public static void writeCurrentLatAndLng(final Context c, float lat, float lng) {
        if(c != null){
            final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
            final SharedPreferences.Editor editor = settings.edit();
            editor.putFloat(CURRENT_LAT_KEY, lat);
            editor.putFloat(CURRENT_LNG_KEY, lng);
            editor.apply();
        }
    }

    /**
     * Tell us if user has checked the option "Do not ask me again"
     * About Bluetooth activation
     *
     * @param c context of the application
     * @return true if user has checked the option, false otherwise
     */
    public static boolean readAskBleAgain(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(ASK_BLE_AGAIN, true);
    }

    /**
     * Save user's choice about asking again if he wants
     * to activate bluetooth
     *
     * @param c context of the application
     */
    public static void writeAskBleAgain(final Context c, boolean askBleAgain) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(ASK_BLE_AGAIN, askBleAgain);
        editor.apply();
    }

    /**
     * Retrieve categories filtered by the user in the search field
     *
     * @param c context of the application
     * @return String containing categories filtered by the user
     */
    public static String readCategories(final Context c, boolean forHeader) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        if(forHeader)
            return settings.getString(CATEGORIES_FILTERED_ID, "");
        else
            return settings.getString(CATEGORIES_FILTERED_NAME, "");
    }

    /**
     * Save categories that the user has filtered in the search field
     *
     * @param c context of the application
     */
    public static void writeCategories(final Context c, String categoriesFilteredIds,String categoriesFilteredNames){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(CATEGORIES_FILTERED_ID, categoriesFilteredIds);
        editor.putString(CATEGORIES_FILTERED_NAME, categoriesFilteredNames);
        editor.apply();
    }

    /**
     * Retrieve current Activity
     *
     * @param c context of the application
     * @return String containing current activity's name
     */
    public static String readCurrentActivity(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(CURRENT_ACTIVITY, "");
    }

    /**
     * Save current Activity
     *
     * @param c context of the application
     * @param currentActivity current activity's name
     *
     */
    public static void writeCurrentActivity(final Context c, String currentActivity){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(CURRENT_ACTIVITY, currentActivity);
        editor.apply();
    }

    /**
     * Tell us if sdk user retrieves current activity
     *
     * @param c context of the application
     * @return true if user retrieves current activity
     */
    public static boolean readKeepCurrentActivity(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(USE_CURRENT_ACTIVITY, false);
    }

    /**
     * Save sdk user's choice to retrieve current activity
     *
     * @param c context of the application
     * @param useCurrentActivity choose to retrieve current activity or not
     */
    public static void writeKeepCurrentActivity(final Context c, boolean useCurrentActivity) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(USE_CURRENT_ACTIVITY, useCurrentActivity);
        editor.apply();
    }

    /**
     * Retrieve main Activity
     *
     * @param c context of the application
     * @return String containing main activity's name
     */
    public static String readMainActivity(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(MAIN_ACTIVITY, "");
    }

    /**
     * Save main Activity
     *
     * @param c context of the application
     * @param mainActivity main activity's name
     *
     */
    public static void writeMainActivity(final Context c, String mainActivity){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(MAIN_ACTIVITY, mainActivity);
        editor.apply();
    }

    /**
     * Retrieve share fragment
     *
     * @param c context of the application
     * @return String containing share fragment's name
     */
    public static String readShareFragment(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(SHARE_FRAGMENT, "");

    }

    /**
     * Save share fragment
     *
     * @param c context of the application
     * @param shareFragment share fragment's name
     *
     */
    public static void writeShareFragment(final Context c, String shareFragment){
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(SHARE_FRAGMENT, shareFragment);
        editor.apply();
    }

    // -------------------------     USER MODES     -------------------------
    // ----------------------------------------------------------------------

    /**
     * A way to know if hungry mode is elected by user. It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c context of the application
     * @return true if user has checked the option, false otherwise
     * @see #readShopping(android.content.Context)
     * @see #readTourist(android.content.Context)
     * @see #readThirsty(android.content.Context)
     */
    public static boolean readHungry(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(ProximitiesConstants.HUNGRY, false);
    }

    /**
     * Retain if user has chosen hungry mode as express filter
     * Used to send informations about userModes on WS.
     * It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c      context of the application
     * @param hungry <code>true</code> if this filter is activated, <code>false</code> otherwise
     * @see #writeShopping(android.content.Context, boolean)
     * @see #writeTourist(android.content.Context, boolean)
     * @see #writeThirsty(android.content.Context, boolean)
     */
    public static void writeHungry(final Context c, boolean hungry) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(ProximitiesConstants.HUNGRY, hungry);
        editor.apply();
    }

    /**
     * A way to know if shopping mode is elected by user. It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c context of the application
     * @return true if user has checked the option, false otherwise
     * @see #readHungry(android.content.Context)
     * @see #readTourist(android.content.Context)
     * @see #readThirsty(android.content.Context)
     */
    public static boolean readShopping(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(ProximitiesConstants.SHOPPING, false);
    }

    /**
     * Retain if user has chosen shopping mode as express filter
     * Used to send informations about userModes on WS
     * It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c        context of the application
     * @param shopping true if this filter is activated, false otherwise
     * @see #writeHungry(android.content.Context, boolean)
     * @see #writeTourist(android.content.Context, boolean)
     * @see #writeThirsty(android.content.Context, boolean)
     */
    public static void writeShopping(final Context c, boolean shopping) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(ProximitiesConstants.SHOPPING, shopping);
        editor.apply();
    }

    /**
     * A way to know if tourism mode is elected by user. It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c context of the application
     * @return true if user has checked the option, false otherwise
     * @see #readHungry(android.content.Context)
     * @see #readShopping(android.content.Context)
     * @see #readThirsty(android.content.Context)
     */
    public static boolean readTourist(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(ProximitiesConstants.TOURIST, false);
    }

    /**
     * Retain if user has chosen hungry mode as express filter
     * Used to send informations about userModes on WS
     * It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c       context of the application
     * @param tourism true if this filter is activated, false otherwise
     * @see #writeHungry(android.content.Context, boolean)
     * @see #writeShopping(android.content.Context, boolean)
     * @see #writeThirsty(android.content.Context, boolean)
     */
    public static void writeTourist(final Context c, boolean tourism) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(ProximitiesConstants.TOURIST, tourism);
        editor.apply();
    }

    /**
     * A way to know if thursty mode is elected by user. It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c context of the application
     * @return true if user has checked the option, false otherwise
     * @see #readHungry(android.content.Context)
     * @see #readShopping(android.content.Context)
     * @see #readTourist(android.content.Context)
     */
    public static boolean readThirsty(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(ProximitiesConstants.THIRSTY, false);
    }

    /**
     * Retain if user has chosen hungry mode as express filter
     * Used to send informations about userModes on WS
     * It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c       context of the application
     * @param thirsty true if this filter is activated, false otherwise
     * @see #writeHungry(android.content.Context, boolean)
     * @see #writeShopping(android.content.Context, boolean)
     * @see #writeTourist(android.content.Context, boolean)
     */
    public static void writeThirsty(final Context c, boolean thirsty) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(ProximitiesConstants.THIRSTY, thirsty);
        editor.apply();
    }

    /**
     * A way to know if do not disturb mode is elected by user.
     * @param c context of the application
     * @return true if user has checked the option, false otherwise
     */
    public static boolean readDoNotDisturb(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(ProximitiesConstants.DO_NOT_DISTURB, false);
    }

    /**
     * Retain if user has chosen do not disturb mode as express filter
     * Used to send informations about userModes on WS
     * @param c       context of the application
     * @param doNotDisturb true if this filter is activated, false otherwise
     */
    public static void writeDoNotDisturb(final Context c, boolean doNotDisturb) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(ProximitiesConstants.DO_NOT_DISTURB, doNotDisturb);
        editor.apply();
    }

    /**
     * Disable all user modes
     * It exists 4 user modes :
     * {@value ProximitiesConstants#HUNGRY}
     * {@value ProximitiesConstants#SHOPPING}
     * {@value ProximitiesConstants#TOURIST}
     * {@value ProximitiesConstants#THIRSTY}
     *
     * @param c Context of the application
     */
    public static void writeAllUserModesDisable(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(ProximitiesConstants.HUNGRY, false);
        editor.putBoolean(ProximitiesConstants.SHOPPING, false);
        editor.putBoolean(ProximitiesConstants.TOURIST, false);
        editor.putBoolean(ProximitiesConstants.THIRSTY, false);
        editor.apply();
    }
    // ----------------------------------------------------------------------
    // ----------------------------------------------------------------------

    public static boolean readDisableAnim(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(DISABLE_ANIM, false);
    }

    public static void writeDisableAnim(final Context c, boolean disableAnim) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(DISABLE_ANIM, disableAnim);
        editor.apply();
    }

    public static boolean readDisableFavorites(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getBoolean(DISABLE_FAVORITES, false);
    }

    public static void writeDisableFavorites(final Context c, boolean disableFav) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putBoolean(DISABLE_FAVORITES, disableFav);
        editor.apply();
    }
    // ----------------------------------------------------------------------
    // -------------------------     FOR SDK's USERS     --------------------
    // ----------------------------------------------------------------------

    /**
     * Retrieve the notification icon
     *
     * @param c context of the application
     * @return notification icon
     */
    public static int readNotificationIcon(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(NOTIFICATION_ICON, R.drawable.ic_launcher);
    }

    /**
     * Set the icon shown in Notifications
     *
     * @param c   context of the application
     * @param notifIcon  notification icon
     */
    public static void writeNotificationIcon(final Context c, int notifIcon) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putInt(NOTIFICATION_ICON, notifIcon);
        editor.apply();
    }

    /**
     * Retrieve content for notifications
     *
     * @param c context of the application
     * @return content
     */
    public static int readContentNotification(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(NOTIFICATION_CONTENT, R.string.multiple_anim_title);
    }

    /**
     * Set the content for notifications
     *
     * @param c   context of the application
     * @param contentNotif content
     */
    public static void writeContentNotification(final Context c, int contentNotif) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putInt(NOTIFICATION_CONTENT, contentNotif);
        editor.apply();
    }

    /**
     * Retrieve light color for notifications
     *
     * @param c context of the application
     * @return light color
     */
    public static int readLightColorNotification(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(NOTIFICATION_LIGHT, android.R.color.white);
    }

    /**
     * Set the light color for notifications
     *
     * @param c   context of the application
     * @param lightColor light color
     */
    public static void writeLightColorNotification(final Context c, int lightColor) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putInt(NOTIFICATION_LIGHT, lightColor);
        editor.apply();
    }

    /**
     * Retrieve the duration of the scan in milliseconds to detect beacons when the app is in foreground
     *
     * @param c context of the application
     * @return duration of the scan
     */
    public static int readForegroundScanPeriod(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(FOREGROUND_SCAN_PERIOD, 1100);
    }

    /**
     * Retrieve the duration between each Bluetooth LE scan cycle to detect beacons when the app is in foreground
     *
     * @param c context of the application
     * @return duration between the scan
     */
    public static int readForegroundBetweenScanPeriod(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(FOREGROUND_BETWEEN_SCAN_PERIOD, 0);
    }

    /**
     * Set the duration of the scan in milliseconds and the duration between each Bluetooth LE scan cycle to look for beacons (app in foreground)
     *
     * @param c   context of the application
     * @param foregroundScanPeriod duration of the scan
     * @param foregroundBetweenScanPeriod duration between each scan
     *
     */
    public static void writeForegroundScanPeriod(final Context c, int foregroundScanPeriod, int foregroundBetweenScanPeriod) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putInt(FOREGROUND_SCAN_PERIOD, foregroundScanPeriod);
        editor.putInt(FOREGROUND_BETWEEN_SCAN_PERIOD, foregroundBetweenScanPeriod);
        editor.apply();
    }

    /**
     * Retrieve the duration of the scan in milliseconds to detect beacons when the app is in background
     *
     * @param c context of the application
     * @return duration of the scan
     */
    public static int readBackgroundScanPeriod(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(BACKGROUND_SCAN_PERIOD, 5000);
    }

    /**
     * Retrieve the duration between each Bluetooth LE scan cycle to detect beacons when the app is in background
     *
     * @param c context of the application
     * @return duration between the scan
     */
    public static int readBackgroundBetweenScanPeriod(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getInt(BACKGROUND_BETWEEN_SCAN_PERIOD, 25000);
    }

    /**
     * Set the duration of the scan in milliseconds and the duration between each Bluetooth LE scan cycle to look for beacons (app in background)
     *
     * @param backgroundScanPeriod duration of the scan
     * @param backgroundBetweenScanPeriod duration between each scan
     */
    public static void writeBackgroundScanPeriod(final Context c, int backgroundScanPeriod, int backgroundBetweenScanPeriod) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putInt(BACKGROUND_SCAN_PERIOD, backgroundScanPeriod);
        editor.putInt(BACKGROUND_BETWEEN_SCAN_PERIOD, backgroundBetweenScanPeriod);
        editor.apply();
    }

    /**
     * Retrieve the last beacon detected
     *
     * @param c context of the application
     * @return uuid-major-minor from the last beacon detected
     */
    public static String readLastBeaconDetected(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        return settings.getString(LAST_BEACON_DETECTED, "");
    }

    /**
     * Set the the last beacon detected
     *
     * @param beacon uuid-major-minor from the last beacon detected
     */
    public static void writeLastBeaconDetected(final Context c, String beacon ) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        editor.putString(LAST_BEACON_DETECTED, beacon);
        editor.apply();
    }

    /**
     *
     */
    /*public static ProximitiesConfig readProximitiesConfig(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        Gson gson = new Gson();
        String json = settings.getString("proximities-config", "");
        ProximitiesConfig obj = gson.fromJson(json, ProximitiesConfig.class);
        return obj;
    }*/

    /**
     *
     */
    /*public static void writeProximitiesConfig(final Context c) {
        final SharedPreferences settings = c.getSharedPreferences(PREF_NAME, 0);
        final SharedPreferences.Editor editor = settings.edit();
        Gson gson = new Gson();
        String json = gson.toJson(ProximitiesConfig.getInstance());
        editor.putString("proximities-config", json);
        editor.apply();
    }*/
}
